/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * public abstract class AbstractDeviceButton
 * extends tle.vpx.gui.SplitButton
 *
 * A virtual device toolbar button.  This is a specific type of
 * <code>SplitButton</code> that knows of the connected state of the device,
 * always uses a specific icon based on the connected state of the device,
 * and will automatically register itself to receive notifications of relevant
 * updates to the device (and then update itself accordingly).
 *
 * @version 1.0 (Jun 29, 2006)
 */

/**
 * Creates a new virtual device menu button that will respond to updates on the
 * given view id.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the button
 * @param vmViewId String
 *    The id of the server-side view that will provide updates that pertain to
 *    this button
 * @param dataItemId String
 *    The id of the server view's device data item that corresponds to this
 *    button
 * @param name String
 *    The name of the device
 * @param connected boolean
 *    true if the device is currently connected; false otherwise
 */
vpx.context.AbstractDeviceButton = function (view, vmViewId, dataItemId, name, connected) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }
   if (!this._concrete) {
      throw new Error("vpx.context.AbstractDeviceButton: cannot instantiate abstract class");
   }

   // super(view, null, null)
   tle.vpx.gui.SplitButton.call(this, view, null, null);

   this.vmViewId = vmViewId;
   this.dataItemId = dataItemId;
   this.icons = new vpx.core.HashMap();
   this.setName(name);
   this.setConnected(connected);

   this.lsnId = tle.listenForUpdates(vmViewId, "VmView", this.__processUpdate.bind(this));
};

// AbstractDeviceButton extends tle.vpx.gui.SplitButton
vpx.context.AbstractDeviceButton.prototype = new tle.vpx.gui.SplitButton(tle.vpx.ABSTRACT_PASS);
vpx.context.AbstractDeviceButton.prototype.constructor = vpx.context.AbstractDeviceButton;

// Shorthand for brevity's sake
var _c = vpx.context.AbstractDeviceButton;// Class
var _i = _c.prototype;                    // Instance
_i._concrete = false;                     // vpx system flag for concrete classes

// class constants
_c.ICON_CLASS_CONNECTED = "icon";
_c.ICON_CLASS_DISCONNECTED = "icon";

// Instance variables
_i.vmViewId = null;                       // private String
_i.dataItemId = null;                     // private String
_i.icons = null;                          // private vpx.core.HashMap
_i.name = null;                           // protected String
_i.connected = null;                      // protected boolean
_i.lsnId = null;                          // private int

/**
 * Gets the name of this device.
 *
 * @return String
 *    The name of this device
 */
_i.getName = function() {
   return this.name;
};

/**
 * Sets the name of this device.
 *
 * @param str String
 *    The new name of the device
 */
_i.setName = function(str) {
   this.name = str;
};

/**
 * Sets the connected state of this device button.
 *
 * @param b boolean
 *    true if the device is connected; false otherwise
 */
_i.setConnected = function(b) {
   var key = (b ? "connected" : "disconnected");
   var icon = this.icons.get(key);
   if (icon == null) {
      var iconClass;
      if (b) {
         iconClass = this.constructor.ICON_CLASS_CONNECTED;
      } else {
         iconClass = this.constructor.ICON_CLASS_DISCONNECTED;
      }
      icon = new tle.vpx.gui.ImageIcon(this.view, iconClass, null);
      this.icons.put(key, icon);
   }
   this.setIcon(icon);
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Processes updates on the view that contains data for this device.  This will
 * ignore any changes that do not directly pertain to this device button.
 */
_i.__processUpdate = function(agent, refresh, changeSets) {
   if (refresh) {
      /**
       * The entire set of device menu buttons need to be reloaded.  This is
       * handled outside the scope of any individual button because it is the
       * responsibility of the object that instantiated the buttons.
       */
      return;
   }

   for (var i = 0; i < changeSets.length; i++) {
      var changeSet = changeSets[i];

      if (changeSet.id != this.dataItemId) {
         // The changeSet doesn't pertain to us
         continue;
      }

      for (var j = 0; j < changeSet.changes.length; j++) {
         this._processChange(changeSet.changes[j]);
      }
   }
};

/**
 * Processes an individual change pertaining to this device.
 *
 * @param change Object
 *    Change object, containing 'property' and 'value' attributes
 * @return boolean
 *    true if we handled the change; false if we ignored it
 */
_i._processChange = function(change) {
   switch (change.property) {

   case "connected":
      var connected = change.value.toBool();
      this.setConnected(connected);
      return true;

   case "key":
      this.setAttribute("key", change.value);
      return true;

   default:
      // Ignore
      //alert("change: (prop=" + change.property + " val=" + change.value + ")");
      return false;

   }
};
