/**
 * Sets up the vm list browser and requests the initial population of data.
 * NOTICE: this javascript calls performance indicator functions located in
 * ~/js/common/performance.js. Make sure this script is included before this one.
 */

function initVmListBrowser() {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   c.xmlSpec = vpx.net.XmlSpec.fromUrl("vmXmlListView.do?vmContainerId=" + c.vmFolderId);
   c.paginator = new vpx.browser.Paginator($("paginatorContainer"));
   c.browser = new vpx.browser.Browser($("browserContainer"), c.PAGE_SIZE,
      vpx.browser.Browser.HEIGHT_POLICY.USER, true);
   c.paginator.setProvider(c.browser);

   c.browser.registerListener(vpx.browser.EVENT.NAVIGATE, handleNavigate);
   c.browser.registerListener(vpx.browser.EVENT.SELECTION, handleSelectionChanged);
   c.browser.registerListener(vpx.browser.EVENT.SORT, handleSort);
   c.browser.registerListener(vpx.browser.EVENT.PAGE, handlePage);
   c.browser.registerListener(vpx.browser.EVENT.DATA, handleDataProcessed);
   c.xmlSpec.setAttribute("pageSize", c.browser.getPageSize());
   c.browser.requestData(c.xmlSpec);
   
   handleResize();
   vpx.xua.event.listen(self, "resize", handleResize);
}

/**
 * Tears down any active processes that the vm list browser is running.
 */
function unloadVmListBrowser() {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   if (c.updatesRunning) {
      vpx.log.debug("VM list browser: Stopping updates");
      tle.ignoreUpdates(c.updatesId);
      c.updatesRunning = false;
   }
   
   if (c.toolbarUpdatesRunning) {
      vpx.log.debug("VM list browser: Stopping toolbar updates");
      tle.ignoreUpdates(c.toolbarUpdatesId);
      c.toolbarUpdatesRunning = false;
   }   
   
   tle.getTopBarPane().disableToolbarNMenu();
      
}

/**
 * Handles the event of the vm list browser's item being
 * navigated to. It delegates the action to the navigation browser.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 */
function handleNavigate() {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   var selected = c.browser.getSelected();
   var view = eval("tle.getBrowsersPane().view_" + c.VM_NAVIGATION_BROWSER);
   view.selectEntity(selected[0].getId());
}

function handleSelectionChanged() {
   var c = vpx.context.vmListBrowser;
   var selected = c.browser.getSelected()[0];
   if (!isNull(selected)) {
      var entityId = selected.getId();   
      selectToolbar(entityId);      
   }
      
}

/**
 * Passively notifies the server of a change to the currently selected item.  
 * If an existing notification (request) has not yet returned, this will abort 
 * the existing request before sending the new one.
 * It will also defer execution of this method as needed when the request pool
 * is empty.
 *
 * @param entityId String
 *    The entity id of the newly selected entity
 */
function selectToolbar(entityId) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;
   
   if (c.toolbarSelectionReq != null) {
      // Cancel the now irrelevant request
      try {
         c.toolbarSelectionReq.abort();
         delete c.toolbarSelectionReq;
         c.toolbarSelectionReq = null;
      } catch (e) {
         ; 
      }
   }
   
   if (c.toolbarSelectionTimeoutId != null) {
      // Cancel the now irrelevant deferred execution of this method
      try {
         window.clearTimeout(c.toolbarSelectionTimeoutId);
         delete c.toolbarSelectionTimeoutId;
         c.toolbarSelectionTimeoutId = null;
      } catch (e) {
         ; // Ignore
      }
   }

   var url = "vmToolbarSelection.do?entityId=" + entityId;
   var req = tle.getRequest(url);
   if (req == null) {
         // Schedule deferred execution
      c.toolbarSelectionTimeoutId =
         window.setTimeout(selectToolbar.bind(self, entityId), 500);
      return;
   }

   try {
      req.addResponseListener(this);
      req.send();
      c.selectionUpdateReq = req; 
     
   } finally {
      tle.releaseRequest(req);
   }
}


/**
 * Processes a response.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The response event generated from the server response
 * @see vpx.net.event.ResponseListener#responseReceived(vpx.net.event.ResponseEvent)
 */
function responseReceived(e) {  
   
    var c = vpx.context.vmListBrowser;
    var req = e.getRequest();
    var resp = e.getSource();   
    var status = resp.getStatus();
    
    var contentType = resp.getContentType();    
    if (contentType != "text/xml") {
      throw new Error("VmListBrowser#responseReceived(): invalid content-type: " +
                      contentType + " (status " + status + ")");
    }
          
    var xml = resp.getXml();
    var root = xml.documentElement;
    if (root == null) {
      // XML was not well-formed
      throw new Error("VmListBrowser#responseReceived(): xml was not well-formed " +
                      "while processing " + req.getUrl());
    }
     
    if (root.nodeName != "response") {
      return;
    }
   
    var overallPowerStatus = root.getElementsByTagName("overallPowerStatus")[0].firstChild.nodeValue;
    var powerOffType = root.getElementsByTagName("powerOffType")[0].firstChild.nodeValue;
    var powerOffPreset = root.getElementsByTagName("powerOffPreset")[0].firstChild.nodeValue;
    var resetType = root.getElementsByTagName("resetType")[0].firstChild.nodeValue;
    var resetPreset = root.getElementsByTagName("resetPreset")[0].firstChild.nodeValue;
    
    // The following power ops should be boolean values to setup the toolbar.
    
    var PowerOffVM = root.getElementsByTagName("PowerOffVM")[0].firstChild.nodeValue.toBool();
    var PowerOnVM = root.getElementsByTagName("PowerOnVM")[0].firstChild.nodeValue.toBool();
    var SuspendVM = root.getElementsByTagName("SuspendVM")[0].firstChild.nodeValue.toBool();
    var ResetVM = root.getElementsByTagName("ResetVM")[0].firstChild.nodeValue.toBool();
    var ShutdownGuest = root.getElementsByTagName("ShutdownGuest")[0].firstChild.nodeValue.toBool();
    var RebootGuest = root.getElementsByTagName("RebootGuest")[0].firstChild.nodeValue.toBool();
    var viewId = root.getElementsByTagName("viewId")[0].firstChild.nodeValue;         
    
    tle.getTopBarPane().setupToolbar.monitoredInvoke(overallPowerStatus, powerOffType, powerOffPreset, resetType,
       resetPreset, PowerOffVM, PowerOnVM, SuspendVM, ResetVM, ShutdownGuest, RebootGuest);
       
    if (!c.toolbarUpdatesRunning) {
      c.toolbarUpdatesRunning = true;      
      vpx.log.debug("VM list browser: Turning on updates for toolbar and the VmViewId " + viewId);
      c.toolbarUpdatesId = tle.listenForUpdates(viewId, "VmView", processToolbarUpdates); 
   }   
          
}

   /**
    * process toolbar updates. 
    *
    * @param agent vpx.updates.Agent
    *    The agent that fired the event
    * @param refresh boolean
    *    true if a full refresh is required; false otherwise
    * @param changeSets Object[]
    *    An array of changeSet objects, each having the
    *    following attributes:
    *       id      : String
    *       type    : String
    *       changes : Object[] {property:String, value:String}
    */

  function processToolbarUpdates(agent, refresh, changeSets){
      if (refresh) {
         return;
      }
      if (changeSets == null || changeSets.length == 0){
         return;
      }
      
      var toolbar = tle.getTopBarPane().toolbar;     
      for (var i = 0; i < changeSets.length; i++) {
         var changeSet = changeSets[i];
         vpx.log.debug("changeSet: (id=" + changeSet.id + " type=" + changeSet.type + ")");

         for (var j = 0; j < changeSet.changes.length; j++) {
            var change = changeSet.changes[j];
            vpx.log.debug("change: (prop=" + change.property + " val=" + change.value + ")");
            toolbar.updateVMMenu(change);
            toolbar.updateToolbar(change);
         }
      }
}


/**
 * Handles a GUI sort event for the vm list browser.  It issues a
 * sorting action command to the browser's xml spec and re-requests browser
 * data.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param column vpx.browser.Column
 *    The column being sorted on
 * @param sortOrder const String
 *    The sort type (e.g. vpx.browser.SORT.ASC)
 */
function handleSort(browser, column, sortOrder) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   var viewId = c.xmlSpec.getAttribute("viewId");
   c.xmlSpec.clearAttributes();
   c.xmlSpec.setAttribute("viewId", viewId);
   c.xmlSpec.setAttribute("sortOrder", sortOrder);
   c.xmlSpec.setAttribute("sortColumns", column.getName());
   c.browser.requestData(c.xmlSpec);
   
   tle.getTopBarPane().disableToolbarNMenu();
}

/**
 * Handles a GUI page event for the vm list browser.  It issues the
 * corresponding page action command to the browser's xml spec and re-requests
 * browser data.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param pageType const String
 *    The page command (e.g. vpx.browser.PAGE.FIRST)
 */
function handlePage(browser, pageType) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   var viewId = c.xmlSpec.getAttribute("viewId");
   c.xmlSpec.clearAttributes();
   c.xmlSpec.setAttribute("viewId", viewId);
   c.xmlSpec.setAttribute("pageAction", pageType);
   c.browser.requestData(c.xmlSpec);
   
   tle.getTopBarPane().disableToolbarNMenu();
}

/**
 * Handles the completion of data processing for the vm list browser.  It
 * turns on the updates agent for the browser's backing view if the agent has
 * not already been started.  This must be done after the browser has
 * processed data because we don't know the viewId until the browser xml sets
 * it as a response attribute.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param xmlSpec vpx.net.XmlSpec
 *    The xml spec used to request the data
 */
function handleDataProcessed(browser, xmlSpec) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   var key = "viewId";
   if (!c.updatesRunning && xmlSpec.hasAttribute(key)) {
      c.updatesRunning = true;
      var viewId = xmlSpec.getAttribute(key);
      vpx.log.debug("VM list browser: Turning on updates for viewId " + viewId);
      c.updatesId = tle.listenForUpdates(viewId, "VmListView", processUpdates);
   }
}

/**
 * Processes updates to the current vm list browser's backing view.
 *
 * @param agent vpx.updates.Agent
 *    The agent that fired the event
 * @param refresh boolean
 *    true if a full refresh is required; false otherwise
 * @param changeSets Object[]
 *    An array of changeSet objects, each having the
 *    following attributes:
 *       id      : String
 *       type    : String
 *       changes : Object[] {property:String, value:String}
 */
function processUpdates(agent, refresh, changeSets) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;

   var viewId = c.xmlSpec.getAttribute("viewId");
   c.xmlSpec.clearAttributes();
   c.xmlSpec.setAttribute("viewId", viewId);

   vpx.log.debug("VM list browser: Received update: (refresh=" + refresh + ")");
   if (refresh) {
      c.browser.requestData(c.xmlSpec);
      return;
   }

   for (var i = 0; i < changeSets.length; i++) {
      var changeSet = changeSets[i];
      vpx.log.debug("changeSet: (id=" + changeSet.id + " type=" + changeSet.type + ")");

      switch (changeSet.type) {

      case "VirtualMachine":
         if (!processVmUpdates(changeSet)) {
            // We're requesting a refresh from the browser; stop processing updates
            return;
         }
         break;

      case "Host":
         if (!processHostUpdates(changeSet)) {
            // We're requesting a refresh from the browser; stop processing updates
            return;
         }
         break;

      default:
         vpx.log.info("VmListBrowser: IGNORING changeSet type: " + changeSet.type);
         break;

      }
   }
}



/** EditorListener: provides a data listener for HtmlEditor used within processVmUpdates() 
  * for updating values in the respective cell.
  *
  * @param element  the element upon a the callback function can operate
  */
vpx.context.EditorListener = function EditorListener(element) {
   this.element = element;
};
vpx.context.EditorListener.prototype.element = null;
vpx.context.EditorListener.prototype.dataReceived = function(data) {
   this.element.innerHTML=data;
}

/**
 * Processes a changeset specific to a <code>VirtualMachine</code> entity.
 *
 * @param changeSet Object
 *    id      : String
 *    type    : String
 *    changes : Object[] {property:String, value:String}
 * @return boolean
 *    true if we should continue processing updates; false otherwise
 */
function processVmUpdates(changeSet) {
   // Shorthand for brevity's sake
   var c = vpx.context.vmListBrowser;
   
   var node = c.browser.getNodeById(changeSet.id);
   for (var i = 0; i < changeSet.changes.length; i++) {
      var change = changeSet.changes[i];
      vpx.log.debug("change: (prop=" + change.property + " val=" + change.value + ")");
     
      switch (change.property) {

      case "question":
         // We need to also know the powerState, so we need a refresh
         c.browser.requestData(c.xmlSpec);
         return false;

      case "connectionState":
         // The node html needs to be reconstructed, so we need a refresh
         c.browser.requestData(c.xmlSpec);
         return false;

      case "powerState":
         var className = "vm-16x16";
         if (change.value == c.POWEREDON) {
            className = "poweredOn-16x16";
         } else if (change.value == c.POWEREDOFF) {
            className = "poweredOff-16x16";
         } else if (change.value == c.SUSPENDED) {
            className = "suspended-16x16";
         }
         node.setIconClass(className);
         break;
      
      case "cpuUsage":
         var column = c.browser.getColByName(change.property);
         if (isDefined(node) && isDefined(column)) {
            var cell = node.getCell(column, 0);
            var editorListener = new vpx.context.EditorListener(cell.firstChild);
            setCpuPerfLabel(change.value, editorListener );
         }
         break;
      case "memoryUsage":
         var column = c.browser.getColByName(change.property);
         if (isDefined(node) && isDefined(column)) {
            var cell = node.getCell(column, 0);
            var editorListener = new vpx.context.EditorListener(cell.firstChild);
            setMemoryPerfLabel(change.value, editorListener );
         }
         break;
      case "name":
         var column = c.browser.getColByName(change.property);
         if (isDefined(node) && isDefined(column)) {
            node.setCellHtml(column, 0, change.value.escapeElement());
         }
         break;
      default:
         ; // ignore
      }
   }
   
   return true;
}

/**
 * Processes a changeset specific to a <code>Host</code> entity.
 *
 * @param changeSet Object
 *    id      : String
 *    type    : String
 *    changes : Object[] {property:String, value:String}
 * @return boolean
 *    true if we should continue processing updates; false otherwise
 */
function processHostUpdates(changeSet) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   for (var j = 0; j < changeSet.changes.length; j++) {
      var change = changeSet.changes[j];
      vpx.log.debug("change: (prop=" + change.property + " val=" + change.value + ")");

      switch (change.property) {

      case "vms":
         // If the list of VMs changed, we need to refresh
         c.browser.requestData(c.xmlSpec);
         return false;

      }
   }

   return true;
}

/**
 * Sets attributes for the DIVs so that scroll bars will appear in Firefox when needed
 *
 */

function handleResize() {
   var tbh = vpx.xua.getHeight("vmTitlebar");
   var pch = vpx.xua.getHeight("paginatorContainer");
   var off = vpx.xua.getY("virtualMachines") * 2;
      
   var height = vpx.xua.viewport.getHeight(self);
   
   // XXX Don't know why we need five more pixels here.
   vpx.xua.setHeight("browserContainer", height - tbh - pch - off - 5);
}
