/**
 * Runs setup code that can only be run once the html page body has loaded
 * completely.
 */
function handleBodyLoad() {
   // Shorthand for brevity's sake
   var c = vpx.context;

   c.xmlSpec = new vpx.net.XmlSpec("vmNavigationXml.do");
   c.paginator = new vpx.browser.Paginator($("paginatorContainer"));
   c.browser = new vpx.browser.Browser($("browserContainer"), c.PAGE_SIZE,
      vpx.browser.Browser.HEIGHT_POLICY.USER, false);
   c.paginator.setProvider(c.browser);

   // only show the menu and up controls in a multi-host case
   var tle = vpx.getTle();
   var showControls = true;
   if (!tle.isMultiHost()) {
      showControls = false;
   }
   c.navigator = new vpx.browser.Navigator($("navigatorContainer"), showControls);
   c.navigator.setProvider(c.browser);

   c.browser.registerListener(vpx.browser.EVENT.SELECTION, handleSelectionChanged);
   c.browser.registerListener(vpx.browser.EVENT.SORT, handleSort);
   c.browser.registerListener(vpx.browser.EVENT.PAGE, handlePage);
   c.browser.registerListener(vpx.browser.EVENT.DATA, handleDataProcessed);
   c.browser.registerListener(vpx.browser.EVENT.NAVIGATE, handleNavigate);
   c.browser.requestData(c.xmlSpec);

   setHeight();
   vpx.xua.event.listen(parent, "resize", setHeight);
}

/**
 * Runs cleanup code that prepares the page to be safely destroyed.
 */
function handleBodyUnload() {
   // Shorthand for brevity's sake
   var c = vpx.context;

   if (c.updatesRunning) {
      tle.ignoreUpdates(c.updatesId);
      c.updatesRunning = false;
   }
}

/**
 * Handles the event of the current navigation browser's selected item having
 * changed.  It notifies the workspace pane of the selection.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 */
function handleSelectionChanged(browser) {
   var selected = browser.getSelected()[0];
   if (!isNull(selected)) {
      var entityId = selected.getId();
      var entityType = selected.getAttribute("entityType");
      updateNavigationView(entityId);
      notifyEntitySelected("detailDispatch.do", entityId, entityType, true);
   }
}

/**
 * Handles a GUI sort event for the current navigation browser.  It issues a
 * sorting action command to the browser's xml spec and re-requests browser
 * data.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param column vpx.browser.Column
 *    The column being sorted on
 * @param sortOrder const String
 *    The sort type (e.g. vpx.browser.SORT.ASC)
 */
function handleSort(browser, column, sortOrder) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   c.xmlSpec.setAttribute(c.PARAM_NAME_SORT_COL, column.getName());
   c.xmlSpec.setAttribute(c.PARAM_NAME_SORT_DIR, sortOrder);
   browser.requestData(c.xmlSpec);
   c.xmlSpec.removeAttribute(c.PARAM_NAME_SORT_COL);
   c.xmlSpec.removeAttribute(c.PARAM_NAME_SORT_DIR);
}

/**
 * Handles a GUI page event for the current navigation browser.  It
 * issues the corresponding page action command to the browser's xml spec
 * and re-requests browser data.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param pageType const String
 *    The page command (e.g. vpx.browser.PAGE.FIRST)
 */
function handlePage(browser, pageType) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   switch (pageType) {
   case vpx.browser.PAGE.PREV:
      c.xmlSpec.setAttribute(c.PARAM_NAME_PAGE, c.DC_PREV_PAGE);
      break;
   case vpx.browser.PAGE.NEXT:
      c.xmlSpec.setAttribute(c.PARAM_NAME_PAGE, c.DC_NEXT_PAGE);
      break;
   default:
      throw new Error("Unsupported page type: " + pageType);
   }

   browser.requestData(c.xmlSpec);
   c.xmlSpec.removeAttribute(c.PARAM_NAME_PAGE);
}

/**
 * Handles the completion of data processing for the current navigation
 * browser.  It turns on the updates agent for the browser's backing view
 * if the agent has not already been started.  This must be done after the
 * browser has processed data because we don't know the viewId until the
 * browser xml sets it as a response attribute.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param xmlSpec vpx.net.XmlSpec
 *    The xml spec used to request the data
 */
function handleDataProcessed(browser, xmlSpec) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   var key = "viewId";
   if (!c.updatesRunning && xmlSpec.hasAttribute(key)) {
      c.updatesRunning = true;
      var viewId = xmlSpec.getAttribute(key);
      c.updatesId = tle.listenForUpdates(viewId, "NavigationView", processUpdates);
   }
}

/**
 * Handles a GUI navigate event for the current navigation browser.  It issues
 * a navigate action command to the browser's xml spec and re-requests browser
 * data.
 *
 * @param browser vpx.browser.Browser
 *    The object that fired the event
 * @param baseItem vpx.ServerObj
 *    The object being navigated to
 */
function handleNavigate(browser, baseItem) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   c.xmlSpec.setAttribute(c.PARAM_NAME_NAV_ITEM, baseItem.getId());
   browser.requestData(c.xmlSpec);
   c.xmlSpec.removeAttribute(c.PARAM_NAME_NAV_ITEM);
}

/**
 * Selects the given entity in the current navigation browser.  If the given
 * entity is already part of the browser's current page, the entity is
 * immediately selected.  Otherwise, a refresh directive is issued to the
 * browser's xml spec with the given entity as a leaf, and the browser data
 * is re-requested.
 *
 * @param entityId String
 *    The id of the entity to select
 */
function selectEntity(entityId) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   var node = c.browser.getNodeById(entityId);
   if (!isNull(node)) {
      c.browser.select([node], true);
   } else {
      if (c.updatesRunning) {
         // Turn off updates before view is disposed
         tle.ignoreUpdates(c.updatesId);
         c.updatesRunning = false;
      }

      // The view needs to be refreshed to include the selected item
      c.xmlSpec.setAttribute(c.PARAM_NAME_REFRESH, "true");
      c.xmlSpec.setAttribute(c.PARAM_NAME_SELECT_ID, entityId);
      c.browser.requestData(c.xmlSpec);
      c.xmlSpec.removeAttribute(c.PARAM_NAME_REFRESH);
      c.xmlSpec.removeAttribute(c.PARAM_NAME_SELECT_ID);
   }
}

/**
 * Processes updates to the current navigation browser's backing view.
 *
 * @param agent vpx.updates.Agent
 *    The agent that fired the event
 * @param refresh boolean
 *    true if a full refresh is required; false otherwise
 * @param changeSets Object[]
 *    An array of changeSet objects, each having the
 *    following attributes:
 *       id      : String
 *       type    : String
 *       changes : Object[] {property:String, value:String}
 */
function processUpdates(agent, refresh, changeSets) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   vpx.log.debug("received update: (refresh=" + refresh + ")");

   if (refresh) {
      c.browser.requestData(c.xmlSpec);
      return;
   }

   for (var i = 0; i < changeSets.length; i++) {
      var changeSet = changeSets[i];
      vpx.log.debug("changeSet: (id=" + changeSet.id + " type=" + changeSet.type + ")");

      switch (changeSet.type) {
      case "VirtualMachine":
         processVmUpdates(changeSet);
         break;
      case "Host":
         processHostUpdates(changeSet);
         break;
      }
   }
}

/**
 * Processes a changeset specific to a <code>VirtualMachine</code> entity.
 *
 * @param changeSet Object
 *    id      : String
 *    type    : String
 *    changes : Object[] {property:String, value:String}
 */
function processVmUpdates(changeSet) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   for (var j = 0; j < changeSet.changes.length; j++) {
      var change = changeSet.changes[j];
      vpx.log.debug("change: (prop=" + change.property + " val=" + change.value + ")");

      if (change.property == "question") {
         // We need to also know the powerState, so we need a refresh
         c.browser.requestData(c.xmlSpec);
         return;
      }

      var node = c.browser.getNodeById(changeSet.id);
      if (node == null) {
         // This node lies in another page
         continue;
      }
      if (change.property == "powerState") {
         var className = "vm-16x16";
         if (change.value == c.POWEREDON) {
            className = "poweredOn-16x16";
         } else if (change.value == c.POWEREDOFF) {
            className = "poweredOff-16x16";
         } else if (change.value == c.SUSPENDED) {
            className = "suspended-16x16";
         }
         node.setIconClass(className);
      } else if (change.property == "connectionState") {
          var column = vpx.context.browser.getColumns()[0];
          var entityName = node.getAttribute("entityName");
          var value = null;

          if (change.value == c.VM_STATE_CONNECTED) {
             value = entityName;
          } else if (change.value == c.VM_STATE_DISCONNECTED) {
             value = "<span class=\"disabled\">" + entityName
               + c.VM_STATE_DISCONNECTED_APPEND_TEXT + "</span>";
          } else if (change.value == c.VM_STATE_ORPHANED) {
             value = "<span class=\"disabled\">" + entityName
               + c.VM_STATE_ORPHANED_APPEND_TEXT + "</span>";
          }
          node.setCellHtml(column, 0, value);
      }  else {
         var column = vpx.context.browser.getColByName(change.property);
         if (isDefined(node) && isDefined(column)) {
            node.setCellHtml(column, 0, change.value);
         }
      }
   }
}

/**
 * Processes a changeset specific to a <code>Host</code> entity.
 *
 * @param changeSet Object
 *    id      : String
 *    type    : String
 *    changes : Object[] {property:String, value:String}
 */
function processHostUpdates(changeSet) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   for (var j = 0; j < changeSet.changes.length; j++) {
      var change = changeSet.changes[j];
      vpx.log.debug("change: (prop=" + change.property + " val=" + change.value + ")");

      switch (change.property) {
      case "hostname":
         //?? TODO Update navigator. Can hostname ever change in a live env?
         break;
      }
   }
}

/**
 * Passively notifies the server of a change to the currently selected item in
 * the navigation view.  If an existing notification (request) has not yet
 * returned, this will abort the existing request before sending the new one.
 * It will also defer execution of this method as needed when the request pool
 * is empty.
 *
 * @param entityId String
 *    The entity id of the newly selected entity
 */
function updateNavigationView(entityId) {
   // Shorthand for brevity's sake
   var c = vpx.context;

   if (c.selectionUpdateReq != null) {
      // Cancel the now irrelevant request
      try {
         c.selectionUpdateReq.abort();
         delete c.selectionUpdateReq;
         c.selectionUpdateReq = null;
      } catch (e) {
         ; // Ignore; selectionUpdateReq got set to null under our feet
      }
   }

   if (c.selectionUpdateTimeoutId != null) {
      // Cancel the now irrelevant deferred execution of this method
      try {
         window.clearTimeout(c.selectionUpdateTimeoutId);
         delete c.selectionUpdateTimeoutId;
         c.selectionUpdateTimeoutId = null;
      } catch (e) {
         ; // Ignore
      }
   }

   var url = "navigationSelectionUpdate.do?entityId=" + entityId;
   var req = tle.getRequest(url);
   if (req == null) {
      // Schedule deferred execution
      c.selectionUpdateTimeoutId =
         window.setTimeout(updateNavigationView.bind(self, entityId), 500);
      return;
   }

   try {
      req.addResponseListener(window);
      req.send();
      c.selectionUpdateReq = req;
   } finally {
      tle.releaseRequest(req);
   }
}

/**
 * Processes a response to the selection update action from the server.  This
 * simply records the response by setting the current request pointer to
 * <code>NULL</code> (the server has responded, so by definition there is no
 * more active request).
 *
 * @param e vpx.net.event.ResponseEvent
 *    The response event generated from the server response
 * @see vpx.net.event.ResponseListener#responseReceived(vpx.net.event.ResponseEvent)
 */
function responseReceived(e) {
   vpx.context.selectionUpdateReq = null;
}

/**
 * Notifies the workspace pane that an entity in the navigation pane has been
 * selected.
 *
 * @param actionTarget String
 *    The action url
 * @param entityId String
 *    The id of the entity that was selected
 * @param entityType String
 *    The type of entity that was selected
 * @param syncType boolean
 *    ??
 */
function notifyEntitySelected(actionTarget, entityId, entityType, syncType){
   var strLocation = "";
   strLocation += actionTarget;
   if (!isNull(entityId)) {
      strLocation += "?entityId=" + entityId;
   }
   if (!isNull(entityType)) {
      strLocation += "&entityType=" + entityType;
   }
   strLocation += "&syncType=" + syncType;

   try {
      var wsPane = tle.getWorkspacePane();
      if (wsPane.location.href.indexOf(strLocation) == -1) {
         // Only set location if it's not already the current location
         wsPane.location.href = strLocation;
      }
   } catch (ex) {
      // Workspace pane is not yet loaded - ignore
   }

   if (window.parent.navSelectionState) {
      window.parent.navSelectionState.entityId(entityId);
   }
}

/**
 * Sets attributes for the DIVs so that scroll bars will appear in Firefox when needed
 *
 */

function setHeight() {
   var nc = $("navigatorContainer");
   var bc = $("browserContainer");
   var pc = $("paginatorContainer");

   var nch = vpx.xua.setHeight(nc, 22);
   var pch = vpx.xua.getHeight(pc);

   var height = vpx.xua.viewport.getHeight(self);

   vpx.xua.setPosition(nc, 0, 0);
   vpx.xua.setPosition(pc, 0, height - pch);
   vpx.xua.setPosition(bc, 0, nch);

   vpx.xua.setHeight(bc, height - nch - pch);
}
