/* **********************************************************
 * navigationPane.js
 *
 * Copyright (C) 2004 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/

/**
* navigationPane.js
* <p>
* This is the Javascript module dealing with the accordion style navigation
* pane's behavior and browsers' selection.
*/

// ------------------------ selected objects related ---------------------------

/**
* @global_var XXX
*/
var selectedBrowser = null;

/**
* @global_var boolean indicating whether to animate the navigation pane's
* browsers selection (accordion style expand/collapse)
*/
var BOOL_ANIMATE = false;

/**
* @global_var If animated, the speed in which the browsers will be expanded and
* collapse in (interval between resize operations).
*/
var ANIMATE_SPEED = 10; // in ms

/**
* @global_var If animated the percentage of resize in each resize operation.
*/
var ANIMATE_RATE = 2; // in precents

/**
* @global_var The id for the interval used in the animation process. When animation
* is done, the interval will have to be cleared.
*/
var intervalId = -1;

/**
 * Dispatches the browser selection.
 * <p>
 * Note: No selection will be attempted if the naviagtion browser requested is
 * already currently selected.
 * @param strBrowser The name of the browser to select
 */

function selectBrowser(strBrowser){
   // do nothing if the category clicked is already expanded, or if no browser name was specified
   if (strBrowser == selectedBrowser || strBrowser == "") {
      return;
   }
   if (BOOL_ANIMATE) {
      animatedSelectBrowser(strBrowser);
   } else {
      normalSelectBrowser(strBrowser);
   }
}

/**
 * Selects a navigation browser (no animation).
 * <p>
 * Will collapse an open navigation browser if there's one open, and only then
 * will expand the requested browser.
 *
 * @param strBrowser The name of the navigation browser to expand
 *
 */
function normalSelectBrowser(strBrowser){
   // do nothing if the category clicked is already expanded, or if no browser name was specified
   if (strBrowser == selectedBrowser || strBrowser == "") {
      return;
   }

   if (selectedBrowser != null){
      collapseBrowser(selectedBrowser);
   }

   selectedBrowser = strBrowser;
   expandBrowser(selectedBrowser);
}

/**
 * Expands a navigation browser.
 * <p>
 * This method relies on the conventions of a navigation browser as implemented
 * in the including calling jsp page (navigationPane.jsp).<br>
 * <ul>
 *    <li> browser_browserName - is the table row containing the navigation
 *    browser contents</li>
 *    <li> browserView_browserName - is the table row containing the browser's
 *       IFRAME</li>
 *    <li>view_browserName - is the IFRAME itself which will retrieve a list of
 *    objects from the server for selection </li>
 * </ul>
 * <p>
 * Note: the urlMap array used in this function MUST be implemented in the
 *    including page (navigationPane.jsp)
 *
 * @param The browser name to expand
 *
 */
function expandBrowser(strBrowser){
   // TODO protect against no such browser in the urlMap array
   var trBrowser = document.getElementById("browser_"+strBrowser);
   var trBrowserView = document.getElementById("browserView_"+strBrowser);
   var view = window.frames["view_"+strBrowser];
   var trLeftOver = document.getElementById("trLeftover");

   trBrowser.style.height = "100%";
   trBrowserView.style.display = "";
   trLeftOver.style.display = "none";
   try {
      view.location.replace(urlMap[strBrowser]);
   } catch (e){
      window.setTimeout("expandBrowser('" + strBrowser + "')",100);
   }
}

/**
 * Collapses a navigation browser.
 * <p>
 * This method relies on the conventions of a navigation browser as implemented
 * in the including jsp page (navigationPane.jsp).<br>
 * <ul>
 *    <li> browser_browserName - is the table row containing the navigation
 *    browser contents</li>
 *    <li> browserView_browserName - is the table row containing the browser's
 *       IFRAME</li>
 *    <li>view_browserName - is the IFRAME itself which will retrieve a list of
 *    objects from the server for selection </li>
 * </ul>
 *
 * @param The browser name to collapse
 *
 */

function collapseBrowser(strBrowser){
   var trBrowser = document.getElementById("browser_"+strBrowser);
   var trBrowserView = document.getElementById("browserView_"+strBrowser);
   var view = window.frames["view_"+strBrowser];

   trBrowser.style.height = "0%";
   trBrowserView.style.display = "none";
   view.location.replace("html/loading.html"); //TODO hardwired!!!
}

/**
 * Fits the selected browser's height to a specified CSS valid height value
 * <p>
 * This method relies on the conventions of a navigation browser as implemented
 * in the including jsp page (navigationPane.jsp).<br>
 * <ul>
 *    <li> browser_browserName - is the table row containing the navigation
 *    browser contents</li>
 *    <li> browserHeader__browserName - is the table row containing the
 *       browser's label</li>
 * </ul>
 * <p>
 * Note: In case resizing is done, the rest of the navigationPane height should
 *    dynamically be taken over by the blank leftOver table row
 *
 * @param intHeight A valid CSS height value
 */
function fitBrowser(intHeight){
   if (selectedBrowser == null) {
      return;
   }

   var trBrowser = document.getElementById("browser_"+selectedBrowser);
   var trBrowserHeader = document.getElementById("browserHeader_"+selectedBrowser);
   var trLeftOver = document.getElementById("trLeftover");

   trBrowser.style.height = (intHeight + trBrowserHeader.offsetHeight) + "px";
   trLeftOver.style.display = "";
   trLeftOver.style.height = "100%";
}

//-- navigationSelection objet

function navigationSelection(){
   this.strContainerId = null;
   this.strEntityId = null;

   this.containerId = navigationSelection_getSetContainerId;
   this.entityId = navigationSelection_getSetEntityId;
   this.clear = navigationSelection_clear;
   this.toString = navigationSelection_toString;

}

function navigationSelection_getSetContainerId(strId){
   if (strId != null){
      this.strContainerId = strId;
   }
   return this.strContainerId;
}

function navigationSelection_getSetEntityId(strId){
   if (strId != null){
      this.strEntityId = strId;
   }
   return this.strEntityId;
}

function navigationSelection_clear(){
   this.strContainerId = null;
   this.strEntityId = null;
}

function navigationSelection_toString(){
   var strTmp = "";
   strTmp += "\nClient Navigation Selection --";
   strTmp += "\nContainer Id: " + ((this.containerId == null) ? "none" : this.strContainerId);
   strTmp += "\nEntity Id: " + ((this.entityId == null) ? "none" : this.strEntityId);
   return strTmp;
}

var navSelectionState = new navigationSelection();

//------------------- ANIMATED GAMES ---------------------

/**
 * Toggles the animation state of the navigation pane.
 *
 * @param boolAnimate [optional] boolean indicating whether to animate the
 *    accordion widget or not.
 */

function setAnimate(boolAnimate){
   if (!boolAnimate || boolAnimate != true) {
      BOOL_ANIMATE = false;
   } else {
      BOOL_ANIMATE = true;
   }
}

function animatedSelectBrowser(strBrowser){
   // do nothing if the category clicked is already expanded
   if (strBrowser == selectedBrowser || strBrowser == "") {
      return;
   }

   if (selectedBrowser != null){
      document.getElementById("trLeftover").style.display = "";
      intervalId = window.setInterval("animatedCollapseBrowser('"+selectedBrowser+"','"+strBrowser+"')",ANIMATE_SPEED);
      return;
   }

   selectedBrowser = strBrowser;
   intervalId = window.setInterval("animatedExpandBrowser('"+selectedBrowser+"')",ANIMATE_SPEED);
}

function animatedCollapseBrowser(strBrowser,strNewBrowser){
   var trBrowser = document.getElementById("browser_"+strBrowser);
   var trBrowserView = document.getElementById("browserView_"+strBrowser);
   var view = window.frames["view_"+strBrowser];

   if ( parseInt(trBrowser.style.height,10) <= 0){
      window.clearInterval(intervalId);
      intervalId = -1;
      trBrowser.style.height = "0%";
      trBrowserView.style.display = "none";
      view.location.replace("html/loading.html");
      selectedBrowser = null;
      animatedSelectBrowser(strNewBrowser);
   } else {
      var tmpHeight = (parseInt(trBrowser.style.height,10) - ANIMATE_RATE);
      trBrowser.style.height = ((tmpHeight < 0) ? 0 : tmpHeight) + "%";
   }
}

function animatedExpandBrowser(strBrowser){
   var trBrowser = document.getElementById("browser_"+strBrowser);
   var trBrowserView = document.getElementById("browserView_"+strBrowser);
   var view = window.frames["view_"+strBrowser];
   var trLeftOver = document.getElementById("trLeftover");

   trBrowserView.style.display = "";

   if ( parseInt(trBrowser.style.height,10) >= 100) {
      window.clearInterval(intervalId);
      intervalId = -1;
      trBrowser.style.height = "100%";
    try{
       view.location.replace(urlMap[strBrowser]);
    } catch (e) {
       window.setTimeout("view.location.replace(urlMap[strBrowser])",100);
    }
      trLeftOver.style.display = "none";
   } else {
      var tmpHeight = (parseInt(trBrowser.style.height,10) + ANIMATE_RATE);
      if ( isNaN(tmpHeight) ) {
         tmpHeight = 0;
      }
      trBrowser.style.height = ((tmpHeight > 100) ? 100 : tmpHeight) + "%";
   }
}
