/* **********************************************************
 * tle.js
 *
 * Copyright (C) 2004 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/

/**
* tle.js
* <p>
* This is the Javascript module dealing with cross pages context objects
* and operations.
* <p>
* This module MUST be included in, and only in, the web application's top most
* window and jsp page (top.jsp)
*/


/**
 * @global_var boolean indicating this is the top level element in the application
 */
var isWebcenterTle = true;

//==============================================================================
//    Begin VPX Wrapper Section
//==============================================================================

/**
 * vpx.updates.Agent
 *
 * The singleton vpx.updates.Agent instance, responsible for requesting updates
 * from the server on registered views and invoking the appropriate callbacks.
 */
var updatesAgent = null;

/**
 * Listens for updates on the given view, and registers the callback to be
 * invoked when updates are ready for the given view.  The callback will be
 * invoked with the following parameters:
 *    agent       [type: vpx.updates.Agent]
 *       The agent that fired the event
 *    refresh     [type: boolean]
 *       true if a full refresh is required; false otherwise
 *    changeSets  [type Object[]] (optional)
 *       An array of changeSet objects, each having the following attributes:
 *          id      : String
 *          type    : String
 *          changes : Object[] { property:String, value:String }
 *
 * @param viewId String
 *    The id of the back-end <code>View</code> whose updates you're listening
 *    for
 * @param name String
 *    A logical name for the view, used for debugging purposes
 * @param callback Function
 *    The function to invoke when an update is ready for the given view
 * @return int
 *    A listenerId that can be used to later remove this listener
 */
function listenForUpdates(viewId, name, callback) {
   if (updatesAgent == null) {
      throw new Error("tle#listenForUpdates(): updates agent has not been instantiated");
   }
   return updatesAgent.registerView(viewId, name, callback);
}

/**
 * Stops listening for updates.  This may only be called after having started
 * listening for updates, because upon starting listening for updates, you
 * receive an opaque id that is your handle by which you may stop listening.
 *
 * @param listenerId int
 *		The id returned upon adding the listener
 */
function ignoreUpdates(listenerId) {
   if (updatesAgent == null) {
      throw new Error("tle#ignoreUpdates(): updates agent has not been instantiated");
   }
   updatesAgent.unregisterListener(listenerId);
}

/**
 * Parses the given url according to the standard VPX GUI XML contract and
 * creates GUI objects as necessary.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @param xmlUrl String
 *    The url of the xml resource
 * @return vpx.gui.Component[]
 *    The components that were created, in the order they were created
 */
function createGui(view, xmlUrl) {
   var digester = new vpx.gui.digest.Digester();
   return digester.parse(view, new vpx.net.XmlSpec(xmlUrl));
   delete digester;
}

/**
 * vpx.gui.MenuBar
 *
 * The top bar menubar, which holds the webCenter menus.  This is of type
 * <code>vpx.gui.MenuBar</code>. The code that calls <code>tle.createGui</code>
 * to create the top menubar should set this variable to the menubar that is
 * returned.
 */
var menuBar = null;

/**
 * Gets the application (currently "WebCenter") menu.
 *
 * @return vpx.gui.Menu
 *    The app menu, or <code>null</code> if no such menu exists, or if the
 *    menuBar has not yet been instantiated
 */
function getAppMenu() {
   if (tle.menuBar == null) return null;
   return tle.menuBar.getMenu(0);
}

/**
 * Gets the "View" menu.
 *
 * @return vpx.gui.Menu
 *    The "View" menu, or <code>null</code> if no such menu exists, or if the
 *    menuBar has not yet been instantiated
 */
function getViewMenu() {
   if (tle.menuBar == null) return null;
   return tle.menuBar.getMenu(1);
}

/**
 * Gets the "Virtual Machine" menu.
 *
 * @return vpx.gui.Menu
 *    The "VM" menu, or <code>null</code> if no such menu exists, or if the
 *    menuBar has not yet been instantiated
 */
function getVMMenu() {
   if (tle.menuBar == null) return null;
   return tle.menuBar.getMenu(2);
}

/**
 * Binds a string to the current user's session, using the name specified. If
 * an object of the same name is already bound to the session, the object is
 * replaced.
 * <p/>
 * If the value passed in is <code>null</code>, this has the same effect as
 * calling <code>removeAttribute()</code>.
 *
 * @param name String
 *    The name to which the object is bound; cannot be <code>null</code>
 * @param value String
 *    The string to be bound
 * @param args ...
 *    Any further args will be treated pairwise as name/value pairs and will
 *    also be set as session attributes. e.g.
 *    <code>setSessionAttr("one", "1", "two", "2")</code>
 */
function setSessionAttrs(name, value, args) {
   vpx.session.setAutoCommit(false);
   for (var i = arguments.length - 2; i >= 0; i -= 2) {
      vpx.session.setAttribute(arguments[i], arguments[i + 1]);
   }
   vpx.session.commit();
   vpx.session.setAutoCommit(true);
}

/**
 * Removes the object bound with the specified name from the current user's
 * session. If the session does not have an object bound with the specified
 * name, this method does nothing.
 *
 * @param name String
 *    The name of the object to remove from the current user's session
 * @param args ...
 *    Any further arguments will be treated as session attribute names and will
 *    also be removed from the user's session
 */
function removeSessionAttrs(name, args) {
   vpx.session.setAutoCommit(false);
   for (var i = 0; i < arguments.length; i++) {
      vpx.session.removeAttribute(arguments[i]);
   }
   vpx.session.commit();
   vpx.session.setAutoCommit(true);
}

/**
 * vpx.net.HttpRequestPool
 *
 * The global request pool shared by all sub-contexts of <code>tle</code>.
 *
 * @see tle#getRequest()
 * @see tle#releaseRequest(vpx.net.HttpRequest)
 */
var requestPool = null;

/**
 * Checks out a request from the global request pool, initializes it, and
 * reserves it for use by the caller.  When the caller is done using the
 * request, they <b>must</b> release the request back into the pool by calling
 * <code>releaseRequest</code>.  If this is not done, requests will be leaked,
 * and the pool will eventually (very quickly) run out of available requests.
 * To ensure that this is done properly, it is wise to place your code in a
 * try/finally block similar to the following:
 * <pre>
 *    var req = tle.getRequest("sample.do?param=value");
 *    try {
 *       req.addResponseListener(myResponseListener);
 *       req.send();
 *    } finally {
 *       tle.releaseRequest(req);
 *    }
 * </pre>
 *
 * @param url String
 *    The url to initialize the request to
 * @return vpx.net.HttpRequest
 *    An <code>HttpRequest</code> reserved for use by the caller
 */
function getRequest(url) {
   if (requestPool == null) {
      throw new Error("tle#getRequest(): request pool has not been instantiated");
   }
   var req = requestPool.getRequest();
   if (req == null) {
      return null;
   }

   req.setUrl(url);
   return req;
}

/**
 * Releases a "checked-out" request back into the global request pool.
 *
 * @param req vpx.net.HttpRequest
 *    An <code>HttpRequest</code> that may be released back into the pool
 */
function releaseRequest(req) {
   if (requestPool == null) {
      throw new Error("tle#releaseRequest(): request pool has not been instantiated");
   }
   requestPool.releaseRequest(req);
}


//==============================================================================
//    End VPX Wrapper Section
//==============================================================================

//==============================================================================
//    Begin Panes Resizing Section
//
// Hiding and showing the navigation pane, and toggling the system's modal state
//==============================================================================

/**
 * @global_var valid css width for the normal navigation pane width
 */
var NAV_PANE_WIDTH = "225px";

/**
 * @global_var valid css width for the navigation pane width in the collapsed state
 */
var NAV_PANE_WIDTH_MODAL = "21px";

/**
 * The settings that control the user's view of the app.  These settings
 * can be controlled by the user in the "View" menu.
 */
var viewSettings = {
   toolbarVisible : true,
   tabsVisible    : true,
   navVisible     : true,
   navResizable   : true
};

/**
 * Syncs the user's view to match that of the user's view settings. The app
 * may temporarily override the view (such as in the case of modality), but
 * it should always call <code>tle.syncView()</code> when it's done in order
 * to return the user to the view that they were in before the app interfered.
 */

function syncView() {
   setToolbarVisible(viewSettings.toolbarVisible, true);
   setTabsVisible(viewSettings.tabsVisible, true);
   setNavVisible(viewSettings.navVisible, true);
   setNavResizable(viewSettings.navResizable, true);
}

/**
 * Saves an update to a view attribute.
 *
 * @param key String
 *    The <code>viewSettings</code> attribute to update
 *    (e.g. <code>toolbarVisible</code>)
 * @param value boolean
 *    The new value of the setting.  This will be saved both in the client-side
 *    array and on the server for persistence.
 */

function updateView(key, value) {
   if (viewSettings[key] == value) {
      return;
   }

   // Update client
   viewSettings[key] = value;

   // Update server
   var settings = vpx.xua.urlEncode(key + "=" + value);
   var url = getContextPath() + "/viewSettingsUpdate.do?settings=" + settings;
   var req = getRequest(url);
   try {
      req.send();
   } finally {
      releaseRequest(req);
   }
}

/**
 * Sets the visibility of the toolbar.  This will update the user's view
 * settings unless otherwise specified.
 *
 * @param b boolean
 *    true to make the toolbar visible; false otherwise
 * @param isTransient boolean (optional)
 *    true to prevent the API from saving this change in the
 *    user's view settings.  false or unspecified otherwise
 */

function setToolbarVisible(b, isTransient) {
   getTopBarPane().toolbar.hide(!b);

   var rowsStr = vpx.xua.getHeight.call(getTopBarPane(), "toolbar") + ",*";
   window.wc_appFrame.document.getElementsByTagName('frameset')[0].rows = rowsStr;

   if (!isTransient) {
      updateView("toolbarVisible", b);
   }
}

/**
 * Sets the visibility of the tabs.  This will update the user's view
 * settings unless otherwise specified.
 *
 * @param b boolean
 *    true to make the tabs visible; false otherwise
 * @param isTransient boolean (optional)
 *    true to prevent the API from saving this change in the
 *    user's view settings.  false or unspecified otherwise
 */

function setTabsVisible(b, isTransient) {
   var wsPane = getWorkspacePane();
   var display = (b ? "" : "none");
   wsPane.document.getElementById("workspaceTabs").style.display = display;

   //restore the top Border to the workspaceDetails iframe
   if (b) {
      wsPane.document.getElementById("workspaceDetails").className = "workspaceDetailsNonExclusive";
   } else {
      wsPane.document.getElementById("workspaceDetails").className = "";
   }


   if (!isTransient) {
      updateView("tabsVisible", b);
   }
}

/**
 * isNavVisible
 * Checks if the navigation is currently visible.
 * Returns true if visible
 */

function isNavVisible() {
   var navPane = window.wc_appFrame.wc_bottomPane.wc_navigationPane;
   if (!isDefined(navPane)) {
      return false;
   }

   var doc = navPane.document;
   var visible = true;
   if (doc.getElementById("browser_vMachines").style.display == "none") {
      visible = false;
   }
   return visible;
}

/**
 * Sets the visibility of the navigation.  This will update the user's view
 * settings unless otherwise specified.
 *
 * @param b boolean
 *    true to make the navigation pane visible; false otherwise
 * @param isTransient boolean (optional)
 *    true to prevent the API from saving this change in the
 *    user's view settings.  false or unspecified otherwise
 */

function setNavVisible(b, isTransient) {
   var navPane = window.wc_appFrame.wc_bottomPane.wc_navigationPane;
   if (isDefined(navPane)) {
      // Update the width of the nav pane itself
      if (!(isNavVisible() && b)){
         var colsStr = (b ? NAV_PANE_WIDTH : NAV_PANE_WIDTH_MODAL) + ",*";
         window.wc_appFrame.wc_bottomPane.document.getElementsByTagName('frameset')[0].cols = colsStr;
      }

      // Update the display of the various browsers in the nav pane
      var doc = navPane.document;
      var strDisp = (b ? "" : "none");
      doc.getElementById("browser_vMachines").style.display = strDisp;
      if (isMultiHost()) {
         var browsers = [
            "browser_alarms"
            //"browser_hosts",
            //"browser_customViews",
            //"browser_networks",
            //"browser_datastores",
         ];
         for (var i = 0; i < browsers.length; i++) {
            var browser = doc.getElementById(browsers[i]);
            if (browser != null) {
               browser.style.display = strDisp;
            }
         }
      }
   }

   if (!isTransient) {
      updateView("navVisible", b);
   }
}

/**
 * Sets the resizability of the navigation.  This will update the user's view
 * settings unless otherwise specified.
 *
 * @param b boolean
 *    true to make the navigation pane resizable; false otherwise
 * @param isTransient boolean (optional)
 *    true to prevent the API from saving this change in the
 *    user's view settings.  false or unspecified otherwise
 */

function setNavResizable(b, isTransient) {
   var navPane = window.wc_appFrame.wc_bottomPane.wc_navigationPane;
   if (isDefined(navPane)) {
      if (!b){
         window.wc_appFrame.wc_bottomPane.document.getElementsByTagName('frameset')[0].cols = "0,*";
      }

      var doc = navPane.document;
      var strDisp = (b ? "" : "none");
      doc.images["resizeNavPane"].style.display = strDisp;
   }

   if (!isTransient) {
      updateView("navResizable", b);
   }
}

/**
 * Toggles the application between a modal and non modal state.
 * <p/>
 * Modal state means that the navigation bar on the left is almost fully
 * covered, with all controls on it hidden and non clickable. Modal also hides
 * the toolbar preventing any buttons of being clicked.
 * <p/>
 * Documents requiring a modal application state should call
 * <code>toggleModal()</code> in their <code>onload</code> method.
 *
 * @param b boolean
 *    true to make the user's view modal, false to restore the view to its
 *    default state
 */

function toggleModal(b) {
   if (b) {
      // Modality doesn't affect user's view settings; all ops are transient
      setToolbarVisible(false, true);
      setTabsVisible(false, true);
      setNavVisible(false, true);
      setNavResizable(false, true);
   } else {
      syncView();
   }
}


//==============================================================================
// End Panes Resizing Section
//==============================================================================


//==============================================================================
//    Begin Window Locators Section
// Methods for locating the different frames used in the application.
// It is a best practice for any document wishing to refer to another document
// within a different window or to another window's javascript context should
// retrieve that window through the the top level element using the methods
// below.
//
// Using this best practice will allow us to shuffle the windows around without
// breaking the javascript throughout the applicaiton.
//==============================================================================

/**
 * Locates the workspace pane
 *
 * @return the workspace pane
 */
function getWorkspacePane(){
   return window.wc_appFrame.wc_bottomPane.wc_workspacePane;
}

/**
 * Locates the workspace details window
 *
 * @return the workspace details
 */
function getWorkspaceDetails(){
   return window.wc_appFrame.wc_bottomPane.wc_workspacePane.workspaceDetails;
}

/**
 * Retrieves the topBar pane.
 *
 * @return topBar pane.
 */
function getTopBarPane() {
   return window.wc_appFrame.wc_topBar;
}


/**
 * Locates the browsers (navigation pane)
 *
 * @return the browsers pane
 */
function getBrowsersPane(){
   return window.wc_appFrame.wc_bottomPane.wc_navigationPane;
}

/**
 * Replaces the entire application frame structure and contents with a given URL
 *
 * @param strURL The URL to replace the application's content with
 */
function replaceAll(strURL){
   window.location.replace(strURL);
}

//==============================================================================
// End Window Locators Section
//==============================================================================

//==============================================================================
//Begin Window Status and title
//Methods for manipulating the window status and title bars
//
// Any document wishing to change the value of those should do so through
// tle.setStatusBar and tle.setTitleBar (tle is provided by including init.js in the page).
//==============================================================================

/**
 * Sets the application's title bar to a given text (blank string if none given)
 *
 * @param strTitle The text to put in the application's title bar
 */

function setTitleBar(strTitle){
   if (strTitle == null) {
      strTitle="";
   }
   window.document.title = strTitle;
}

/**
 * Sets the application's status bar to a given text (blank string if none given)
 *
 * Note !
 * As of February 21, Firefox V1.0 does not let javascript change the status bar
 * text by default. In order for that to be enabled, the users need to allow it
 * through tools->options->web features->javascript->advanced, or through the
 * config page available through the about:config URL (disable_window_status_change)
 *
 * @param strStatus The text to put in the application's status bar
 */

function setStatusBar(strStatus){
   if (strStatus == null) {
      strStatus="";
   }
   window.status = strStatus;
}

//==============================================================================
//End Window Status and title
//==============================================================================

//==============================================================================
// Begin javscript "server variables"
// Server variables are javascript variables that are populated when the system
// comes up with server side values.
// top.jsp has an init method to populate the variables in this section when
// the application comes up.
//==============================================================================

/**
 * @global_var boolean indicating whether the application is managing a single
 * ESX server or mutiple hosts through VC.
 * <p/>
 * Note! Populated by top.jsp:initSessionContextVariables() with a server side
 * session context value.
 */
var multiHost = true;

/**
 * Global string indicating the portion of the request URI that indicates the
 * context of the request.  The context path always comes first in a request
 * URI. The path starts with a "/" character but does not end with a "/"
 * character. For servlets in the default (root) context, this method returns
 * "".
 * <p/>
 * Note: populated by top.jsp
 */
var contextPath = null;

/**
 * Global string indicating the portion of the request URI that indicates the
 * name of the web server.
 * <p/>
 * Note: populated by top.jsp
 */
var serverName = null;

/**
 * Tells whether this application is managing a single ESX server or mutiple
 * hosts through VC.
 *
 * @return boolean
 *    true if we're in a multi-host environment; false otherwise
 */
function isMultiHost(){
   return multiHost;
}

/**
 * Sets the multiHost session context variable.  Typically populated when the
 * application comes up with a server side value.
 *
 * @param b
 *    true if we're in a multi-host environment; false if we're not
 * @throws Error
 *    If <code>b</code> is not a boolean value
 *
 */
function setMultiHost(b) {
   if (b == true || b == false) {
      multiHost = b;
   } else {
      throw new Error("While trying to set multiHost in tle value passed is not a boolean value");
   }
}

/**
 * Gets the value of the context path.
 *
 * @return String
 *    The current value of the context path
 */
function getContextPath() {
   return urlUtil.getPath();
}

/**
 * Sets the new value of the context path
 *
 * @param str String
 *    The new context path
 */
function setContextPath(str) {
   contextPath = str;
}

/**
 * Sets the new value of the server name
 *
 * @param str String
 *    The web server name.
 */
function setServerName(str) {
   serverName = str;
}

//==============================================================================
//End javscript "server variables"
//==============================================================================


/**
 * Converts a string to a DOM-node hierarchy.
 *
 * @param str String
 *    The xml string (e.g. "<errors><error>...</error></errors>")
 * @return [DOM Level 1 Core]Element
 *    A node representing the top level of the node hierarchy created from
 *    the string
 * @exception Error
 *    Thrown if the xml string appears to not be well-formed
 */
function strToDOM(str) {
   var suc = true;
   var doc = vpx.xua.cull(
      function() {
         var d = new ActiveXObject('Microsoft.XMLDOM');
         suc = d.loadXML(str);
         return d;
      },
      function() {
         var p = new DOMParser();
         return p.parseFromString(str, "text/xml");
      }
   );
   if (!suc) {
      throw new Error("parse error");
   }
   return doc.documentElement;
}

/**
 * Process errors according to the standard error XML contract:
 *    <errors redirect="[@optional:String:url]" class="[String:cssClass]">
 *       <error><![CDATA[String:errorMsg]]></error>
 *       ...
 *       <error><![CDATA[String:errorMsg]]></error>
 *    </errors>
 *
 * @param el [DOM Level 1 Core]Element
 *    The top-level <errors> node
 * @param doDefaultPage boolean
 *    true if you want the dialog to load the default webCenter page after the
 *    user acknowledges the error; false to leave the user on the current
 *    page after they acknowledge the error
 */
function processErrors(el, doDefaultPage) {

   var redir = el.getAttribute("redirect");
   if (redir != null) {
      window.location.href = redir;
      return;
   }

   var msgs = [];
   var iconClass = el.getAttribute("class");

   var errorNodes = el.getElementsByTagName("error");
   for (var i = 0; i < errorNodes.length; i++) {
      if (errorNodes[i].firstChild != null) {
         // XXX Assuming node only has one child: a CDATA_SECTION_NODE
         var msg = errorNodes[i].firstChild.nodeValue;
         if (msg != null && msg != "") {
            msgs.push(msg);
         }
      }
   }

   if (!doDefaultPage) {

     var handlerNode = el.getElementsByTagName("handler");
     if (!isNull(handlerNode) && handlerNode.length > 0) {
        doDefaultPage = handlerNode[0].getAttribute("type");
        var fatalHandler = handlerNode[0].getAttribute("fatal");
        // If the exception is fatal logout
        if (doDefaultPage == fatalHandler) {
           window.top.location.href = "logout.do";
           return;
        }
     }

   }

   if (msgs.length == 0) {
      // No errors to process
      return;
   }

   // Setup the error messages string
   var errorMsg = msgs[0];
   // TODO: uncomment the for loop when we support
   // chained exceptions.Insert bullets also.
  // for (var i = 0; i < msgs.length; i++) {
  //    errorMsg += msgs[i];
  // }



   toggleModal(true);

   // Determine the context in which errors dialog should live
   var context;
   try {
      context = getWorkspacePane();
      if (!context) {
         context = tle;
      }
   } catch (ex) {
      context = tle;
   }

   // Setup and show the errors dialog
   vpx.win.error(errorMsg, doDefaultPage, context);

   // We've handled the errors; remove them from the user's session
   removeSessionAttrs("com.vmware.webcenter.genericError");
}

//==============================================================================
// Begin client devices connector
// The client device connector is an instance of the vmware mks plugin.
// This one instance has no visual attributes. It is placed here
// in tle for 2 reasons:
//   1. It needs to persist throughout the application life cycle regardless of
//      main application refreshing, switching between different VMs, etc
//	 2. It needs to be accessible by different UI components. Unlike the visual
//			MKS console instance in the console page, this faceless instance will be
//			accessed by the toolbar, the menu, the summary page, and other future
//			unknown UI sections.
//
// Any document wishing to use the client devices connector should do so through
// tle.getClientDeviceConnector (tle is provided by including init.js in the page).
//
//==============================================================================

/**
* @global_var the window name used for the client devices connections window
*/
var CLIENT_DEVICES_WINDOW_NAME = "vmwareClientDevicesAuxiliaryWindow";

/**
* @global_var the URL to load into the client devices connections window
*/
var CLIENT_DEVICES_URL = "clientDeviceConnector.do";

/**
* @global_var object reference, holding a reference to the faceless instance of the VMware mks plugin
*
*/
var clientDeviceConnector = null;

/**
* @global_var object reference, holding a reference to the client devices connections window object
*
*/
var winClientDeviceConnector = null;


/**
 * Launches the client devices connections window and document within it.
 *
 * <p>
 * <ol>Scenarios:
 * <li>Connections window has never been opened and needs to be.</li>
 * <li>Has been opened but was closed and for some reason failed to unregister
 *    using clearClientDevicesConnectObjects()</li>
 * <li>Has been opened, but tle has been refreshed since and lost reference to
 *    the connections window.</li>
 * <li>Is open and feeling good.</li>
 * </ol>
 *
 * <p>
 * <ol>Solutions:
 * <li>winClientDeviceConnector reference variable is null, just open the child.</li>
 * <li>winClientDeviceConnector was previously open but is closed now. Make sure
 *    to NULLify and then deal with as if never opened.<br>
 *    Note:<br>
 *      - IE will not report it as null, use the closed attribute.<br>
 *      - Firefox will report it as null.</li>
 * <li>This cannot be handled here since there's no way to know that the window
 *    has already been opened, and even if, there's no way to get a reference to
 *    it again. Simply opening it again with the same name will trigger an
 *    onunload in the connections window which will promt the user with a
 *    question and will either fail or override the current connections document
 *    with its live connections object.
 *    Therefore, the client connections window implements an interval based
 *    lookup for tle to try and register with it again. Tle, onunload, will init
 *    that re-registration method and interval.
 *    [ It is reasonable to assume that the connections window will only be opened
 *    by some UI usage case, in which case tle is already up and the exising
 *    connections window was already able to re-register.
 *    Assuming this, scenario 3 becomes scenario 4. ]</li>
 * <li>Just focus on the window.</li>
 *
 * @param strFeatures visual window attributes to use when opening the window
 *
 * @todo deal with popup blockers !
 */

function launchClientDeviceConnector(strFeatures) {
  // case client device connection window was closed and did not notify tle of it
  if (winClientDeviceConnector != null && winClientDeviceConnector.closed) {
    winClientDeviceConnector = null;
  }
  if (winClientDeviceConnector == null ||
      (winClientDeviceConnector != null && winClientDeviceConnector.getPlugin == null)) {
      // case never opened,
      // reference lost and not regained,
      // or child window does not contain the plugin (browsed away from and did not unregister)
    winClientDeviceConnector = window.open(CLIENT_DEVICES_URL,
      CLIENT_DEVICES_WINDOW_NAME,strFeatures);
    /* case reference was lost and not recovered (should not happen),
       and original pop up is minimized, window will be reused, but will not pop up.
       Focus on the minimized window.
    */
    winClientDeviceConnector.focus();
  } else { // case open and is available
    winClientDeviceConnector.focus();
  }
}

/**
 * Clears references to the client devices connections window and plugin
 */

function clearClientDevicesConnectObjects(){
  clientDeviceConnector = null;
  winClientDeviceConnector = null;
}

/**
 * sets the reference to the client devices connections window.
 * <p>
 * Note: is used by the connections window to re-register with tle in
 * cases tle comes back up after being reloaded.
 */

function setClientDeviceConnectorWindow(objWinConnector){
  winClientDeviceConnector = objWinConnector;
}

/**
 * Closes the client devices connections window
 */
function closeClientDeviceConnector(){
  if (winClientDeviceConnector != null) {
    winClientDeviceConnector.close();
  }
}

/**
 * Sets the reference to the client devices connections plugin.
 * <p>
 * Note: is called in the connections window's onload event to register the
 * plugin with tle.
 */
function setClientDeviceConnector(objConnector){
  clientDeviceConnector = objConnector;
}

/**
 * Retrieves the reference to the client devices connections plugin.
 * <p>
 * To call this method from any document in the application include init.js and
 * call tle.getClientDeviceConnector().
 */
function getClientDeviceConnector(){
  return clientDeviceConnector;
}

//==============================================================================
//End client devices connector
//==============================================================================

//==============================================================================
// Begin UI onunload
// Code executed when top.jsp (and therefore the entire application unloads).
// 1. If there's a client devices connections window open, trigger its interval
// 		for re-registering with tle once tle comes back.
//==============================================================================

function notifyAppUnload() {
   if (requestPool != null) {
      requestPool.destroy();
      requestPool == null;
   }

   // 1
   if (winClientDeviceConnector != null &&
       winClientDeviceConnector.initRegisterWithTle != null) {
      winClientDeviceConnector.initRegisterWithTle();
   }

   if (updatesAgent != null && updatesAgent.isRunning()) {
      updatesAgent.stop();
   }
}

//==============================================================================
// End UI onunload
//==============================================================================


//==============================================================================
// Begin Misc
//============================================================================

function UrlUtil() {
   this.fqdn = null;
   this.host = null;
   this.port = null;
   this.path = null;
   this.scheme = null;
}

UrlUtil.prototype.getScheme = function () {
   if (!isNull(this.scheme)) {
      return this.scheme;
   }

   this.scheme = window.location.protocol;
   return this.scheme;
}

UrlUtil.prototype.getHost = function () {
   if (!isNull(this.host)) {
      return this.host;
   }

   if (!isNull(serverName)) {
      this.host = serverName;
      return this.host;
   }

   this.host = window.location.host.split(":")[0];
   return this.host;
}

UrlUtil.prototype.getPort = function () {
   if (!isNull(this.port)) {
      return this.port;
   }

   this.port = window.location.port;
   return this.port;
}

UrlUtil.prototype.getPath = function () {
   if (!isNull(this.path)) {
      return this.path;
   }

   if (!isNull(contextPath)) {
      this.path = contextPath;
      return this.path;
   }

   // Initialize var (should have been done in top.jsp)
   var prefix = window.location.pathname.split("/")[1];
   if (prefix != "" && prefix.search(/\./) == -1) {
      this.path = "/" + prefix;
   } else {
      // Last ditch effort to set contextPath
      this.path = "/ui";
   }

   return this.path;
}

UrlUtil.prototype.getFqdn = function () {
   if (!isNull(this.fqdn)) {
      return this.fqdn;
   }

   var urlScheme  = this.getScheme();
   var serverName = this.getHost();
   var serverPort = this.getPort();
   if (!isNull(serverPort) && serverPort != "") {
      if (urlScheme == "http:" && serverPort != "80") {
         serverName += ":" + serverPort;
      } else if (urlScheme == "https:" && serverPort != "443") {
         serverName += ":" + serverPort;
      }
   }

   this.fqdn = urlScheme + "//" + serverName;
   return this.fqdn;
}

var urlUtil = new UrlUtil();

/**
 * Parses out the host name from the tle window location
 * <p>
 *
 * @return the host name without up to the port number (":" excluded)
 *
 * Note: Needed in cases where server tickets report 127.0.0.1 as host name
 * (when the web server and web service are running on the same machine).
 */
function getLocationHostName(){
  return urlUtil.getHost();
}

//==============================================================================
// End Misc
//==============================================================================
