/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * Constructor for a new EventSource object.  This class is abstract, however,
 * so users should not invoke this constructor directly.  Any attempt to do so
 * will result in an error being thrown.
 */
vpx.EventSource = function() {
	// Abstract class
	if (arguments.length != 1 || arguments[0] != null) {
		throw new Error("EventSource is abstract: cannot be instantiated");
	}

	if (this.instantiated) {
		// Base class supports no events; subclasses must add support
		this.listeners = {};
	}
};

vpx.EventSource._seqId = 0;

/**
 * Adds the given callback to the list of listeners for the given event
 * type. When this object fires an event of the given type, it will run all of
 * its registered callback functions in they order that they were added.
 *
 * @param eventType const int
 *		The event type (e.g. vpx.browser.EVENT.DATA)
 * @param callback Function
 *		The function to invoke when the event is fired
 * @return int
 *		A listenerId that can be used to later remove this listener
 * @throws Error
 *		Thrown if the given event type is not supported by this object
 */
vpx.EventSource.prototype.registerListener = function(eventType, callback) {
	var array = this._getListenerArray(eventType);
	var listener = {
		id			: ++vpx.EventSource._seqId,
		callback : callback
	};
	array.push(listener);
	return listener.id;
};

/**
 * Removes the given listener from the list of listeners for the given event
 * type.	 If the listener is not found in the list for the given event type,
 * no action is taken.
 *
 * @param eventType const int
 *		The event type (e.g. vpx.browser.EVENT.DATA)
 * @param listenerId int
 *		The id returned upon adding the listener
 * @throws Error
 *		Thrown if the given event type is not supported by this object
 */
vpx.EventSource.prototype.unregisterListener = function(eventType, listenerId) {
	var array = this._getListenerArray(eventType);
	for (var i = 0; i < array.length; i++) {
		if (array[i].id == listenerId) {
			array.splice(i, 1);
			return;
		}
	}
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/*
 * Fires an event of the given type, causing all registered listeners to get
 * notified (i.e. their callback routines to get executed).
 *
 * All listeners will be invoked with the first argument being this object
 * that triggered the event.
 *
 * @param eventType const int
 *    The event type (e.g. vpx.browser.EVENT.DATA)
 * @param arguments [variable]
 *    Any other arguments will be passed to the listeners after the first
 *    argument (this object)
 * @throws Error
 *    Thrown if the given event type is not supported by this object
 */
vpx.EventSource.prototype._fireEvent = function(eventType) {
   var i;
   vpx.log.trace("EventSource#_fireEvent(): Firing event of type " + eventType);
   var array = this._getListenerArray(eventType);

   var args = [this];
   for (i = 1; i < arguments.length; i++) {
      args.push(arguments[i]);
   }

   for (i = 0; i < array.length; i++) {
      array[i].callback.apply(self, args);
   }
};

/*
 * Gets the array of listeners for the given event type, or throws an error if
 * the event type is unsupported.  There will always be an array of listeners
 * for a supported event type.  If there are no listeners for that event type,
 * the array will be empty.
 *
 * @param eventType const int
 *    The event type (e.g. vpx.browser.EVENT.DATA)
 * @return Object[]
 *    An array of listener objects (each of which has an id and a callback)
 * @throws Error
 *    Thrown if the given event type is not supported by this object
 */
vpx.EventSource.prototype._getListenerArray = function(eventType) {
   var array = this.listeners[eventType];
   if (isNull(array)) {
      throw new Error("EventSource#_getListenerArray(): unsupported event type: " + eventType);
   }
   return array;
};
