/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class Browser
 * extends vpx.browser.Paged
 * implements vpx.net.event.ResponseListener
 *
 * TODO
 */

/**
 * Constructs a new Browser object and initializes its GUI.  This object starts
 * out with no nodes; the caller is responsible for creating an XmlSpec object
 * and making the initial request for data.
 *
 * @param container HTMLElement
 *    The DOM element in which this paginator's GUI is to be created
 * @param pageSize int
 *    The maximum number of items to fit on each page
 * @param heightPolicy object
 *    true if the height of the browser should be fixed based on the pageSize;
 *    false if it should expand and collapse as necessary based on the number
 *    of nodes at any given time; "user" if the API user should set the browser
 *    height manually and listen for DATA events to know when a resize may be
 *    required
 * @param headerVisible boolean
 *    true if the browser should show its column header row; false otherwise
 */
vpx.browser.Browser = function(container, pageSize, heightPolicy, headerVisible) {
   this.instantiated = true;

   // super();
   vpx.browser.Paged.call(this, null);

   // Override default Paged values
   this.pageSize = pageSize;
   this.listeners[vpx.browser.EVENT.SELECTION]     = [];
   this.listeners[vpx.browser.EVENT.NAVIGATE]      = [];
   this.listeners[vpx.browser.EVENT.SORT]          = [];
   this.listeners[vpx.browser.EVENT.ERROR]         = [];

   // Browser instance variables
   this.sortSpec = {type:vpx.browser.SORT.NONE, column:null};
   this.nodes = [];
   this.selected = [];
   this.columns = [];
   this.columnLookup = {};
   this.hierarchy = [];
   this.curr = null;
   this.colRefs = {};
   this.nodeRefs = {};

   this._initGui(container, heightPolicy, headerVisible);
};

/**
 * The types of vertical resizing that can be done.
 */
vpx.browser.Browser.HEIGHT_POLICY = {
   "FIT"   : false,
   "FIXED" : true,
   "USER"  : "user"
};



// Browser extends Paged
vpx.browser.Browser.prototype = new vpx.browser.Paged(null);
vpx.browser.Browser.prototype.constructor = vpx.browser.Browser;

/**
 * Gets this browser's array of columns.  If the browser doesn't have any
 * columns, this will return an empty array.
 *
 * @return vpx.browser.Column[]
 */
vpx.browser.Browser.prototype.getColumns = function() {
   return this.columns;
};

/**
 *
 */
vpx.browser.Browser.prototype.indexOfCol = function(column) {
   var index = this.columnLookup[column.getName()];
   if (isNull(index)) {
      index = this.columns.indexOf(column);
      this.columnLookup[column.getName()] = index;
   }
   return index;
};

/**
 * Gets the list of browser nodes currently visible on the page.  If there are
 * none, this returns an empty array.
 *
 * @return vpx.browser.Node[]
 */
vpx.browser.Browser.prototype.getNodes = function() {
   return this.nodes;
};

/**
 * Gets the page size of this browser.
 *
 * @return int
 *    The maximum number of items to fit on each page
 */
vpx.browser.Browser.prototype.getPageSize = function() {
   return this.pageSize;
};

/**
 * Requests data from the server according to the given specs.  Once the
 * server responds, this browser will automatically process it according to
 * the standard Browser xml processing contract.
 *
 * @param xmlSpec vpx.net.XmlSpec
 *    The spec with which to request server data
 */
vpx.browser.Browser.prototype.requestData = function(xmlSpec) {
   vpx.log.trace("Browser#requestData(): Requesting data from server");

   var pool = vpx.net.HttpRequestPool.getInstance();
   var req = pool.getRequest();
   if (req == null) {
      // Pool is empty; Reschedule request in a little bit
      vpx.log.info("Browser#requestData(): No requests available; rescheduling");
      window.setTimeout(this.requestData.bind(this, xmlSpec), 500);
      return;
   }

   try {
      req.setUrl(xmlSpec.toUrl());
      req.addResponseListener(this);
      req.send();
      this.curr = {
         req : req,
         xmlSpec : xmlSpec
      };
      //Set Loading status
      this._toggleLoading(true);
   } finally {
      pool.releaseRequest(req);
   }
};

/**
 * Tells whether or not the header row is visible.
 *
 * @return boolean
 *    true if the header is visible; false otherwise
 */
vpx.browser.Browser.prototype.isHeaderVisible = function() {
   return this.gui.usrHeaderVisible;
};

/**
 * Sets whether or not the header row is visible. The browser may, at its
 * discretion, temporarily ignore this value if it contains no nodes.
 * In this case, the browser will not show the header row even if the user
 * has instructed that the header row be visible.
 *
 * @param bool boolean
 *    true to set the header to be visible; false to make it hidden
 */
vpx.browser.Browser.prototype.setHeaderVisible = function(bool) {
    if (this.gui.usrHeaderVisible == bool) {
        return;
    }
    this.gui.usrHeaderVisible = bool;
    if (!bool || this.nodes.length > 0) {
       this._setSysHeaderVisible(bool);
    }
};

/**
 * Gets the list of currently selected nodes.  If no nodes are selected, this
 * will return an empty array.
 *
 * @return vpx.browser.Node[]
 *    Array of selected nodes
 */
vpx.browser.Browser.prototype.getSelected = function() {
   return this.selected;
};

/**
 * Sets the current selection to the given nodes, clearing any existing
 * selection first.
 *
 * @param nodes vpx.browser.Node[]
 *    Array of nodes to select. To select none, you may use an empty array or
 *    null
 * @param autoScroll boolean
 *    true to automatically scroll the selection into view; false to leave the
 *    scroll position of the Browser alone
 */
vpx.browser.Browser.prototype.select = function(nodes, autoScroll) {
   vpx.log.trace("Browser#select(): Selecting nodes: " + nodes);

   var selectionChanged = !this.selected.equals(nodes);

   this._unselectAll();
   if (isDefined(nodes)) {
      for (var i = 0; i < nodes.length; i++) {
         if (nodes[i].isSelectable) {
            nodes[i].setSelected(true);
            this.selected.push(nodes[i]);
         }
      }
   }

   if (selectionChanged) {
      if (autoScroll && this.selected.length > 0) {
         // Can only guarantee one is visible - we choose first one
         this.selected[0].scrollIntoView();
      }
      this._fireEvent(vpx.browser.EVENT.SELECTION);
   }
};

/**
 * Selects the nodes in between the two given nodes, inclusive of the two
 * endpoints.
 *
 * @param node1 vpx.browser.Node
 *    The first node
 * @param node2 vpx.browser.Node
 *    The second node. This need not appear after the first node
 * @param autoScroll boolean
 *    true to automatically scroll the selection into view; false to leave the
 *    scroll position of the Browser alone
 * @throws Error
 *    Thrown if either node is not found in the list of nodes
 */
vpx.browser.Browser.prototype.selectBetween = function(node1, node2, autoScroll) {
   var index1 = this._indexOf(node1);
   var index2 = this._indexOf(node2);
   if (index1 == -1 || index2 == -1) {
      throw new Error("Browser#selectBetween(): un-attached node found");
   }

   var start = Math.min(index1, index2);
   var end = Math.max(index1, index2);
   var toSelect = [];
   for (var i = start; i <= end; i++) {
      toSelect.push(this.nodes[i]);
   }
   this.select(toSelect, autoScroll);
};

/**
 * Gets the current location of this browser in the hierarchy.
 *
 * @return vpx.browser.Location
 *    The current location, or null if no such location exists
 */
vpx.browser.Browser.prototype.getLocation = function() {
   if (this.hierarchy.length == 0) {
      return null;
   }
   return this.hierarchy[this.hierarchy.length - 1];
};

/**
 * Navigates this browser's location to the given base item.
 *
 * @param baseItem vpx.ServerObj
 *    The base item to which to navigate our browser
 */
vpx.browser.Browser.prototype.setLocation = function(baseItem) {
   this._fireEvent(vpx.browser.EVENT.NAVIGATE, baseItem);
};

/**
 * Handles a mouse click on one of this browser's nodes.
 *
 * TODO: Add shift & ctrl handling for multi-select
 *
 * @param event Event
 *    The event fired from the user agent. Can be used to detect whether the
 *    shift or ctrl key was held down at the time of the click
 * @param node vpx.browser.Node
 *    The node that was clicked
 */
vpx.browser.Browser.prototype.handleSelectNode = function(event, node) {
   vpx.log.trace("Browser#handleSelectNode(): Registered node select: " + node);

   var alreadySelected = (this.selected.length == 1 && this.selected[0] == node);
   if (alreadySelected && node instanceof vpx.browser.BranchNode) {
      // We only navigate if the node was the only selected one
      this.setLocation(node);
   } else if (!alreadySelected) {
      if (node.isSelectable) {
         this.select([node], false);
      }
   }
};

/**
 * Handles a mouse click on a column header.
 *
 * @param column vpx.browser.Column
 *    The name of the column
 * @param event Event
 *    The event fired from the user agent.
 */
vpx.browser.Browser.prototype.handleSort = function(column, event) {
   if (!column.canSort()) {
      return;
   }
   var type;
   if (this.sortSpec.column == column && this.sortSpec.type == vpx.browser.SORT.ASC) {
      type = vpx.browser.SORT.DESC;
   } else {
      type = vpx.browser.SORT.ASC;
   }
   this._fireEvent(vpx.browser.EVENT.SORT, column, type);
};

/**
 * Retrieves a Column by its name.  Each column's name is unique among
 * all Column instances.
 *
 * @param name String
 * @return vpx.browser.Column
 */
vpx.browser.Browser.prototype.getColByName = function(name) {
   if (!this.colRefs[name]) {
      return null;
   }
   return this.colRefs[name];
};

/**
 * @param id int
 * @return vpx.browser.Node
 */
vpx.browser.Browser.prototype.getNodeById = function(id) {
   if (!this.nodeRefs[id]) {
      return null;
   }
   return this.nodeRefs[id];
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Browser.prototype.toString = function() {
   return "[Object Browser]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * @param container HTMLElement
 *    The DOM element in which this paginator's GUI is to be created
 * @param heightPolicy object
 *    true if the height of the browser should be fixed based on the pageSize;
 *    false if it should expand and collapse as necessary based on the number
 *    of nodes at any given time; "user" if the API user should set the browser
 *    height manually and listen for DATA events to know when a resize may be
 *    required
 * @param headerVisible boolean
 *    true if the browser should show its column header row; false otherwise
 */
vpx.browser.Browser.prototype._initGui = function(container, heightPolicy, headerVisible) {
   this.gui = {
      usrHeaderVisible : headerVisible,
      sysHeaderVisible : headerVisible,
      container        : container,
      heightSpec       : {policy:heightPolicy, complete:false},
      columns          : [],
      header           : null,
      table            : null,
      tbody            : null,
      loading          : null
   };
   var table, tbody, tr;


   // XXX: When cells in the browser table are updated, IE reflows the table
   // such that it spills over the right border of the containing element. If
   // we change the width of the outter elment to 100%, the table will not
   // spill over.
   if (vpx.xua.ie) {
      vpx.xua.setStyle(container.parentNode, "width", "100%");
   }

   // Container
   vpx.xua.setStyle(container, "width", "100%", "overflow", "auto");

   // Table
   table = document.createElement("table");
   vpx.xua.setAttribute(table,
      "class", "listView",
      "border", "0",
      "cellpadding", "0",
      "cellspacing", "0",
      "width", "100%");
   container.appendChild(table);
   this.gui.table = table;

   // Tbody
   tbody = table.getElementsByTagName("TBODY")[0];
   if (!tbody) {
      tbody = document.createElement("tbody");
      table.appendChild(tbody);
   }
   this.gui.tbody = tbody;

   // Header
   tr = document.createElement("tr");
   this.gui.header = tr;
   // visible display should be "table-row", but IE breaks when that is used
   tr.style.display = (headerVisible ? "" : "none");
   tbody.appendChild(tr);

   //Loading status div
   /* Give the container height while loading */
   vpx.xua.setHeight(container, 50);
   this.gui.loading = document.createElement("div");
   vpx.xua.setStyle(this.gui.loading,
                    "border", "none",
		    "position", "absolute");
   container.appendChild(this.gui.loading);
   this.gui.loading.innerHTML = '<table border="0" cellpadding="0" cellspacing="0">' +
      '<tr><td><div class="title" style="display:inline; padding-left:16px; ' +
      'background-image:url(imx/loading-9x9.gif); background-repeat:no-repeat; ' +
      'background-position:0px 3px;">' + strLoading + '</div></td></tr>';

   this._toggleLoading(true);

   /* The following is commented out to fix teststopper 109295.
    * July 7, 2006
    * _setGuiHeight() calls _setGuiOverflow() which calls _resizeGuiHeight().
    * _resizeGuiHeight resizes the height of the browser's container, which
    *    then causes this listener to call _setGuiHeight.
    * In Firefox, the container's height is updated so the loop ends there.
    * In IE, the same height is returned over and over, so the height is
    *    constantly resized, resulting in an infinite loop.
    */
   // vpx.xua.event.listen(vpx.xua.getView(this.gui.container), "resize",
   //   this._setGuiHeight.bind(this));
};

/*
 * Displays the Loading status indicator.
 * Sets the browser table to be translucent, and displays the loading message.
 *
 * @param loading bool
 *    True if loading status is to be displayed.
 *
 */
vpx.browser.Browser.prototype._toggleLoading = function(loading) {
   if (loading) {
      vpx.xua.setOpacity(this.gui.table, 0.1);
      vpx.xua.setStyle(this.gui.loading,
                       "display", "");
      vpx.xua.center(this.gui.loading, vpx.xua.getBoundary(this.gui.container));
   } else {
      vpx.xua.setOpacity(this.gui.table, 1);
      vpx.xua.setStyle(this.gui.loading,
                       "display", "none");
   }
};

/*
 * Appends the given node onto the end of the list of nodes.
 *
 * @param node vpx.browser.Node
 *    The node to append
 * @throws Error
 *    Thrown if the page is already full
 */
vpx.browser.Browser.prototype._append = function(node) {
   if (this.nodes.length == this.pageSize) {
      throw new Error("Browser#_append(): Cannot append node - exceeds pageSize");
   }
   this.nodes.push(node);
   this.gui.tbody.appendChild(node.gui.row);
};

/*
 * Removes the given node from the list of nodes and disposes of it.
 *
 * @param node vpx.browser.Node
 *    The node to remove
 */
vpx.browser.Browser.prototype._remove = function(node) {
   var i = this._indexOf(node);
   node.destroy();
   this.gui.tbody.removeChild(node.gui.row);
   this.nodes.splice(i, 1);
};

/*
 * Empties the list of nodes from this browser, disposing of each one first.
 */
vpx.browser.Browser.prototype._clear = function() {
   for (var i = 0; i < this.nodes.length; i++) {
      var node = this.nodes[i];
      node.destroy();
      this.gui.tbody.removeChild(node.gui.row);
   }
   this.nodes = [];
};

/*
 * Gets the 0-based index of the given node in the list of nodes, or -1 if the
 * node is not in the list.
 *
 * @param node vpx.browser.Node
 *    The node in question
 * @return int
 *    The 0-based index of the node
 */
vpx.browser.Browser.prototype._indexOf = function(node) {
   for (var i = 0; i < this.nodes.length; i++) {
      if (this.nodes[i] == node) {
         return i;
      }
   }
   return -1;
};

/*
 * Gets the node located at the given 0-based index.
 *
 * @param index int
 *    The index at which to look
 * @return vpx.browser.Node
 *    The node at the specified index
 * @throws Error
 *    Thrown if the given index is out of bounds
 */
vpx.browser.Browser.prototype._nodeAt = function(index) {
   if (index < 0 || index >= this.nodes.length) {
      throw new Error("Browser#_nodeAt(): Index out of bounds: " + index);
   }
   return this.nodes[index];
};

/*
 * Sets whether or not the header row is visible.
 *
 * @param bool boolean
 *    true to set the header to be visible; false to make it hidden
 */
vpx.browser.Browser.prototype._setSysHeaderVisible = function(bool) {
   if (this.gui.sysHeaderVisible == bool) {
      return;
   }
   this.gui.sysHeaderVisible = bool;
   // visible display should be "table-row", but IE breaks when that is used
   this.gui.header.style.display = (bool ? "" : "none");
   this._setGuiHeight();
};

/*
 * Processes data from the server.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The response event generated from the server response
 * @see vpx.net.event.ResponseListener#responseReceived(vpx.net.event.ResponseEvent)
 */
vpx.browser.Browser.prototype.responseReceived = function(e) {
   var resp = e.getSource();
   var req = e.getRequest();

   if (this.curr == null || req != this.curr.req) {
      // We've since fired another request; ignore this [now stale] response
      return;
   }
   var xmlSpec = this.curr.xmlSpec;
   this.curr = null;

   var status = resp.getStatus();
   if (status != 200) {
      // Only proceed if HTTP status is "OK"
      var url = req.getUrl();
      var msg = resp.getStatusMsg();
      switch (status) {
      case 404:
         throw new Error("Browser#responseReceived(): 404 Not Found: " + url);
      case 500:
         throw new Error("Browser#responseReceived(): 500 Application Error: " + url);
      default:
         // If server didn't process the request, we must re-send
         vpx.log.warn("Agent#responseReceived(): " + status + " " + msg + ". Resending");
         this.requestData(xmlSpec);
         return;
      }
   }

   var contentType = resp.getContentType();
   if (contentType != "text/xml") {
      throw new Error("Browser#responseReceived(): invalid content-type: " +
                      contentType + " (status " + status +
                      ") while processing " + req.getUrl());
   }

   var xml = resp.getXml();
   vpx.log.trace("Browser#_processData(): Received xml: " + xml);
   var root = xml.documentElement;
   if (root == null) {
      // XML was not well-formed
      throw new Error("Browser#responseReceived(): xml was not well-formed " +
                      "while processing " + req.getUrl());
   }
   root.normalize();

   //Hide the loading div
   this._toggleLoading(false);

   if (root.nodeName == "errors") {
      this._clear();
      this._processErrors(xml, root);
      this._fireEvent(vpx.browser.EVENT.DATA, xmlSpec);
      this._fireEvent(vpx.browser.EVENT.ERROR);
      return;
   }

   // Server uses 1-based page numbers; covert to 0-based
   this.page = parseInt(root.getAttribute("page")) - 1;
   this.totalItems = parseInt(root.getAttribute("total"));

   this._clear();
   this._processTableStructure(xml, root);
   this._processColumns(xml, root);
   this._processNodes(xml, root);
   this._processLocations(xml, root);
   this._processSorting(xml, root);
   this._processAttrs(xml, root, xmlSpec);
   this._processSelections(xml, root);
   this._fireEvent(vpx.browser.EVENT.DATA, xmlSpec);

   if (this.nodes.length == 0) {
      // Header is not shown when there are no rows
      this._setSysHeaderVisible(false);
   } else if (this.gui.usrHeaderVisible) {
      this._setSysHeaderVisible(true);
   }
};

/**
 *
 */
vpx.browser.Browser.prototype._processErrors = function(xml, root) {
   // Setup table structure & columns
   var heading = root.getAttribute("heading");
   this.gui.columns.length = 1;
   var columnGui = this.gui.columns[0] = { colspan : 1 };
   var columnObj = new vpx.browser.Column(this, "errorMsg", heading, false, true, "left");
   this.columns.push(columnObj);
   var th = columnGui.th = document.createElement("th");
   var div = columnGui.div = document.createElement("div");
   th.appendChild(div);
   th.colSpan = 2;
   th.style.width = "100%";
   div.innerHTML = columnObj.displayName;
   th.style.cursor = "default";
   while (this.gui.header.childNodes.length > 0) {
      this.gui.header.removeChild(this.gui.header.lastChild);
   }
   this.gui.header.appendChild(th);

   // Setup error "nodes" (rows)
   var iconClass = root.getAttribute("class");
   var errorNodes = root.getElementsByTagName("error");
   for (var i = 0; i < errorNodes.length; i++) {
      if (errorNodes[i].firstChild != null) {
         // XXX Assuming node only has one child: a CDATA_SECTION_NODE
         var msg = errorNodes[i].firstChild.nodeValue;
         if (msg != null && msg != "") {
            var node = new vpx.browser.Node(this, "#error" + i + "#", iconClass);
            node._addCell(0, null, msg, this.gui.columns[0].colSpan);
            this._append(node);
         }
      }
   }

   // Set navigator to not contain any nodes
   this.hierarchy.length = 0;

   // Reset table structure for next pass
   this.columns.length = 0;
   this.gui.columns.length = 0;
   this.columnLookup = {};

   this._setGuiHeight();
};

/*
 *
 */
vpx.browser.Browser.prototype._processTableStructure = function(xml, root) {
   var i;
   var colsNode = root.getElementsByTagName("cols")[0];
   var numCols = colsNode.getElementsByTagName("col").length;

   //?? XXX Re-initialize this.columns if necessary?

   // Initialize array
   this.gui.columns.length = numCols;
   for (i = 0; i < numCols; i++) {
      if (isNull(this.gui.columns[i])) {
         this.gui.columns[i] = {};
      }
      this.gui.columns[i].colSpan = 0;
   }

   var nodes = root.getElementsByTagName("node");
   for (i = 0; i < nodes.length; i++) {
      var cols = nodes[i].getElementsByTagName("col");
      if (cols.length != numCols) {
         throw new Error("Browser#_processTableStructure(): Expected " +
                         numCols + " columns; found " + cols.length);
      }
      for (var j = 0; j < numCols; j++) {
         var cells = cols[j].getElementsByTagName("cell");
         if (cells.length > this.gui.columns[j].colSpan) {
            this.gui.columns[j].colSpan = cells.length;
         }
      }
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processColumns = function(xml, root) {
   //?? TODO compare columns against existing & sync if necessary
   if (this.columns.length == 0) {
      while (this.gui.header.childNodes.length > 0) {
         this.gui.header.removeChild(this.gui.header.lastChild);
      }
      if (vpx.FEATURE.TRAVERSAL) {
          this._processColumnsTraversal(xml, root);
      } else {
          this._processColumnsCore(xml, root);
      }
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processColumnsTraversal = function(xml, root) {
   var whatToShow = NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_CDATA_SECTION;
   var filter = this._filterFunc(/^(locations|nodes|attrs|selections)$/, "cols");
   var tw = xml.createTreeWalker(root, whatToShow, filter, false);
   var i = 0;
   for (var node = tw.firstChild(); node != null; node = tw.nextSibling()) {
      this._processColumn(i++, node);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processColumnsCore = function(xml, root) {
   var colsNode = root.getElementsByTagName("cols")[0];
   var cols = colsNode.getElementsByTagName("col");
   for (var i = 0; i < cols.length; i++) {
      this._processColumn(i, cols[i]);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processColumn = function(colIndex, node) {
   var columnObj = vpx.browser.Column.fromXml(this, node);
   var columnGui = this.gui.columns[colIndex];
   this.columns.push(columnObj);
   this.columnLookup = {};

   var th = columnGui.th = document.createElement("th");
   var div = columnGui.div = document.createElement("div");
   th.appendChild(div);
   if (colIndex == 0) {
      // First column spans two cells to account for built-in icon
      th.colSpan = 2;
   } else if (columnGui.colSpan > 1) {
      th.colSpan = columnGui.colSpan;
   }
   div.innerHTML = columnObj.displayName;
   th.style.cursor = "default";
   var sortHandler = vpx.browser.Browser.prototype.handleSort.bind(this, columnObj);
   vpx.xua.event.listen(th, "click", sortHandler);
   this.gui.header.appendChild(th);
};

/*
 *
 */
vpx.browser.Browser.prototype._processNodes = function(xml, root) {
   if (vpx.FEATURE.TRAVERSAL) {
      this._processNodesTraversal(xml, root);
   } else {
      this._processNodesCore(xml, root);
   }
   this._setGuiHeight();
};

/*
 *
 */
vpx.browser.Browser.prototype._processNodesTraversal = function(xml, root) {
   var whatToShow = NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_CDATA_SECTION;
   var filter = this._filterFunc(/^(locations|cols|attrs|selections)$/, "nodes");
   var tw = xml.createTreeWalker(root, whatToShow, filter, false);
   for (var node = tw.firstChild(); node != null; node = tw.nextSibling()) {
      node = vpx.browser.Node.fromXml(this, node);
      this._append(node);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processNodesCore = function(xml, root) {
   var nodes = root.getElementsByTagName("node");
   for (var i = 0; i < nodes.length; i++) {
      var node = vpx.browser.Node.fromXml(this, nodes[i]);
      this._append(node);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processLocations = function(xml, root) {
   if (vpx.FEATURE.TRAVERSAL) {
      this._processLocationsTraversal(xml, root);
   } else {
      this._processLocationsCore(xml, root);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processLocationsTraversal = function(xml, root) {
   var whatToShow = NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_CDATA_SECTION;
   var filter = this._filterFunc(/^(cols|nodes|attrs|selections)$/, "locations");
   var tw = xml.createTreeWalker(root, whatToShow, filter, false);

   // Clear the existing hierarchy
   this.hierarchy.length = 0;

   for (var node = tw.firstChild(); node != null; node = tw.nextSibling()) {
      var location = vpx.browser.Location.fromXml(this, node);
      this.hierarchy.push(location);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processLocationsCore = function(xml, root) {
   // Clear the existing hierarchy
   this.hierarchy.length = 0;

   var locations = root.getElementsByTagName("location");
   for (var i = 0; i < locations.length; i++) {
      var location = vpx.browser.Location.fromXml(this, locations[i]);
      this.hierarchy.push(location);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processSorting = function(xml, root) {
   var colsNode = root.getElementsByTagName("cols")[0];
   var sort = colsNode.getAttribute("sort");
   if (isDefined(sort) && sort != "") {
      var type = vpx.browser.Browser._sortTypeFromDir(colsNode.getAttribute("dir"));
      var col = this.getColByName(sort);
      if (isNull(col)) {
         throw new Error("Browser#_processSorting(): Unknown sort column: " + sort);
      }
      this._setSort(type, col);
   }
};

/*
 *
 */
vpx.browser.Browser.prototype._processAttrs = function(xml, root, xmlSpec) {
   var i;
   var attrsNode = null;
   // root.getElementsByTagName("attrs") will return unwanted nodes
   for (i = 0; i < root.childNodes.length; i++) {
      var child = root.childNodes[i];
      if (child.nodeType == vpx.xua.ELEMENT_NODE && child.nodeName == "attrs") {
         attrsNode = child;
         break;
      }
   }
   if (attrsNode != null) {
      var attrNodes = attrsNode.getElementsByTagName("attr");
      for (i = 0; i < attrNodes.length; i++) {
         var key = attrNodes[i].getAttribute("name");
         var value = attrNodes[i].getAttribute("value");
         xmlSpec.setAttribute(key, value);
      }
   }
};

/*
 * Processes the &lt;selections&gt; node of a server xml response (to a request
 * for data).  This node allows the server to specify which nodes should be
 * automatically selected.  This function will only fire a
 * <code>SELECTION</code> event if the selected nodes has <i>changed</i>.  That
 * is: if the selected nodes were the same before data processing as after data
 * processing, no event will be fired.
 *
 * @param xml Document
 *    The XML document returned from the HttpRequest
 * @param root Element
 *    The root element of the XML document
 */
vpx.browser.Browser.prototype._processSelections = function(xml, root) {
   var newSelected = [];
   var selectionsNode = root.getElementsByTagName("selections")[0];
   if (isDefined(selectionsNode)) {
      var selectionNodes = selectionsNode.getElementsByTagName("selection");
      for (var i = 0; i < selectionNodes.length; i++) {
         var nodeId = selectionNodes[i].getAttribute("id");
         var node = this.getNodeById(nodeId);
         if (node == null) {
            throw new Error("Browser#_processSelections(): can't select node '" +
                            nodeId + "'; no such node");
         }
         if (!node.isSelectable) {
            throw new Error("Browser#_processSelections(): can't select node '" +
                            nodeId + "'; node is not selectable");
         }
         newSelected.push(node);
      }
   }

   if (newSelected.length == 0) {
      var l = this.getLocation();
      if (l != null) {
         // No server-side selection translates to selecting Location on client
         newSelected.push(l);
      }
   }

   this.select(newSelected, true);
};

/*
 *
 */
vpx.browser.Browser.prototype._unselectAll = function() {
   for (var i = 0; i < this.selected.length; i++) {
       this.selected[i].setSelected(false);
   }
   this.selected = [];
};

/**
 *
 */
vpx.browser.Browser.prototype._updateColumnSortingGui = function(type, column) {
   if (isDefined(this.sortSpec.column)) {
      // Remove sorting gui for old sort column
      this.gui.columns[this.indexOfCol(this.sortSpec.column)].div.className = null;
   }
   var colIndex = this.indexOfCol(column);
   var div = this.gui.columns[colIndex].div;
   div.className = (type == vpx.browser.SORT.ASC ? "sortedUp" : "sortedDown");
};

/*
 *
 * @param type const int
 *    The type of sort ("ASC", "DESC", or "NONE"). If NONE, column will be ignored
 * @param column vpx.browser.Column
 *    The column on which to sort; optional if specified NONE as the sorting type
 * @throws Error
 *    Thrown if the column is not sortable or null (when type is ASC or DESC)
 */
vpx.browser.Browser.prototype._setSort = function(type, column) {
   this._updateColumnSortingGui(type, column);

   this.sortSpec.type = type;
   if (type == vpx.browser.SORT.NONE) {
      this.sortSpec.column = null;
   } else {
      if (isNull(column) || !column.canSort()) {
         throw new Error("Browser#sort(): column not sortable: " + column);
      }
      this.sortSpec.column = column;
   }
};

/*
 * Sets the height of the GUI's container as necessary. If this browser has
 * been created as having a fixed height, this will look at the height of a
 * sample row and set the height of the container as if all possible rows
 * were of that height.  If this browser has been instantiated as being a
 * "stretch to fit" browser, this will ask the HTMLTableElement that holds
 * the browser's data what its current client height is, and set the container
 * height to that value.
 */
vpx.browser.Browser.prototype._setGuiHeight = function() {
   this.gui.heightSpec.complete = true;

   if (areStrictlyEqual(this.gui.heightSpec.policy,
      vpx.browser.Browser.HEIGHT_POLICY.FIXED))
   {
      // Header row
      var height = this.gui.header.offsetHeight;

      // Nodes
      if (this.nodes.length > 0) {
         var rowHeight = this.nodes[0].gui.row.offsetHeight;
         height += (rowHeight * this.pageSize);
      }

      // Padding
      height += 1;
      this.gui.container.style.height = height + "px";
   }
   else if (areStrictlyEqual(this.gui.heightSpec.policy,
      vpx.browser.Browser.HEIGHT_POLICY.FIT))
   {
      if (this.nodes.length > 0) {
	 vpx.xua.setStyle(this.gui.container, "display", "");
      } else {
	 vpx.xua.setStyle(this.gui.container, "display", "none");
      }

      this.gui.container.style.height = this.gui.table.clientHeight + "px";
   }

   window.setTimeout(this._setGuiOverflow.bind(this), 50);
};


/*
 * Set the container CSS overflow property based on the current height and
 * width of the table. Firefox always displays a horizontal scrollbar when the
 * container's overflow property is set to auto, whether one is needed or not.
 */

vpx.browser.Browser.prototype._setGuiOverflow = function () {
   var minWidth = vpx.xua.getMinWidth(this.gui.table);
   var minHeight = vpx.xua.getMinHeight(this.gui.table);

   if (minWidth > vpx.xua.getWidth(this.gui.container) ||
      minHeight > vpx.xua.getHeight(this.gui.container)) {
      vpx.xua.setStyle(this.gui.container, "overflow", "auto");
   } else {
      vpx.xua.setStyle(this.gui.container, "overflow", "hidden");
   }

   window.setTimeout(this._resizeGuiHeight.bind(this), 50);
};


/*
 * After setGuiHeight sets the height of the container, resizeGuiHeight
 * calculates if the browser needs to be made taller to include the height
 * of a horizontal scrollbar. The height will only be changed if it is greater
 * than the previous height.
 */

vpx.browser.Browser.prototype._resizeGuiHeight = function() {
   if (areStrictlyEqual(this.gui.heightSpec.policy,
      vpx.browser.Browser.HEIGHT_POLICY.USER))
   {
      return;
   }

   var height = this.gui.container.scrollHeight +
      (this.gui.container.offsetHeight - this.gui.container.clientHeight);
   if (height > this.gui.container.clientHeight) {
      this.gui.container.style.height = height + "px";
   }
};

/*
 * Stores the specified column in a static collection for easy retrieval.
 *
 * @param column vpx.browser.Column
 * @throws Error
 *    Thrown if a column already exists with the given column's name
 */
vpx.browser.Browser.prototype._storeColRef = function(column) {
    if (column == null || column.name == null) {
        return;
    }
    if (this.colRefs[column.name]) {
       throw new Error("Browser#_storeColRef(): column " + column.name + " already exists");
    }
    this.colRefs[column.name] = column;
};

/**
 * @param node vpx.browser.Node
 */
vpx.browser.Browser.prototype._storeNodeRef = function(node) {
   if (node == null || node.id == null) {
      return;
   }
   this.nodeRefs[node.id] = node;
};

/**
 * @param node vpx.browser.Node
 */
vpx.browser.Browser.prototype._destroyNodeRef = function(node) {
   if (node == null || node.id == null) {
      return;
   }
   delete this.nodeRefs[node.id];
};

/*
 *
 * @param dir String
 * @return const int
 */
vpx.browser.Browser._sortTypeFromDir = function(dir) {
    if (dir == vpx.browser.SORT.ASC) return vpx.browser.SORT.ASC;
    if (dir == vpx.browser.SORT.DESC) return vpx.browser.SORT.DESC;
    return vpx.browser.SORT.NONE;
};

/*
 * @param reject RegExp
 * @param skip String
 */
vpx.browser.Browser.prototype._filterFunc = function(reject, skip) {
   return function(n) {
      if (n.nodeName.search(reject) >= 0) return NodeFilter.FILTER_REJECT;
      if (n.nodeName == skip) return NodeFilter.FILTER_SKIP;
      return NodeFilter.FILTER_ACCEPT;
   };
};
