/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Constructs a new Column with the specified properties.
 *
 * @param browser vpx.browser.Browser
 *    The container of this column
 * @param name String
 *    The name of this column, unique among all columns in this browser
 * @param displayName String
 *    The human-readable name of this column
 * @param canSort boolean
 *    true if this column is sortable; false otherwise
 * @param isWidest boolean
 *    true if this column should occupy all remaining space in the table;
 *    false to let the table automatically flow this column
 * @param align String
 *    "left", "right", or "center" (optional)
 * @throws Error
 *    Thrown if a column already exists with the specified column's name
 */
vpx.browser.Column = function(browser, name, displayName, canSort, isWidest, align) {
   this.name = name;
   this.displayName = displayName;
   this.sort = canSort;
   this.widest = isWidest;
   this.align = align;

   browser._storeColRef(this);
};

/**
 * Gets the name of this column.
 *
 * @return String
 *    The name of this column
 */
vpx.browser.Column.prototype.getName = function() {
   return this.name;
};

/**
 * Gets the human-readable display name of this column.
 *
 * @return String
 *    The display name of this column
 */
vpx.browser.Column.prototype.getDisplayName = function() {
   return this.displayName;
};

/**
 * Gets the value of whether or not this column allows sorting.
 *
 * @return boolean
 *    true if this column is sortable; false otherwise
 */
vpx.browser.Column.prototype.canSort = function() {
   return this.sort;
};

/**
 * Gets the value of whether or not this column is the widest in its group.
 *
 * @return boolean
 *    true if this column occupies all remaining space in its table;
 *    false if it lets the table automatically flow the column
 */
vpx.browser.Column.prototype.isWidest = function() {
   return this.widest;
};

/**
 * Tells whether or not the specified column is logically equal to this column.
 *
 * @param column vpx.browser.Column
 *    The column against which to compare
 * @return boolean
 *    true if the columns are logically equal; false otherwise
 */
vpx.browser.Column.prototype.equals = function(column) {
   return (this.name == column.name);
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Column.prototype.toString = function() {
   return "[Object Column (" + this.name + ")]";
};

/**
 * Constructs a new Column from the specified xml node, having all of the
 * properties specified by the node.
 *
 * @param browser vpx.browser.Browser
 * @param node Element
 * @return vpx.browser.Column
 * @throws Error Thrown if the node is not an Element node
 * @throws Error Thrown if a column already exists with the specified column's name
 */
vpx.browser.Column.fromXml = function(browser, node) {
   try {
      var name = node.getAttribute("name");
      var displayName = node.getAttribute("display");
      var canSort = node.getAttribute("canSort").toBool();
      var widestStr = node.getAttribute("widest");
      var isWidest;
      if (widestStr == null || widestStr == "") {
         isWidest = false;
      } else {
         isWidest = widestStr.toBool();
      }
      var align = node.getAttribute("align");
      vpx.log.trace("Column#fromXml(): name=" + name + " displayName=" +
                    displayName + " canSort=" + canSort + " align=" + align);
      return (new vpx.browser.Column(browser, name, displayName, canSort, isWidest, align));
   } catch (ex) {
      throw new Error("Column#fromXml(): Invalid node: " + node + " :: " + ex);
   }
};
