/* Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential */

/**
 * Constructs a new Location within the given browser and with the given
 * properties.
 *
 * @param browser vpx.browser.Browser
 *    The container of this location
 * @param id String
 *    The id of this location, unique among all locations and nodes in this
 *    browser. Typically, this id is created by the server and passed via xml
 *    to this constructor
 * @param iconClass String
 *    The CSS class for the iconic representation of this location
 * @param display String
 *    The display name of this location
 */
vpx.browser.Location = function(browser, id, iconClass, display) {
   this.instantiated = true;

   // super();
   vpx.ServerObj.call(this, null);

   this.id = id;
   this.browser = browser;
   this.isSelectable = true;
   this.selected = false;
   this.attributes = {};
   this.iconClass = iconClass;
   this.display = display;
};

// Location extends vpx.browser.Node
vpx.browser.Location.prototype = new vpx.browser.Node(null);
vpx.browser.Location.prototype.constructor = vpx.browser.Location;

/**
 * Gets the css class name used to render this location's icon.
 *
 * @return String
 *    The CSS class for the iconic representation of this location
 */
vpx.browser.Location.prototype.getIconClass = function() {
   return this.iconClass;
};

/**
 * (non-doc)
 *
 * @see vpx.browser.Node#setSelected(boolean)
 */
vpx.browser.Location.prototype.setSelected = function(b) {
   this.selected = b;
};

/**
 * (non-doc)
 *
 * @see vpx.browser.Node#scrollIntoView()
 */
vpx.browser.Location.prototype.scrollIntoView = function() {
   // Do nothing
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Location.prototype.toString = function() {
   return "[Object Location (" + this.id + ")]";
};

/**
 * Tests the logical equality of this object and another arbitrary object.
 *
 * @param o Object
 *    The object to compare
 * @return boolean
 *    true if the two objects are logically equal; false otherwise
 */
vpx.browser.Location.prototype.equals = function(o) {
   return (!isNull(o) && this.id == o.id && this.display == o.display);
};

/**
 * Creates a new Location from the given xml <location> element.
 *
 * @param browser Browser
 *    The browser in which to create the location
 * @param element Element
 *    The xml <location> element
 * @return vpx.browser.Location
 *    A new location with all of the properties and cells specified in the xml
 */
vpx.browser.Location.fromXml = function(browser, element) {
   var id = element.getAttribute("id");
   var iconClass = element.getAttribute("iconClass");
   var displayNode = element.getElementsByTagName("display")[0];
   var display = vpx.xua.getInnerContent(displayNode);
   var l = new vpx.browser.Location(browser, id, iconClass, display);

   var attrNodes = element.getElementsByTagName("attr");
   for (var i = 0; i < attrNodes.length; i++) {
      var name = attrNodes[i].getAttribute("name");
      var value = attrNodes[i].getAttribute("value");
      l.setAttribute(name, value);
   }

   return l;
};
