/* Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential */

/**
 * public class Navigator
 * implements vpx.gui.event.ActionListener,
 *            vpx.gui.event.ItemListener
 *
 * TODO Document.
 *
 * @version 1.0 (Jun ?, 2005)
 * @version 1.1 (Nov 2, 2005) - Change to use vpx.gui package
 */

/**
 * Constructs a new Navigator, initialized to not be associated with any
 * Browser provider and thus to have all of its controls disabled/empty.
 *
 * @param container HTMLElement
 *    The DOM element in which this navigator's GUI is to be created
 * @param showControls boolean
 *    Optional parameter controlling whether the menu and up controls
 *    should be visible.  Defaults to true.
 */
vpx.browser.Navigator = function(container, showControls) {
   this.locations = [];

   this.boundFunctions = {
      handleDataChange : vpx.browser.Navigator.prototype._handleDataChange.bind(this),
      handleSelection  : vpx.browser.Navigator.prototype._handleSelection.bind(this)
   };

   if (isNull(showControls)) {
      showControls = true;
   }
   this._initGui(container, showControls);
};

// Shorthand for brevity's sake
var _c = vpx.browser.Navigator;       // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.provider = null;                   // protected vpx.browser.Browser
_i.locations = null;                  // protected vpx.browser.Location[]
_i.index = -1;                        // protected int
_i.selected = false;                  // protected boolean
_i.menu = null;                       // protected vpx.gui.Menu
_i.mainButton = null;                 // protected vpx.browser.NodeButton
_i.upButton = null;                   // protected vpx.gui.Button
_i.menuHandler = null;                // private vpx.browser.Navigator.MenuHandler

/**
 * Associates this object with the given Browser data provider.  This is not
 * done upon instantiation to allow for a Navigator to be created before the
 * corresponding Browser object.  Thus, the two are de-coupled and can live in
 * separate contexts but still communicate.
 *
 * @param browser vpx.browser.Browser
 *    The object that this navigator will reflect and control
 * @throws Error
 *    Thrown if this object is already associated with a provider
 */
_i.setProvider = function(browser) {
   vpx.log.trace("Navigator#setProvider(): setting provider to " + browser);
   if (this.provider != null) {
      throw new Error("Navigator#setProvider(): cannot have more than 1 provider");
   }
   this.provider = browser;

   this._handleDataChange(browser, null);
   browser.registerListener(vpx.browser.EVENT.DATA, this.boundFunctions.handleDataChange);

   this._handleSelection(browser);
   browser.registerListener(vpx.browser.EVENT.SELECTION, this.boundFunctions.handleSelection);
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.browser.Navigator]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/*
 * Handles a selection change in the hierarchy menu
 *
 * @param event Event
 *    The event fired from the user agent
 */
_i._handleNavigate = function(event) {
   vpx.log.trace("Navigator#_handleNavigate(): Registered location change");
   if (isDefined(this.provider)) {
    var location = this.provider.hierarchy[this.gui.select.selectedIndex];
    this.provider.setLocation(location);
   }
};

/**
 * Handles a mouse click on the "up" clickable area.  This sets the location
 * of this navigator's provider to  be one up the location hierarchy.
 *
 * @param e vpx.gui.event.ActionEvent
 *    The action event
 */
_i.actionPerformed = function(e) {
   vpx.log.trace("Navigator#actionPerformed(): Registered mouse click");
   if (isDefined(this.provider) && this.index > 0) {
      this.provider.setLocation(this.locations[this.index - 1]);
   }
};

/**
 * Handles an item state change for this navigator's main button (the
 * <code>NodeButton</code> with which the user can select the current
 * location).  This handler gets called whenever the main button is either
 * selected or de-selected.
 *
 * @param e vpx.gui.event.ItemEvent
 *    The item event
 */
_i.itemStateChanged = function(e) {
   if (this.locations.length == 0 || this.index == -1) {
      return;
   }

   var l = this.locations[this.index];

   var c = vpx.gui.event.ItemEvent;
   switch (e.getStateChange()) {
   case c.SELECTED:
      this.selected = true;
      this.provider.select([l], false);
      break;
   case c.DESELECTED:
      this.selected = false;
      break;
   default:
      throw new Error("Navigator#itemStateChanged(): Unknown state change");
      break;
   }
};


/*
 * Handles a DATA event fired from this object's provider.
 *
 * @param browser vpx.browser.Browser
 *    The provider object whose hierarchy location has changed
 * @param xmlSpec vpx.net.XmlSpec
 *    The xml spec used to request the data
 */
_i._handleDataChange = function(browser, xmlSpec) {
   var i;
   var updateNeeded = false;
   for (i = 0; i < browser.hierarchy.length; i++) {
      if (i >= this.locations.length || !browser.hierarchy[i].equals(this.locations[i])) {
         updateNeeded = true;
         break;
      }
   }
   if (i < this.locations.length) {
      updateNeeded = true;
   }
   if (!updateNeeded) {
      return;
   }

   var menuListener = this._getMenuHandler();

   // Make sure not to leak memory
   var items = this.menu.getMenuComponents();
   for (i = 0; i < items.length; i++) {
      items[i].removeActionListener(menuListener);
   }

   // Clear the locations list; we'll re-add them from scratch
   this.locations.splice(0, this.locations.length);
   this.menu.removeAll();

   var tle = vpx.getTle();
   for (i = 0; i < browser.hierarchy.length; i++) {
      var l = browser.hierarchy[i];
      this.locations.push(l);
      var item = new vpx.gui.MenuItem(tle, l.display, null);
      item.setActionCommand(l.getId());
      item.addActionListener(menuListener);
      this.menu.add(item);
   }

   this.index = i - 1;
   if (i > 0) {
      var l = browser.hierarchy[i - 1];
      this.gui.iconDiv.className = "icon " + l.iconClass;
      this.mainButton.setText(l.display);
      this.mainButton.setSelected(this.selected);

      this._updateSelectedState(browser);
   }

   var canNavigate = (this.index > 0);
   this.menu.setEnabled(canNavigate);
   this.upButton.setEnabled(canNavigate);
};

_i._handleSelection = function(browser) {
   this._updateSelectedState(browser);
};

_i._updateSelectedState = function(browser) {
   var s = browser.getSelected();
   if (s.length > 0 && !(s[0] instanceof vpx.browser.Location)) {
      // browser node was selected - mainButton must NOT be selected
      this.mainButton.setSelected(false);
   } else {
      // browser node was un-selected - mainButton must BE selected
      this.mainButton.setSelected(true);
   }
};

/*
 * Initializes this object's GUI and places it in the specified GUI container.
 *
 * @param container HTMLElement
 *    The DOM element in which this object's GUI is to be created
 * @param showControls boolean
 *    true to show the menu and up controls, false to hide them.
 */
_i._initGui = function(container, showControls) {
   this.gui = {
      container:container,
      table : null,
      iconDiv : null
   };

   var table, tbody, tr, td, menutd, uptd, img;
   var view = vpx.xua.getView(container) || window;
   var doc = view.document;

   // table
   table = doc.createElement("table");
   table.className = "listView";
   table.width = "100%";
   table.border = "0";
   table.cellPadding = "0";
   table.cellSpacing = "0";
   container.appendChild(table);
   this.gui.table = table;

   // tbody & row
   tbody = doc.createElement("tbody");
   table.appendChild(tbody);
   tr = doc.createElement("tr");
   tr.className = "navigator";
   tbody.appendChild(tr);

   // icon cell
   td = doc.createElement("td");
   td.width = "1";
   tr.appendChild(td);
   td.className = "lcol listKeyIcon";
   div = doc.createElement("div");
   div.className = "icon";
   td.appendChild(div);
   this.gui.iconDiv = div;

   // location cell
   td = document.createElement("td");
   td.style.width = "100%";
   td.className = "listKey";
   var n = new vpx.browser.NodeButton(view, null, null, false);
   n.addItemListener(this);
   tr.appendChild(td);
   td.appendChild(n.getPeer());
   this.mainButton = n;

   // menuCtrl cell
   menutd = document.createElement("td");
   menutd.className = "menuControl";
   menutd.style.display = "none";
   var m = new vpx.gui.MenuButton(view, null, new vpx.gui.ImageIcon(view, "menuPullDownArrow icon", "menuPullDownArrow"));
   m.setDisabledIcon(new vpx.gui.ImageIcon(view, "menuPullDownArrowDisabled icon", "menuPullDownArrowDisabled"));
   m.getPeer().style.display = "";  //XXX: See PR 115192
   var preferredSize;
   if (vpx.xua.ie) {
      //?? TODO: We should be calculating these values, not hardcoding them
      m.setMenuLocation(-173, 18);
      preferredSize = new vpx.gui.Dimension(196, -1);
   } else {
      m.setMenuLocation(-180, 20);
      preferredSize = new vpx.gui.Dimension(200, -1);
   }
   m.getPopupMenu().setPreferredSize(preferredSize);
   m.setEnabled(false);
   tr.appendChild(menutd);
   menutd.appendChild(m.getPeer());
   this.menu = m;

   // up cell
   uptd = document.createElement("td");
   uptd.className = "upControl";
   uptd.style.display = "none";
   var b = new vpx.gui.Button(view, null, null);
   b.setIcon(new vpx.gui.ImageIcon(view, "gotoParentDirectory-16x16"));
   b.setDisabledIcon(new vpx.gui.ImageIcon(view, "gotoParentDirectoryDisabled-16x16"));
   b.addActionListener(this);
   b.getPeer().style.display = ""; //XXX: See PR 115192
   b.setEnabled(false);
   tr.appendChild(uptd);
   uptd.appendChild(b.getPeer());
   this.upButton = b;

   var display = "";
   if (!showControls) {
      display = "none";
   }
   menutd.style.display = display;
   uptd.style.display = display;
};

/**
 * Returns a menu handler for use by this Navigator.  The handler is created
 * lazily.
 *
 * @return vpx.browser.Navigator.MenuHandler
 *    The menu handler for this navigator
 */
_i._getMenuHandler = function() {
   if (this.menuHandler == null) {
      this.menuHandler = new vpx.browser.Navigator.MenuHandler(this);
   }
   return this.menuHandler;
};


/**
 * class MenuHandler
 * extends Object
 * implements vpx.gui.event.ActionListener
 *
 * Listener that automatically gets registered with this navigator. This
 * listener handles events that are fired from this navigator's pulldown menu.
 *
 * @version 1.0 (Nov 16, 2005)
 */

/**
 * Constructs a new MenuHandler.
 */
_c.MenuHandler = function(nav) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   this.nav = nav;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.event.ActionListener#actionPerformed(vpx.gui.event.ActionEvent)
 */
_c.MenuHandler.prototype.actionPerformed = function(e) {
   if (this.nav.provider != null) {
      var entityId = e.getActionCommand();
      for (var i = 0; i < this.nav.locations.length; i++) {
         var l = this.nav.locations[i];
         if (l.getId() == entityId) {
            this.nav.provider.setLocation(l);
         }
      }
   }
};

