/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Constructs a new Node within the given browser and with the given
 * properties.
 *
 * @param browser vpx.browser.Browser
 *    The container of this node
 * @param id String
 *    The id of this node, unique among all nodes in this browser. Typically,
 *    this id is created by the server and passed via xml to this constructor
 * @param iconClass String
 *    The CSS class for the iconic representation of this node
 */
vpx.browser.Node = function(browser, id, iconClass) {
   if (browser != null) {
      this.instantiated = true;

      // super();
      vpx.ServerObj.call(this, null);

      this.id = id;
      this.browser = browser;
      this.isSelectable = true;
      this.selected = false;
      this.attributes = {};

      this.boundFunctions = {
         handleClick : vpx.browser.Node.prototype._handleClick.bind(this),
         handleDblClick : vpx.browser.Node.prototype._handleDblClick.bind(this)
      };

      browser._storeNodeRef(this);
      this._initGui(browser, iconClass);
   }
};

// Node extends ServerObj
vpx.browser.Node.prototype = new vpx.ServerObj(null);
vpx.browser.Node.prototype.constructor = vpx.browser.Node;

/**
 * Gets the css class name used to render this node's icon.
 *
 * @return String
 *    The CSS class for the iconic representation of this node
 */
vpx.browser.Node.prototype.getIconClass = function() {
   return this.gui.iconClass;
};

/**
 * Sets the css class name used to render this node's icon. This will change
 * the look & feel of this node in real-time.
 *
 * @param className String
 *    The CSS class for the iconic representation of this node
 */
vpx.browser.Node.prototype.setIconClass = function(className) {
   var td = this.gui.row.firstChild;
   td.firstChild.className = className + " icon";
};

/**
 *
 * @param column vpx.browser.Column
 * @return int
 */
vpx.browser.Node.prototype.getCellCount = function(column) {
   var colIndex = this.browser.indexOfCol(column);
   return this.gui.cells[colIndex].length;
};

/*
 * Retrieves a cell identified by the given column and index.
 *
 * @param column vpx.browser.Column
 *
 * @param index int
 *    The index of the cell in its group. If only one cell exists for the
 *    given column, this "cell group" index should be 0
 * @return HTMLTableCellElement
 *    the cell element (td)
 */
vpx.browser.Node.prototype.getCell = function(column, index) {
   return this._getCell(this.browser.indexOfCol(column), index);
};

/*
 * Sets the html content of a cell (identified by the given column).
 *
 * @param column vpx.browser.Column
 *
 * @param index int
 *    The index of the cell in its group. If only one cell exists for the
 *    given column, this "cell group" index should be 0
 * @param html String
 *    The new html of the cell
 */
vpx.browser.Node.prototype.setCellHtml = function(column, index, html) {
   this._setCellHtml(this.browser.indexOfCol(column), index, html, null);
};

/**
 * Sets whether or not this node is selectable by users.
 *
 * @param b boolean
 *    true to make the node selectable; false otherwise
 */
vpx.browser.Node.prototype.setSelectable = function(b) {
   this.isSelectable = b;
};

/**
 * @param bool boolean
 */
vpx.browser.Node.prototype.setSelected = function(bool) {
   if (!this.isSelectable) {
      return;
   }
   this.selected = bool;
   var cell = this._getMainCell();
   var href = cell.firstChild;
   var outerDiv = href.firstChild;
   var innerDiv = outerDiv.firstChild;

   var tipSelectedStr;
   if (this.getAttribute("tipSelected") != null) {
      tipSelectedStr = this.getAttribute("tipSelected");
   } else {
      tipSelectedStr = "";
   }

   var tipUnselectedStr;
   if (this.getAttribute("tipUnselected") != null) {
      tipUnselectedStr = this.getAttribute("tipUnselected");
   } else {
      tipUnselectedStr = "";
   }

   if (bool) {
      outerDiv.className = "listKeyHilite listKeyHilited";
      if (this instanceof vpx.browser.BranchNode) {
         innerDiv.className = "listKey listKeySlctd listKeyParent";
         href.title = tipSelectedStr;
      } else {
         innerDiv.className = "listKey listKeySlctd";
         href.title = "";
      }
   } else {
      outerDiv.className = "listKeyHilite";
      innerDiv.className = "listKey";
      if (this instanceof vpx.browser.BranchNode) {
         href.title = tipUnselectedStr;
      } else {
         href.title = tipSelectedStr;
      }
   }
};

/**
 * Scrolls this node into view within its containing <code>Browser</code>.
 * If the node is already visible, nothing happens.
 */
vpx.browser.Node.prototype.scrollIntoView = function() {
   if (this.gui.row.scrollIntoView) {
      this.gui.row.scrollIntoView(true);
   }
};

/**
 * Tells whether or not this node has the given attribute set in its list
 * of attributes.
 *
 * @param key String
 *    The attribute key
 * @return boolean
 *    true if the attribute exists; false otherwise
 */
vpx.browser.Node.prototype.hasAttribute = function(key) {
   return isDefined(this.attributes[key]);
};

/**
 * Gets an attribute of the node. An attribute is a key/value pair that
 * augments the shared Class information for a particular Node.
 *
 * @param key String
 *    The attribute key
 * @return String
 *    The attribute value, or null if no such attribute exists
 */
vpx.browser.Node.prototype.getAttribute = function(key) {
   if (isNull(this.attributes[key])) {
      return null;
   }
   return this.attributes[key];
};

/**
 * Sets or overrides an attribute of the node.
 *
 * @param key String
 *    The attribute key
 * @param value String
 *    The attribute value
 */
vpx.browser.Node.prototype.setAttribute = function(key, value) {
   this.attributes[key] = value;
};

/**
 * Removes an attribute from the node.  If the attribute did not
 * exist to begin with, nothing happens.
 *
 * @param key String
 *    The attribute key
 */
vpx.browser.Node.prototype.removeAttribute = function(key) {
   delete this.attributes[key];
};

/**
 * Removes all attributes associated with this node.
 */
vpx.browser.Node.prototype.clearAttributes = function() {
   delete this.attributes;
   this.attributes = {};
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Node.prototype.toString = function() {
   return "[Object Node (" + this.id + ")]";
};

/*
 * (non-doc)
 *
 * @see Object#equals(Object)
 */
vpx.browser.Node.prototype.equals = function(o) {
   if (!(o instanceof vpx.browser.Node)) {
      return false;
   }
   return (this.id == o.id);
};

/**
 *
 */
vpx.browser.Node.prototype.destroy = function() {
   this.browser._destroyNodeRef(this);
   //?? XXX Anything more to do here?
};

/**
 * Creates a new Node from the given xml <node> element.
 *
 * @param browser vpx.browser.Browser
 *    The browser in which to create the node
 * @param element Element
 *    The xml <node> element
 * @return vpx.browser.Node
 *    A new node with all of the properties and cells specified in the xml
 * @throws Error
 *    Thrown if the element is not an Element node
 */
vpx.browser.Node.fromXml = function(browser, element) {
   var i;
   //?? XXX IE doesn't support node.hasAttributes()
   //if (!element.hasAttributes()) {
   //  throw new Error("Column#fromXml(): Invalid node: " + element);
   //}

   var constructor = vpx.browser.Node;
   //?? XXX IE doesn't support element.hasAttribute()
   if (element.getAttribute("children") == "true") {
      constructor = vpx.browser.BranchNode;
   }

   var iconClass = element.getAttribute("iconClass");
   var id = element.getAttribute("id");
   var bn = new constructor(browser, id, iconClass);

   var selectableStr = element.getAttribute("selectable");
   if (selectableStr != null) {
      bn.setSelectable(selectableStr.toBool());
   }

   var attrNodes = element.getElementsByTagName("attr");
   for (i = 0; i < attrNodes.length; i++) {
      var name = attrNodes[i].getAttribute("name");
      var value = attrNodes[i].getAttribute("value");
      bn.setAttribute(name, value);
   }

   var columns = browser.getColumns();
   var colNodes = element.getElementsByTagName("col");
   for (i = 0; i < colNodes.length; i++) {
      var cellNodes = colNodes[i].getElementsByTagName("cell");
      var colSpan = browser.gui.columns[i].colSpan;
      var diff = colSpan - cellNodes.length;
      for (var j = 0; j < cellNodes.length; j++) {
         var cn = cellNodes[j];
         bn._addCell(i, cn.getAttribute("class"), vpx.xua.getInnerContent(cn), diff + 1);
         diff = 0;
      }
   }



   return bn;
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 *
 * @param browser vpx.browser.Browser
 *    The container of this node
 * @param iconClass String
 *    The CSS class for the iconic representation of this node
 */
vpx.browser.Node.prototype._initGui = function(browser, iconClass) {
   this.gui = {iconClass:iconClass};

   this.gui.cells = [];
   var cols = browser.getColumns();
   for (var i = 0; i < cols.length; i++) {
      this.gui.cells[i] = [];
   }

   var tr = document.createElement("tr");
   this.gui.row = tr;

   var td = document.createElement("td");
   tr.appendChild(td);
   td.className = "lcol listKeyIcon";

   var div = document.createElement("div");
   div.className = iconClass + " icon";
   td.appendChild(div);
};

/*
 *
 * @param colIndex int
 * @param iconClass String
 * @param content String
 * @param colSpan int
 */
vpx.browser.Node.prototype._addCell = function(colIndex, iconClass, content, colSpan) {
   // Offset start index by 1 to account for icon cell
   var index = 1;
   for (var i = 0; i <= colIndex; i++) {
      index += this.gui.cells[i].length;
   }

   // Equivalent code to 'var td = this.gui.row.insertCell(index);'
   var td = document.createElement("td");
   if (index >= this.gui.row.childNodes.length) {
      this.gui.row.appendChild(td);
   } else {
      this.gui.row.insertBefore(td, this.gui.row.childNodes[index]);
   }

   if (colSpan > 1) {
      td.colSpan = colSpan;
   }

   var cgIndex = this.gui.cells[colIndex].length;
   this.gui.cells[colIndex].push(td);
   td.style.whiteSpace = "nowrap";
   td.noWrap = true;

   this._setCellHtml(colIndex, cgIndex, content, iconClass);
};

/*
 * Sets the html content of a cell (identified by the given column).
 *
 * @param colIndex int
 *
 * @param cgIndex int
 *    The index of the cell in its group. If only one cell exists for the
 *    given column, this "cell group" index should be 0
 * @param html String
 *    The new html of the cell
 * @param className String
 *    The css class name of the cell's desired style
 */
vpx.browser.Node.prototype._setCellHtml = function(colIndex, cgIndex, html, className) {
   var div;
   var td = this._getCell(colIndex, cgIndex);

   // Clear the cell of any existing content
   td.innerHTML = "";

   if (this.browser.columns[colIndex].isWidest() && cgIndex == 0) {
      td.style.width = "100%";
   }

   if (colIndex == 0) {
      if (isDefined(this.gui.href)) {
         // Ignore events to not leak memory
         vpx.xua.event.ignore(this.gui.href, "click", this.boundFunctions.handleClick);
         vpx.xua.event.ignore(this.gui.href, "dblclick", this.boundFunctions.handleDblClick);
         delete this.gui.href;
      }

      // Wrap the html in "main cell" pre-defined html
      var href = document.createElement("a");
      this.gui.href = href;

      // tvolkert 09/19/2005: Changed (from href="#") for bug #76869
      href.href = html;


      td.appendChild(href);
      vpx.xua.event.listen(href, "click", this.boundFunctions.handleClick);
      vpx.xua.event.listen(href, "dblclick", this.boundFunctions.handleDblClick);

      div = document.createElement("div");
      href.appendChild(div);

      var innerDiv = document.createElement("div");
      div.appendChild(innerDiv);

      // This (as opposed to 'innerDiv.innerHTML = html') avoids Mozilla bug
      var tmp = document.createElement("span");
      tmp.innerHTML = html;
      if (tmp.childNodes.length > 0) {
         innerDiv.appendChild(tmp.firstChild);
      }

      div.className = "listKeyHilite";
      innerDiv.className = "listKey";
      td.className = "listKey";
      if (isDefined(className)) {
         td.className += " " + className;
      }

      var tipSelectedStr;
      if (this.getAttribute("tipSelected") != null) {
         tipSelectedStr = this.getAttribute("tipSelected");
      } else {
         tipSelectedStr = "";
      }

      var tipUnselectedStr;
      if (this.getAttribute("tipUnselected") != null) {
         tipUnselectedStr = this.getAttribute("tipUnselected");
      } else {
         tipUnselectedStr = "";
      }


      if (this.selected) {
         div.className += " listKeyHilited";
         innerDiv.className += " listKeySlctd";
         if (this instanceof vpx.browser.BranchNode) {
            innerDiv.className += " listKeyParent";
            href.title = tipSelectedStr;
         } else {
            href.title = "";
         }
      } else {
         if (this instanceof vpx.browser.BranchNode) {
            href.title = tipUnselectedStr;
         } else {
            href.title = tipSelectedStr;
         }
      }
   } else {
      if (isDefined(className)) {
         td.className = className;
      }
      div = document.createElement("div");
      td.appendChild(div);
      div.innerHTML = html;
   }
};

/*
 * Handles a mouse click on this node's GUI row.  It marshalls the event up
 * to this node's containing browser, since the browser needs to know
 * whether to add this node to the list of selected nodes, unselect the
 * currently selected node, etc.
 *
 * @param event Event
 *    The event fired from the user agent
 */
vpx.browser.Node.prototype._handleClick = function(event) {
   try {
      this.browser.handleSelectNode(event, this);
   } finally {
      vpx.xua.event.getSource(event).blur();
      vpx.xua.event.cancel(event);
      return false;
   }
};

/*
 * Handles a mouse double-click on this node's GUI row.  It marshalls the event
 * up to this node's containing browser, since the browser needs to know
 * whether to add this node to the list of selected nodes, unselect the
 * currently selected node, etc.  A double click is treated as two distinct
 * clicks.  This is handled specially because IE does not fire two clicks; it
 * fires one click event, followed by a double click event.
 *
 * @param event Event
 *    The event fired from the user agent
 */
vpx.browser.Node.prototype._handleDblClick = function(event) {
   if (vpx.xua.ie) {
      /**
       * IE behaves differently than other browsers upon double click. It
       * sends one "click" event, and one "double click" event, whereas
       * most user agents send two "click" events and one "double click"
       * event. Thus, upon receiving a "double click" event in IE, you
       * know that to mimic other user agents, you need to send a synthetic
       * "click" event.
       */
      return this._handleClick(event);
   }
};

/*
 * @param colIndex int
 * @param cgIndex int
 * @return HTMLTableCellElement
 */
vpx.browser.Node.prototype._getCell = function(colIndex, cgIndex) {
   return this.gui.cells[colIndex][cgIndex];
};

/*
 *
 */
vpx.browser.Node.prototype._getMainCell = function() {
   return this._getCell(0, 0);
};
