/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Constructor for a new Paged object.  This class is abstract, however, so
 * users should not invoke this constructor directly.  Any attempt to do so
 * will result in an error being thrown.
 */
vpx.browser.Paged = function() {
   // Abstract class
   if (arguments.length != 1 || arguments[0] != null) {
      throw new Error("Paged is abstract: cannot be instantiated");
   }

   if (this.instantiated) {
      // super();
      vpx.EventSource.call(this, null);

      this.totalItems = 0;
      this.page = 0;
      this.pageSize = vpx.browser.Paged.DEFAULT_PAGE_SIZE;
      this.listeners[vpx.browser.EVENT.DATA] = [];
      this.listeners[vpx.browser.EVENT.PAGE] = [];
   }
};

// Paged extends vpx.EventSource
vpx.browser.Paged.prototype = new vpx.EventSource(null);
vpx.browser.Paged.prototype.constructor = vpx.browser.Paged;

vpx.browser.Paged.DEFAULT_PAGE_SIZE = Infinity;

/**
 * Tells whether or not this object has data on a previous page.
 *
 * @return boolean
 *    true if this has a previous page; false otherwise
 */
vpx.browser.Paged.prototype.hasPrevPage = function() {
   return (this.page > 0);
};

/**
 * Tells whether or not this object has data on a next page, i.e. whether this
 * page's data overflows to the next.
 *
 * @return boolean
 *    true if this has a next page; false otherwise
 */
vpx.browser.Paged.prototype.hasNextPage = function() {
   return ((this.page + 1) * this.pageSize < this.totalItems);
};

/**
 * Gets the number of data items that will fit on one page.
 *
 * @return int
 *    The page size (note: Infinity is allowed)
 */
vpx.browser.Paged.prototype.getPageSize = function() {
   return this.pageSize;
};

/**
 * Gets the 0-based page that is currently being displayed.
 *
 * @return int
 *    The current page
 */
vpx.browser.Paged.prototype.getPage = function() {
   return this.page;
};

/**
 * Increments the current page.
 */
vpx.browser.Paged.prototype.nextPage = function() {
   if (!this.hasNextPage()) {
      throw new Error("Paged#nextPage(): no next page");
   }
   this._fireEvent(vpx.browser.EVENT.PAGE, vpx.browser.PAGE.NEXT);
};

/**
 * Decrements the current page.
 */
vpx.browser.Paged.prototype.prevPage = function() {
   if (!this.hasPrevPage()) {
      throw new Error("Paged#prevPage(): no prev page");
   }
   this._fireEvent(vpx.browser.EVENT.PAGE, vpx.browser.PAGE.PREV);
};

/**
 * Sets the current page to the specified page.  If the specified page is the
 * same as the current page, nothing happens.
 *
 * @param page int
 *    The new page (0-based)
 * @throws Error
 *    Thrown if the specified page is out of bounds
 */
vpx.browser.Paged.prototype.setPage = function(page) {
   if (page == this.page) {
      // Do nothing
      return;
   }
   if (page < 0 || page >= this.getTotalPages()) {
      throw new Error("Page out of bounds: " + page);
   }
   this._fireEvent(vpx.browser.EVENT.PAGE, vpx.browser.PAGE.JUMP, page);
};

/**
 * Gets the 0-based index of the first item on the current page, or null if
 * there are no items in the data set.
 *
 * @return int
 *    The index of the first item on the page
 */
vpx.browser.Paged.prototype.getPageStart = function() {
   if (this.totalItems == 0) {
      return null;
   }
   if (this.pageSize == Infinity) {
      return 0;
   }
   return (this.page * this.pageSize);
};

/**
 * Gets the 0-based index of the last item on the current page, or null if
 * there are no items in the data set.
 *
 * @return int
 *    The index of the last item on the page
 */
vpx.browser.Paged.prototype.getPageEnd = function() {
   if (this.totalItems == 0) {
      return null;
   }
   if (!this.hasNextPage()) {
      return (this.totalItems - 1);
   }
   return ((this.page + 1) * this.pageSize - 1);
};

/**
 * Gets the total number of data items in this object's collection, spanning
 * all pages.
 *
 * @return int
 *    The total number of items in the collection
 */
vpx.browser.Paged.prototype.getTotalItems = function() {
   return this.totalItems;
};

/**
 * Gets the total number of pages needed to display this object's data. If this
 * object contains no data, it is said to require 0 pages to display that data.
 *
 * @return int
 *    The total number of pages
 */
vpx.browser.Paged.prototype.getTotalPages = function() {
   if (this.totalItems == 0) {
      return 0;
   }
   if (this.pageSize == Infinity) {
      return 1;
   }
   return Math.ceil(this.totalItems / this.pageSize);
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Paged.prototype.toString = function() {
   return "[Object Paged]";
};
