/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * Constructs a new Paginator, initialized to not be associated with any Paged
 * provider and thus to have all of its controls disabled.
 *
 * @param container HTMLElement
 *    The DOM element in which this paginator's GUI is to be created
 */
vpx.browser.Paginator = function(container) {
   this.provider = null;
   this.prevEnabled = false;
   this.nextEnabled = false;

   this.boundFunctions = {
      handlePrev : vpx.browser.Paginator.prototype._handlePrev.bind(this),
      handleNext : vpx.browser.Paginator.prototype._handleNext.bind(this),
      handleDataChange : vpx.browser.Paginator.prototype._handleDataChange.bind(this)
   };

   this._initGui(container);
};

/**
 * Associates this object with the given Paged data provider.  This is not done
 * upon instantiation to allow for a Paginator to be created before the
 * corresponding Paged object.  Thus, the two are de-coupled and can live in
 * separate contexts but still communicate.
 *
 * @param paged Paged
 *    The object that this paginator will reflect and control
 * @throws Error
 *    Thrown if this object is already associated with a provider
 */
vpx.browser.Paginator.prototype.setProvider = function(paged) {
   vpx.log.trace("Paginator#setProvider(): setting provider to " + paged);
   if (this.provider != null) {
      throw new Error("Paginator#setProvider(): Cannot have more than 1 provider");
   }
   this.provider = paged;
   this._handleDataChange(paged, null);
   paged.registerListener(vpx.browser.EVENT.DATA, this.boundFunctions.handleDataChange);
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
vpx.browser.Paginator.prototype.toString = function() {
   return "[Object Paginator]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/*
 * Initializes this object's GUI and places it in the specified GUI container.
 *
 * @param container HTMLElement
 *    The DOM element in which this object's GUI is to be created
 */
vpx.browser.Paginator.prototype._initGui = function(container) {
   this.gui = {container:container};
   this.gui.divs = {};
   this.gui.hrefs = {};

   var table, tbody, row, cell, div, href, img;

   // Table
   table = document.createElement("table");
   table.className = "pageCtrl";
   table.width = "100%";
   table.border = "0";
   table.cellPadding = "0";
   table.cellSpacing = "0";
   container.appendChild(table);

   // Tbody & row
   tbody = table.getElementsByTagName("TBODY")[0];
   if (vpx.log.isTraceEnabled()) {
      var debugTxt = "Paginator#_initGui(): tbody existence\n--true\n--";
      debugTxt += (table.childNodes.length - 1 == table.getElementsByTagName("TBODY").length);
      vpx.log.trace(debugTxt);
   }
   if (!tbody) {
      tbody = document.createElement("tbody");
      table.appendChild(tbody);
   }
   row = document.createElement("tr");
   tbody.appendChild(row);

   // Txt cell
   cell = document.createElement("td");
   cell.className = "lcol";
   cell.style.width = "100%";
   row.appendChild(cell);
   div = document.createElement("div");
   this.gui.divs.txt = div;
   cell.appendChild(div);
   div.appendChild(document.createTextNode("None"));

   // Prev cell
   cell = document.createElement("td");
   row.appendChild(cell);
   div = document.createElement("div");
   div.className = "icon";
   cell.appendChild(div);
   href = document.createElement("a");
   this.gui.hrefs.prev = href;
   // tvolkert 04/17/2006: Changed (from href="#") for automated testing
   href.href = "PreviousPage";
   vpx.xua.event.listen(href, "click", this.boundFunctions.handlePrev);
   div.appendChild(href);
   img = document.createElement("img");
   img.border = "0";
   img.src = "imx/spacer.png";
   img.className = "pageLeftDisabled";
   href.appendChild(img);

   // Next cell
   cell = document.createElement("td");
   cell.className = "rcol";
   row.appendChild(cell);
   div = document.createElement("div");
   div.className = "icon";
   cell.appendChild(div);
   href = document.createElement("a");
   this.gui.hrefs.next = href;
   // tvolkert 04/17/2006: Changed (from href="#") for automated testing
   href.href = "NextPage";
   vpx.xua.event.listen(href, "click", this.boundFunctions.handleNext);
   div.appendChild(href);
   img = document.createElement("img");
   img.border = "0";
   img.src = "imx/spacer.png";
   img.className = "pageRightDisabled";
   href.appendChild(img);
};

/*
 * Handles a DATA_STATE event fired from this object's provider.
 *
 * @param paged vpx.browser.Paged
 *    The provider object whose data pagination state has changed
 * @param xmlSpec vpx.net.XmlSpec
 *    The xml spec that the paged object used to get its data
 */
vpx.browser.Paginator.prototype._handleDataChange = function(paged, xmlSpec) {
   vpx.log.trace("Paginator#_handleDataChange(): handling data change");
   this._setPrevEnabled(paged.hasPrevPage());
   this._setNextEnabled(paged.hasNextPage());

   var total = paged.getTotalItems();
   if (total == 0) {
      this._setTxt(zNone);
   } else {
      var txt = zItems + " " + (paged.getPageStart() + 1) + " - ";
      txt += (paged.getPageEnd() + 1) + " / " + total;
      this._setTxt(txt);
   }
};

/*
 * Handles a mouse click on the "prev" clickable area.
 *
 * @param event Event
 *    The event fired from the user agent
 */
vpx.browser.Paginator.prototype._handlePrev = function(event) {
   try {
      vpx.log.trace("Paginator#_handlePrev(): Registered prev click");
      if ((this.prevEnabled) && (isDefined(this.provider))) {
         this.provider.prevPage();
      }
   } finally {
      vpx.xua.event.cancel(event);
      return false;
   }
};

/*
 * Handles a mouse click on the "next" clickable area.
 *
 * @param event Event
 *    The event fired from the user agent
 */
vpx.browser.Paginator.prototype._handleNext = function(event) {
   try {
      vpx.log.trace("Paginator#_handleNext(): Registered next click");
      if ((this.nextEnabled) && (isDefined(this.provider))) {
         this.provider.nextPage();
      }
   } finally {
      vpx.xua.event.cancel(event);
      return false;
   }
};

/*
 * Sets the value of this object's GUI text area.
 *
 * @param txt String
 *    The text this paginator should show to the user
 */
vpx.browser.Paginator.prototype._setTxt = function(txt) {
   this.gui.divs.txt.innerHTML = txt;
};

/*
 * Sets whether this GUI's "prev" area is enabled (clickable) or not.
 *
 * @param bool boolean
 *    true iff the prev area should be enabled
 */
vpx.browser.Paginator.prototype._setPrevEnabled = function(bool) {
   if (this.prevEnabled == bool) {
      return;
   }
   vpx.log.trace("Paginator#_setPrevEnabled(): setting prevEnabled to " + bool);
   this.prevEnabled = bool;
   var href = this.gui.hrefs.prev;
   if (bool) {
      href.style.cursor = "pointer";
      href.childNodes[0].className = "pageLeft";
   } else {
      href.style.cursor = "";
      href.childNodes[0].className = "pageLeftDisabled";
   }
};

/*
 * Sets whether this GUI's "next" area is enabled (clickable) or not.
 *
 * @param bool boolean
 *    true iff the next area should be enabled
 */
vpx.browser.Paginator.prototype._setNextEnabled = function(bool) {
   if (this.nextEnabled == bool) {
      return;
   }
   vpx.log.trace("Paginator#_setNextEnabled(): setting nextEnabled to " + bool);
   this.nextEnabled = bool;
   var href = this.gui.hrefs.next;
   if (bool) {
      href.style.cursor = "pointer";
      href.childNodes[0].className = "pageRight";
   } else {
      href.style.cursor = "";
      href.childNodes[0].className = "pageRightDisabled";
   }
};
