/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Constructs a new SnapshotNode with the specified properties.
 *
 * @param String id
 * @param String snapshotID
 * @param String name 
 * @param String desc
 * @param String date
 * @param boolean isLeaf
 * @param String parentId
 * @param String[] childrenIds
 */
vpx.browser.SnapshotNode = function SnapshotNode(id, snapshotID, name, desc, date, isLeaf, parentId, childrenIds, domNode ) {
   this.id = id;
   this.snapshotID = snapshotID;
   this.name = name;
   this.desc = desc;
   this.date = date;
   this.isLeaf = isLeaf;
   this.isToggleable = !isLeaf;
   this.parentId = parentId;
   this.childrenIds = childrenIds;
   this.domNode = domNode;
}

// Shorthand for brevity's sake
var _c = vpx.browser.SnapshotNode;    // Class
var _i = _c.prototype;                // Instance
_i._c = _c;                           // this._c --> static vars (class)
_i._concrete = true;                  // vpx system flag for concrete classes

_c.LINE_WIDTH      = '10';            // const int [px]

_c.LBL_OP_CMD      = 'snapshotOp';    // VmSnapshotOperationsAction.LBL_SNAPSHOT_CMD
_c.LBL_SNAPSHOT_ID = 'snapshotID';    // VmSnapshotOperationsAction.LBL_SNAPSHOT_MOR

_c.LBL_NAME        = "name";          // VmSnapshotOperationsAction.LBL_SNAPSHOT_NAME
_c.LBL_DESC        = "desc";          // VmSnapshotOperationsAction.LBL_SNAPSHOT_DESC
_c.LBL_MEMORY      = "memory";        // VmSnapshotOperationsAction.LBL_SNAPSHOT_MEMORY
_c.LBL_QUIESCE     = "quiesce";       // VmSnapshotOperationsAction.LBL_SNAPSHOT_QUIESCE

_c.OP_REVERT       = 'revert';        // VmSnapshotOperationsAction.OP_SNAPSHOT_REMOVE
_c.OP_REMOVE       = 'remove';        // VmSnapshotOperationsAction.OP_SNAPSHOT_REVERT
_c.OP_CREATE       = 'create';        // VmSnapshotOperationsAction.OP_SNAPSHOT_CREATE
_c.OP_RENAME       = 'rename';        // VmSnapshotOperationsAction.OP_SNAPSHOT_RENAME

_c.STATUS_SUCCESS  = 'success';       // VmSnapshotOperationsAction.OP_SNAPSHOT_SUCCESS
_c.STATUS_FAILURE  = 'failure';       // VmSnapshotOperationsAction.OP_SNAPSHOT_FAILURE

_i.id;
_i.name;
_i.desc;
_i.date;
_i.isLeaf;
_i.isToggleable;
_i.isExpanded;
_i.context;
_i.domNode;
_i.isCurrentSnapshot;

/**
 * toString()
 */
_i.toString = function() {
   return "[Object vpx.browser.SnapshotNode@"+this.id+"]";
}

/**
 * toHtml()
 */
_i.toHtml = function () {
   var curSnapshotExt = '';
   var aBegin = '';
   var aEnd = '';
   var curSnapshotId = '';
   var node;
   var connectionToNextNode = '';

   if (this.isCurrentSnapshot()) {
      curSnapshotExt = ' (current)';
      curSnapshotId = 'id="currentSnapshot"';
   }

   if (!this.isLeaf) {
      aBegin = '<a href="#">';
      aEnd = '</a>';
      connectionToNextNode = '<tr><td><img class="pre" src="imx/spacer.png" border="0"  style="width:10px" /></td><td class="verticalLine"><img src="imx/spacer.png" border="0" /></td><td></td></tr>';
   }
   
   node =  '<tr>' +
              '<td class="horizontalLine pre"><img id="pre" src="imx/spacer.png" border="0" style="width:10px" /></td>' +
              '<td class="vmImg" onClick="vpx.context.snapshots[\'' + this.id +'\'].toggle();">' + aBegin + '<img id="spacer" src="imx/spacer.png" border="0" />' + aEnd + '</a></td>' +
              '<td ' + curSnapshotId + ' class="vmName" ' +
                 'onclick="vpx.context.snapshots[\'' + this.id +'\'].showInfo();">' + 
                    '<a href="#node_' + this.id + '">' + this.name + curSnapshotExt + '</a><a name="node_' + this.id + '">&nbsp;</a>' +
              '</td>' +
           '</tr>';
   // if name is not set, then this is a fake root
   if (this.name == null) {
      node = '<tr></tr>';
   }
   return   '<table class="snapshotItem" border="0" cellpadding="0" cellspacing="0" style="margin-top:6px;table-layout:auto" >' +
               node +
               connectionToNextNode +
            '</table>';
}

/**
 * init()
 */
_i.init = function () {
   var _s = vpx.context.snapshots;
   
   if (!_s) {
      throw new Error('vpx.browser.SnapshotNode#init() the required data repository "vpx.context.snapshots" has not been defined.');
   }
   // register itself in the snapshot repository
   _s[this.id] = this;
   
   if (this.isCurrentSnapshot()) {
      _s.currentSnapshotID = this.id;
   }
   
   this.isExpanded = true;
   this.domNode.innerHTML = this.toHtml();
};


/**
 * destroy() nullifies references to other objects
 */

_i.destroy = function () {
   this.childrenIds = null;
   this.domNode = null;
}


/**
 * isCurrentSnapshot() returns true if this node represents the current snapshot
 */
 
_i.isCurrentSnapshot = function () {
   return vpx.context.snapshots.currentSnapshot == this.snapshotID;
}

/**
 * showInspectorInfo() prints information and operations available to this snapshot 
 * into the "inspector" section in the workspaceDetails
 */
 
_i.showInspectorInfo = function () {
   // shortcut for static vpx.context
   var _s = vpx.context.snapshots

   // make this node appear selected, but first reset the former one to normal if not null
   if ( isDefined(_s.selectedNode) ) {
      vpx.xua.removeClass(_s.selectedNode, 'nodeSelected');
   }
   _s.selectedNode = this.getSnapshotNameElem();
   vpx.xua.addClass(_s.selectedNode, 'nodeSelected');
   
   // show details in inspector
   var inspectorDiv = _s.context.document.getElementById('inspector');
   
   inspectorDiv.innerHTML =   '<table class="info">' +
                                 '<tr>' +
                                    '<td class="info title">Name:</td>' +
                                    '<td class="info value">' +
                                       '<input id="change_name" ' +
                                              'class="modifiable" type="text" ' +
                                              'onfocus="vpx.xua.removeClass(\'change_tr\', \'disabled\');" ' +
                                              'onchange="vpx.context.snapshots.renamePossible = true;" ' +
                                              'onmouseover="vpx.xua.addClass(this, \'highlight\');" ' +
                                              'onmouseout="vpx.xua.removeClass(this, \'highlight\');" ' +
                                              'value="' + this.name + '"' +
                                    '</td>' +
                                 '</tr>' +
                                 '<tr>' +
                                    '<td class="info title">Description:</td>' +
                                    '<td class="info value">' +
                                       '<textarea id="change_desc" ' +
                                                 'class="modifiable" value="" ' +
                                                 'onfocus="vpx.xua.removeClass(\'change_tr\', \'disabled\');" ' +
                                                 'onchange="vpx.context.snapshots.renamePossible = true;" ' +
                                                 'onmouseover="vpx.xua.addClass(this, \'highlight\');" ' +
                                                 'onmouseout="vpx.xua.removeClass(this, \'highlight\');" ' +
                                                 'type="text" >' + 
                                          this.desc + 
                                       '</textarea>' +
                                    '</td>' +
                                 '</tr>' +
                                 '<tr>' +
                                    '<td class="info title">Creation date:</td>' +
                                    '<td class="info value">' + this.date + '</td>' +
                                 '</tr>' +
                                 '<tr id="change_tr" class="disabled">' +
                                    '<td class="info title"></td>' +
                                    '<td class="info value">' + 
                                       '<input id="change_button" type="button" value="' + 'change' + '" ' + 
                                          'onClick="vpx.browser.SnapshotNode.renameSnapshot(\''+this.snapshotID+'\', $(\'change_name\').value, $(\'change_desc\').value); '+
                                                   'vpx.xua.addClass(\'change_tr\', \'disabled\');" '+
                                       '\>' +
                                    '</td>' +
                                 '</tr>' +
                              '</table>';
}


/**
 * showOperationsInfo() prints information and operations available to this snapshot 
 * into the "operations" section in the workspaceDetails
 */
 
_i.showOperationsInfo = function () {
   // shortcut for static vpx.context
   var _s = vpx.context.snapshots;

   // show available operations in inspector
   var operationsDiv = _s.context.document.getElementById('operations');
   var operationsDivContent = '<span class="disabled">No snapshot operations are supported on this VM.</span>';
   var removeSnapshotCmd = '';
   var revertToSnapshotCmd = '';
   var createSnapshotCmd = '';
   
   // remove is always allow when operations are allowed
   removeSnapshotCmd      = '<li><a href="javascript:vpx.win.confirm(\'Do you really want to remove this snapshot?<br/>['+this.name+']\', window, vpx.browser.SnapshotNode.removeSnapshot.bind(vpx.win, \'' + this.snapshotID + '\'), null );">Remove "' + this.name + '"</a></li>';
   
   if (_s.revertToSnapshotSupported && !this.isCurrentSnapshot() ) {
      revertToSnapshotCmd = '<li><a href="javascript:vpx.win.confirm(\'Do you really want to revert this snapshot?<br/>['+this.name+']\', window, vpx.browser.SnapshotNode.revertToSnapshot.bind(vpx.win, \'' + this.snapshotID + '\'), null ) ;">Revert to "' + this.name + '"</a></li>';
   }

   if (_s.poweredOffSnapshotsSupported || !_s.isVmPoweredOff ) {
      createSnapshotCmd = vpx.browser.SnapshotNode.getSnapshotCreateEntry(_s.memorySnapshotsSupported, _s.quiescedSnapshotsSupported, 'node_'+this.id );
   }
   
   if (_s.operationsSupported) {
	   operationsDivContent = '<ul>' +
	                             createSnapshotCmd +
	                             removeSnapshotCmd +
	                             revertToSnapshotCmd +
                              '</ul>';
   }

   operationsDiv.innerHTML = operationsDivContent;
};


/**
 * Static function for rendering the HTML for Snapshot Creation command
 * 
 * @param memory boolean if memory snapshotting is supported
 * @param quiesced boolean if quiesced snapshotting is supported
 * @return HTML string
 */
 
_c.getSnapshotCreateEntry = function(memory, quiesced) {
   var _s = vpx.context.snapshots;
   var basedOn = '';
   
   var memorySnapshotOption   =  '<tr>' +
					                '<td class="info title checkbox"><input type="checkbox" id="create_memory"></td>' +
					                '<td class="info value">' + 'Snaphot the virtual machine\'s memory' + '</td>' +
					             '</tr>';
                            
   var quiescedSnapshotOption =  '<tr>' +
                                    '<td class="info title checkbox"><input type="checkbox" id="create_quiesce"></td>' +
	                                '<td class="info value">' + 'Use quiesce mode' + '</td>' +
 	                             '</tr>';

   if (!isDefined(memory) || !memory) {
      memorySnapshotOption = '';
   }
   
   if (!isDefined(quiesced) || !quiesced) {
      quiescedSnapshotOption = '';
   }
   
   var createSnapshotCmd ='<tr>' +
	                         '<td class="info title"></td>' +
	                         '<td id="createSnapshot" class="info value">' +
	                            '<input type="button" value="' + 'create' + '"' +
	                               'onClick="vpx.browser.SnapshotNode.createSnapshot( $(\'create_name\').value, \
	                                                                                  $(\'create_desc\').value, \
	                                                                                  $(\'create_memory\').checked, \
	                                                                                  $(\'create_quiesce\').checked \
	                                                                                 ); \
	                                        $(\'create_memory\').checked=\'\'; \
	                                        $(\'create_quiesce\').checked=\'\'; \
	                                        $(\'create_name\').value=\'\'; \
	                                        $(\'create_desc\').value=\'\';" ' +
	                             '/>' +
	                         '</td>' +
	                      '</tr>';
   
   if (isDefined(_s.currentSnapshotID)) {
      basedOn = '<span class="disabled">(based on state of "'+ _s[_s.currentSnapshotID].name +')</span>';
   }
	                  
   var createSnapshotOption ='<div onclick="toggleShadeState(\'create\'); toggleShadeIcon(\'create_icon\')">' +
                                '<li id="create_icon" class="collapsed">' +
                                   '<a href="#node_'+_s.currentSnapshotID+'">Create Snapshot.. <br/>' + basedOn + '</a>' +
                                '</li>' +
                             '</div>' +
                             '<div id="create" class="shaded">' +
                                '<table class="info">' +
                                   '<tr>' +
                                '<td class="info title">Name:</td>' +
                                '<td class="info value">' +
                                   '<input id="create_name" ' +
                                      'class="modifiable" type="text" value="' + '" ' + 
                                      'onmouseover="vpx.xua.addClass(this, \'highlight\');" ' +
                                      'onmouseout="vpx.xua.removeClass(this, \'highlight\');" ' +
                                      ' />' +
                                '</td>' +
                             '</tr>' +
                             '<tr>' +
                                '<td class="info title">Description:</td>' +
                                '<td class="info value">' +
                                   '<textarea id="create_desc" ' +
                                      'class="modifiable" type="text" ' +
                                      'onmouseover="vpx.xua.addClass(this, \'highlight\');" ' +
                                      'onmouseout="vpx.xua.removeClass(this, \'highlight\');" ' +
                                      ' ></textarea>' +
                                '</td>' +
                             '</tr>' +
                                memorySnapshotOption +
                                quiescedSnapshotOption +
                                createSnapshotCmd +
                             '</table>' +
                             '</div>';
                             
   return createSnapshotOption;
};


/**
 * showInfo() is a convenience wrapper for show{Inspector, Options}Info()
 */
 
_i.showInfo = function () {
   this.showInspectorInfo();
   this.showOperationsInfo();
}

/**
 * toggle() toggles between collapsed and expanded views of this node
 */
 
_i.toggle = function () {
   this.showInspectorInfo();
   this.showOperationsInfo();
   var curNode;
   if (!this.isToggleable) {
      // we cannot toggle a leaf node
      return;
   }
   
   if (this.isExpanded) {
   // COLLAPSING
      vpx.xua.removeClass(this.getCornerElem(), 'verticalLine');
      for (var i=0; i<this.childrenIds.length; i++) {
         vpx.context.snapshots[this.childrenIds[i]].collapse();
      }
   } else {
   // EXPANDING
      vpx.xua.addClass(this.getCornerElem(), 'verticalLine');
      for (var i=0; i<this.childrenIds.length; i++) {
         vpx.context.snapshots[this.childrenIds[i]].expand();
      }
   }
   this.isExpanded = !this.isExpanded;
};


/**
 * expand() expands this node and shows its children in their respective state
 */

_i.expand = function () {
   this.domNode.style.visibility = 'visible';
   //this.domNode.innerHTML = this.toHtml();
   if (this.id != '0_0' ) {
      vpx.browser.SnapshotNode.drawBranch(this.id, this.parentId);
   }
   // make this' children visible as well
   var child;
   for (var i=0; i<this.childrenIds.length; i++) {
      child = vpx.context.snapshots[this.childrenIds[i]];
      if (this.isExpanded) {
         if (!this.isLeaf) {
		    vpx.xua.addClass(this.getCornerElem(), 'verticalLine');
		 }
         child.expand();
      }
   }
};


/**
 * collapse() hides the whole branch hierarchically below this node
 */

_i.collapse = function () {
   if (!this.isLeaf) {
      vpx.xua.removeClass(this.getCornerElem(), 'verticalLine');
   }
   this.domNode.style.visibility = 'hidden';
   //this.domNode.innerHTML = '<img src="imx/spacer.png" border="0" />';
   
   vpx.browser.SnapshotNode.removeBranch(this.id, this.parentId)
   
   // make this' children invisible as well
   for (var i=0; i<this.childrenIds.length; i++) {
      vpx.context.snapshots[this.childrenIds[i]].collapse();
   }
};


/**
 * getChildren()
 * 
 * @return SnapshotNode[]
 *    all children of this node 
 */
 
_i.getChildren = function() {
   var arr = Array();
   for (var i=0; i<this.childrenIds.length; i++) {
      arr.push( vpx.context.snapshots[this.childrenIds[i]] );
   }
   return arr;
};


/**
 * getRootNode() static function returning the root SnapshotNode instance
 */
 
_c.getRootNode = function() {
   return vpx.context.snapshots['0_0'];
};


/**
 * i.getCornerElem()
 * @ return TD Html Object
 *    the table cell that stores the corner element (only available on non-leaf nodes)
 */

_i.getCornerElem = function() {
   return this.domNode.getElementsByTagName('tr')[1].getElementsByTagName('td')[1];
};


/**
 * i.getSnapshotNameElem()
 * @ return TD Html Object
 *    the table cell that stores the snapshot name element (only available on non-leaf nodes)
 */

_i.getSnapshotNameElem = function() {
   return this.domNode.getElementsByTagName('tr')[0].getElementsByTagName('td')[2];
};


/**
 * drawBranch() static function that draws a branch from this node to its parent
 */
 
_c.drawBranch = function(childId, parentId) {
   var xChild, yChild, xParent, yParent, tmp, elem;

   // convention for id is: x-coord + "_" + y-coord
   // --> extract coordinates
   tmp = childId.split('_');
   xChild = tmp[0];
   yChild = tmp[1];
   
   tmp = parentId.split('_');
   xParent = tmp[0];
   yParent = tmp[1];
   
   // assume this invariant for coord: {x,y}Child>={x,y}Parent
   for(var x=xChild-1; x>=xParent; x--) {
      elem = x+'_'+yChild;
      if ( vpx.xua.hasClass(elem, 'node')) {
         continue;
      }
      vpx.xua.addClass(elem, 'horizontal');
   }
   for(var y=yChild; y>yParent; y--) {
      elem = vpx.xua.getElement(xParent+'_'+y);
      if ( vpx.xua.hasClass(elem, 'node')) {
         continue;
      }
      if ( vpx.xua.hasClass(elem, 'horizontal') ) {
         //tmp.style.backgroundColor='yellow';
         vpx.xua.removeClass(elem, 'horizontal');
         vpx.xua.addClass(elem, 'cross');
         elem.innerHTML = vpx.browser.SnapshotNode.drawCorner();
      } else {
         vpx.xua.addClass(elem, 'vertical');
      }
   }
};


/**
 * removeBranch() static function that removes a branch from this node to its parent
 */
 
_c.removeBranch = function (childId, parentId) {
   var xParent, yParent, xChild, yChild;
   
   // convention for id is: x-coord + "_" + y-coord
   // --> extract coordinates
   tmp = childId.split('_');
   xChild = tmp[0];
   yChild = tmp[1];
   tmp = parentId.split('_');
   xParent = tmp[0];
   yParent = tmp[1];

   // assume this invariant for coord: {x,y}Child>={x,y}Parent
   for(var y=yChild; y>yParent; y--) {
      elem = vpx.xua.getElement(xParent+'_'+y);
      if ( vpx.xua.hasClass(elem, 'node')) {
         continue;
      }
      vpx.xua.removeClass(elem, 'cross');
      vpx.xua.removeClass(elem, 'vertical');
      vpx.xua.removeClass(elem, 'horizontal');
      elem.innerHTML = '';
   }
   for(var x=xChild; x>=xParent; x--) {
      elem = x+'_'+yChild;
      if ( vpx.xua.hasClass(elem, 'node')) {
         continue;
      }
      vpx.xua.removeClass(elem, 'cross');
      vpx.xua.removeClass(elem, 'vertical');
      vpx.xua.removeClass(elem, 'horizontal');
      elem.innerHTML = '';
   }
};

/**
 * drawCorner() Static function..
 */
 
_c.drawCorner = function() {
   return  '<table class="corner" border="0" cellpadding="0" cellspacing="0"  >' +
              '<tr>' +
                 '<td class="verticalLine"></td>' +
                 '<td class="horizontalLine"><img id="spacer" src="imx/spacer.png" border="0" /></td>' +
              '</tr>' +
            '</table>';
}

/*********************/
/* SNAPSHOT COMMANDS */
/*********************/


/**
 * removeSnapshot() removes this snapshot from current VM.
 */

_c.removeSnapshot = function(snapshotID) {
   var _s = vpx.context.snapshots;
   var _c = vpx.browser.SnapshotNode;
   var cmdStr = 'vmSnapshotOps.do?'+_c.LBL_OP_CMD     + '=' + _c.OP_REMOVE + '&'
                                   +_c.LBL_SNAPSHOT_ID + '=' + vpx.xua.urlEncode(snapshotID);
   var req = tle.getRequest(cmdStr);
   try {
      req.addResponseListener( _s.responseListener );
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
//   vpx.session.setAttribute("isSnapshotCmdPending", true);
   _s.context.toggleProcessing('snapshotInspector', true);
   _s.context.toggleProcessing('snapshotBrowser', true);
   _s.context.toggleProcessing('snapshotOperations', true);
};


/**
 * renameSnapshot() removes this snapshot from current VM.
 */

_c.renameSnapshot = function(snapshotID, newName, newDesc) {
   var _s = vpx.context.snapshots;
   var _c = vpx.browser.SnapshotNode;
   var cmdStr = 'vmSnapshotOps.do?'+_c.LBL_OP_CMD       + '=' + _c.OP_RENAME + '&'
                                   +_c.LBL_SNAPSHOT_ID  + '=' + vpx.xua.urlEncode(snapshotID)+ '&'
                                   +_c.LBL_NAME         + '=' + vpx.xua.urlEncode(newName)+ '&'
                                   +_c.LBL_DESC         + '=' + vpx.xua.urlEncode(newDesc);
   var req = tle.getRequest(cmdStr);
   try {
      req.addResponseListener( _s.responseListener );
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
//   vpx.session.setAttribute("isSnapshotCmdPending", true);
   _s.context.toggleProcessing('snapshotInspector', true);
   _s.context.toggleProcessing('snapshotBrowser', true);
   _s.context.toggleProcessing('snapshotOperations', true);
};


/**
 * revertToSnapshot() reverts to this snapshot on the current VM.
 */

_c.revertToSnapshot = function(snapshotID) {
   var _s = vpx.context.snapshots;
   var _c = vpx.browser.SnapshotNode;
   var cmdStr = 'vmSnapshotOps.do?'+_c.LBL_OP_CMD      + '=' + _c.OP_REVERT + '&'
                                   +_c.LBL_SNAPSHOT_ID + '=' + vpx.xua.urlEncode(snapshotID);
   var req = tle.getRequest(cmdStr);
   try {
      req.addResponseListener( _s.responseListener );
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
//   vpx.session.setAttribute("isSnapshotCmdPending", true);
   _s.context.toggleProcessing('snapshotInspector', true);
   _s.context.toggleProcessing('snapshotBrowser', true);
   _s.context.toggleProcessing('snapshotOperations', true);
};


/**
 * createSnapshot() static function that creates a new snapshot on current VM.
 */

_c.createSnapshot = function(name, desc, memory, quiesce) {
   var _s = vpx.context.snapshots;
   var _c = vpx.browser.SnapshotNode;
   var cmdStr = 'vmSnapshotOps.do?'+_c.LBL_OP_CMD  + '=' + _c.OP_CREATE + '&'
                                   +_c.LBL_NAME    + '=' + vpx.xua.urlEncode(name) + '&'
                                   +_c.LBL_DESC    + '=' + vpx.xua.urlEncode(desc) + '&'
                                   +_c.LBL_MEMORY  + '=' + memory + '&'
                                   +_c.LBL_QUIESCE + '=' + quiesce ;
   var req = tle.getRequest(cmdStr);
   try {
      req.addResponseListener( _s.responseListener );
      req.send();
   } finally {
      tle.releaseRequest(req);
   }
//   vpx.session.setAttribute("isSnapshotCmdPending", true);
   _s.context.toggleProcessing('snapshotInspector', true);
   _s.context.toggleProcessing('snapshotBrowser', true);
   _s.context.toggleProcessing('snapshotOperations', true);
};