/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

// Ensure package dependencies are satisfied
if (!(vpx.xua && vpx.gui)) {
   throw new Error("vpx.browser: Package dependency failed (xua,gui)");
}

// Create the package namespace
vpx.browser = {};

// Once loaded, icons are cached (keyed by src) for quicker rendering
vpx.browser._cachedIcons = {};

try {
   vpx.browser.TRAVERSAL = document.implementation.hasFeature("Traversal", 2.0);
} catch (ex) {
   vpx.browser.TRAVERSAL = false;
}

/**
 * Events that can be listened for using vpx.EventSource#registerListener().
 * When events are fired, each listener will be invoked in the order in which
 * they subscribed to the event.  They will automatically be passed certain
 * arguments, which are documented inline for each event.
 */
vpx.browser.EVENT = {
   "DATA"           : 0, /**
                          * Browser has finished processing server data.
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          * @param xmlSpec vpx.net.XmlSpec
                          *    The xml spec used to request the data
                          */

   "SELECTION"      : 1, /**
                          * User changed the current selection.
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          */

   "NAVIGATE"       : 2, /**
                          * User navigated to a different base item
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          * @param baseItem vpx.ServerObj
                          *    The object being navigated to
                          */

   "PAGE"           : 3, /**
                          * User paged.
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          * @param pageType const String
                          *    The page command (e.g. vpx.browser.PAGE.FIRST)
                          * @param pageNum (optional) int
                          *    The 0-based page number
                          */

   "SORT"           : 4,  /**
                          * User sorted on a column.
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          * @param column vpx.browser.Column
                          *    The column being sorted on
                          * @param type const String
                          *    The sort type (e.g. vpx.browser.SORT.ASC)
                          */

   "ERROR"          : 5  /**
                          * The browser processed error data.
                          *
                          * @param browser vpx.browser.Browser
                          *    The object that fired the event
                          */
};

/**
 * The types of paging actions, passed as an extra argument to handlers of
 * event type PAGE.
 */
vpx.browser.PAGE = {
   "FIRST"         : "firstPage",
   "LAST"          : "lastPage",
   "NEXT"          : "nextPage",
   "PREV"          : "prevPage",
   "REFRESH"       : "refreshPage",
   "JUMP"          : "jumpPage"
};

/**
 * The types of sorting that can be done.
 */
vpx.browser.SORT = {
   "NONE"          : null,
   "ASC"           : "ascending",
   "DESC"          : "descending"
};

/**
 * Gets an icon by its source. For performance improvement, this looks for a
 * previously loaded image with the same source and clones it if possible.
 * If not, it will create the image from scratch.
 *
 * @param src String
 *    The image source
 * @return HTMLImageElement
 *    An icon ready for insertion into the document
 */
vpx.browser.getIcon = function(src) {
   var icon = vpx.browser._cachedIcons[src];
   if (isNull(icon)) {
       icon = document.createElement("img");
       icon.alt = "";
       icon.src = src;
       icon.width = icon.height = 16;
       vpx.browser._cachedIcons[src] = icon;
   } else {
       // Cloning a cached node is faster than creating a new node
       icon = icon.cloneNode(false);
   }
   return icon;
};
