/* Copyright 2006 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class HashMap
 * extends vpx.core.VpxObject
 *
 * An object that maps keys to values. A map cannot contain duplicate keys;
 * each key can map to at most one value.
 *
 * @version 1.0 (Jun 21, 2006)
 */

/**
 * Constructs an empty <code>HashMap</code>
 */
vpx.core.HashMap = function() {
   this.attrKeys = [];
   this.attributes = {};
};

// HashMap extends vpx.core.VpxObject
vpx.core.HashMap.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.core.HashMap.prototype.constructor = vpx.core.HashMap;

// Shorthand for brevity's sake
var _c = vpx.core.HashMap;            // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.attrKeys = null;                   // private Array
_i.attributes = null;                 // private Object

/**
 * Removes all mappings from this map.
 */
_i.clear = function() {
   for (var i = 0; i < this.attrKeys.length; i++) {
      var key = this.attrKeys[i];
      delete this.attributes[key];
   }
   this.attrKeys.length = 0;
};

/**
 * Returns an array view of the keys contained in this map.  This is provided
 * as an alternative to <code>Map.keySet().toArray()</code> since
 * <code>Map.keySet()</code> is not yet implemented.
 *
 * @return String[]
 *    An array of the keys contained in this map
 */
_i.keyArray = function() {
   var result = new Array(this.attrKeys.length);
   vpx.core.System.arraycopy(this.attrKeys, 0, result, 0, this.attrKeys.length);
   return result;
};

/**
 * Returns <code>true</code> if this map contains a mapping for the specified
 * key. More formally, returns <code>true</code> if and only if this map
 * contains a mapping for a key <code>k</code> such that
 * <code>key.equals(k)</code>. (There can be at most one such mapping.)
 *
 * @param key String
 *    The key whose presence in this map is to be tested
 * @return boolean
 *    <code>true</code> if this map contains a mapping for the specified key
 * @throws Error
 *    If <code>key</code> is <code>null</code>
 */
_i.containsKey = function(key) {
   if (key == null) {
      throw new Error("HashMap#containsKey(): null keys not supported");
   }
   return (this.attrKeys.indexOf(key) >= 0);
};

/**
 * Returns the value to which this map maps the specified key. Returns
 * <code>null</code> if the map contains no mapping for this key. A return
 * value of <code>null</code> does not necessarily indicate that the map
 * contains no mapping for the key; it's also possible that the map explicitly
 * maps the key to <code>null</code>. The <code>containsKey</code> operation
 * may be used to distinguish these two cases.
 * <p/>
 * More formally, if this map contains a mapping from a key <code>k</code> to a
 * value <code>v</code> such that <code>key.equals(k)</code>, then this method
 * returns <code>v</code>; otherwise it returns <code>null</code>. (There can
 * be at most one such mapping.)
 *
 * @param key String
 *    The key whose associated value is to be returned
 * @return Object
 *    The value to which this map maps the specified key, or <code>null</code>
 *    if the map contains no mapping for this key
 * @throws Error
 *    If the key is <code>null</code>
 */
_i.get = function(key) {
   if (key == null) {
      throw new Error("HashMap#get(): null keys not supported");
   }
   return (this.containsKey(key) ? this.attributes[key] : null);
};

/**
 * Returns <code>true</code> if this map contains no key-value mappings.
 *
 * @return boolean
 *    True if this map contains no key-value mappings
 */
_i.isEmpty = function() {
   return (this.attrKeys.length == 0);
};

/**
 * Associates the specified value with the specified key in this map. If the
 * map previously contained a mapping for this key, the old value is replaced
 * by the specified value. (A map <code>m</code> is said to contain a mapping
 * for a key <code>k</code> if and only if <code>m.containsKey(k)</code> would
 * return <code>true</code>.)
 *
 * @param key String
 *    The key with which the specified value is to be associated
 * @param value Object
 *    The value to be associated with the specified key
 * @return Object
 *    The previous value associated with specified key, or <code>null</code> if
 *    there was no mapping for <code>key</code>. A <code>null</code> return can
 *    also indicate that the map previously associated <code>null</code> with
 *    the specified key
 * @throws Error
 *    If the key is <code>null</code>
 */
_i.put = function(key, value) {
   if (key == null) {
      throw new Error("HashMap#put(): null keys not supported");
   }
   var old = this.get(key);
   if (!this.containsKey(key)) {
      this.attrKeys.push(key);
   }
   this.attributes[key] = value;
   return old;
};

/**
 * Removes the mapping for this key from this map if it is present. More
 * formally, if this map contains a mapping from key <code>k</code> to value
 * <code>v</code> such that <code>key.equals(k)</code>, that mapping is
 * removed. (The map can contain at most one such mapping.)
 * <p/>
 * Returns the value to which the map previously associated the key, or
 * <code>null<code> if the map contained no mapping for this key. (A
 * <code>null</code> return can also indicate that the map previously
 * associated <code>null</code> with the specified key.) The map will not
 * contain a mapping for the specified key once the call returns.
 *
 * @param key String
 *    The key whose mapping is to be removed from the map
 * @return Object
 *    The previous value associated with specified key, or <code>null</code> if
 *    there was no mapping for key
 * @throws Error
 *    If the key is <code>null</code>
 */
_i.remove = function(key) {
   if (key == null) {
      throw new Error("HashMap#remove(): null keys not supported");
   }
   var old = this.get(key);
   if (this.containsKey(key)) {
      var index = this.attrKeys.indexOf(key);
      this.attrKeys.splice(index, 1);
      delete this.attributes[key];
   }
   return old;
};

/**
 * Returns the number of key-value mappings in this map.
 *
 * @return int
 *    The number of key-value mappings in this map
 */
_i.size = function() {
   return this.attrKeys.length;
};
