/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class EventListenerList
 * extends Object
 *
 * A class that holds a list of EventListeners.  A single instance can be used
 * to hold all listeners (of all types) for the instance using the list.  It is
 * the responsiblity of the class using the EventListenerList to provide
 * type-safe API (preferably conforming to the JavaBeans spec) and methods that
 * dispatch event notification methods to appropriate Event Listeners on the
 * list.
 * <p/>
 * The main benefits that this class provides are that it is relatively cheap
 * in the case of no listeners, and it provides serialization for
 * event-listener lists in a single place, as well as a degree of MT safety
 * (when used correctly).
 * <p/>
 * Usage example:
 * Say you are defining a class that sends out FooEvents, and you want to allow
 * users of the class to register FooListeners and receive notification when
 * FooEvents occur.  The following should be added to the class definition:
 * <pre>
 * var listenerList = new vpx.core.event.EventListenerList();
 * var fooEvent = null;
 *
 * function addFooListener(l) {
 *     listenerList.add("FooListener", l);
 * }
 *
 * function removeFooListener(l) {
 *     listenerList.remove("FooListener", l);
 * }
 *
 * // Notify all listeners that have registered interest for
 * // notification on this event type.  The event instance
 * // is lazily created using the parameters passed into
 * // the fire method.
 *
 * protected void fireFooXXX() {
 *     // Guaranteed to return a non-null array
 *     var listeners = listenerList.getListenerList();
 *
 *     // Process the listeners last to first, notifying
 *     // those that are interested in this event
 *     for (var i = listeners.length - 2; i >= 0; i -= 2) {
 *         if (listeners[i] == "FooListener") {
 *             // Lazily create the event:
 *             if (fooEvent == null) {
 *                 fooEvent = new vpx.core.event.FooEvent(this);
 *             }
 *
 *             // listeners[i + 1] is of type vpx.core.event.FooListener
 *             listeners[i + 1].fooXXX(fooEvent);
 *         }
 *     }
 * }
 * </pre>
 *
 * foo should be changed to the appropriate name, and fireFooXxx to the
 * appropriate method name. One fire method should exist for each notification
 * method in the FooListener interface.
 */

vpx.core.event.EventListenerList = function() {};

// Shorthand for brevity's sake
var _c = vpx.core.event.EventListenerList; // Class
var _i = _c.prototype;                     // Instance
_i._concrete = true;                       // vpx system flag for concrete classes

_c.NULL_ARRAY = [];                        // private final static Object[]

// Instance variables
_i.listenerList = _c.NULL_ARRAY;           // protected transient Object[]

/**
 * Passes back the event listener list as an array of ListenerType-listener
 * pairs.  Note that for performance reasons, this implementation passes back
 * the actual data structure in which the listener data is stored internally!
 * <p/>
 * This method is guaranteed to pass back a non-null array, so that no
 * null-checking is required in fire methods.  A zero-length array of Object
 * should be returned if there are currently no listeners.
 * <p/>
 * WARNING!!! Absolutely NO modification of the data contained in this array
 * should be made -- if any such manipulation is necessary, it should be done
 * on a copy of the array returned rather than the array itself.
 *
 * @return Object[]
 *    The event listener list, guaranteed to be non-null
 */
_i.getListenerList = function() {
   return this.listenerList;
};

/**
 * Return an array of all the listeners of the given type, guaranteed to be
 * non-null.
 *
 * @param t String
 *    The type string representing a class of objects
 * @return vpx.core.event.EventListener[]
 *    All of the listeners of the specified type.
 */
_i.getListeners = function(t) {
   var lList = this.listenerList;
   var n = this._getListenerCount(lList, t);
   var result = new Array(n);
   var j = 0;
   for (var i = lList.length - 2; i >= 0; i -= 2) {
      if (lList[i] == t) {
         result[j++] = lList[i + 1];
      }
   }
   return result;
};

/**
 * Dispatch function to either <code>getTotalListenerCount()</code> or
 * <code>getListenerCountByType(String)</code>, base on the number of
 * arguments found.
 *
 * @return int
 *    Depends on which function received the dispatch
 */
_i.getListenerCount = function() {
   switch (arguments.length) {
   case 0:
      return this.getTotalListenerCount();
      break;
   case 1:
      return this.getListenerCountByType(arguments[0]);
      break;
   default:
      throw new Error("EventListenerList#getListenerCount(): Unsupported args #: " +
                      arguments.length);
   }
};

/**
 * Returns the total number of listeners for this listener list.
 *
 * @return int
 *    The total number of listeners for this listener list
 */
_i.getTotalListenerCount = function() {
   return this.listenerList.length / 2;
};

/**
 * Returns the total number of listeners of the supplied type for this listener
 * list.
 *
 * @param t String
 *    The type string representing a class of objects
 * @return int
 *    The number of listeners for the given type
 */
_i.getListenerCountByType = function(t) {
   return this._getListenerCount(this.listenerList, t);
};

/**
 * Adds the listener as a listener of the specified type.
 *
 * XXX Is thread safety an issue here?
 *
 * @param t String
 *    The type string representing the type of the listener to be added
 * @param l vpx.core.event.EventListener
 *    the listener to be added
 */
_i.add = function(t, l) {
   if (l == null) {
      //?? XXX Throw error?
      return;
   }

   //?? XXX Enforce type safety of l?

   var cls = vpx.core.event.EventListenerList;
   if (this.listenerList == cls.NULL_ARRAY) {
      // If this is the first listener added, initialize the lists
      this.listenerList = [t, l];
   } else {
      // Otherwise copy the array and add the new listener
      var i = this.listenerList.length;
      var tmp = new Array(i + 2);
      vpx.core.System.arraycopy(this.listenerList, 0, tmp, 0, i);

      tmp[i] = t;
      tmp[i + 1] = l;

      this.listenerList = tmp;
   }
};

/**
 * Removes the listener as a listener of the specified type.
 *
 * XXX Is thread safety an issue here?
 *
 * @param t String
 *    The type string representing the type of the listener to be removed
 * @param l vpx.core.event.EventListener
 *    the listener to be removed
 */
_i.remove = function(t, l) {
   if (l == null) {
      //?? XXX Throw error?
      return;
   }

   //?? XXX Enforce type safety of l?

   // Is l on the list?
   var index = -1;
   for (var i = this.listenerList.length - 2; i >= 0; i -= 2) {
      if (this.listenerList[i] == t && arePolyEqual(this.listenerList[i + 1], l)) {
         index = i;
         break;
      }
   }

   // If so, remove it
   if (index != -1) {
      var tmp = new Array(this.listenerList.length - 2);
      // Copy the list up to index
      vpx.core.System.arraycopy(this.listenerList, 0, tmp, 0, index);
      // Copy from two past the index, up to the end of tmp
      // (which is two elements shorter than the old list)
      if (index < tmp.length) {
         vpx.core.System.arraycopy(this.listenerList, index + 2, tmp, index,
                                   tmp.length - index);
      }
      // set the listener array to the new array or null
      var cls = vpx.core.event.EventListenerList;
      this.listenerList = (tmp.length == 0 ? cls.NULL_ARRAY : tmp);
   }
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Returns the number of listeners in the given list that are of the given
 * type.
 *
 * @param list Object[]
 *    The 2-step array of listeners (Constructor/Listener, ...)
 * @param t String
 *    The type string representing a class of objects
 * @return int
 *    The number of listeners in the given list that are of the given type
 */
_i._getListenerCount = function(list, t) {
   var count = 0;
   for (var i = 0; i < list.length; i += 2) {
      if (t == list[i]) {
         count++;
      }
   }
   return count;
};
