/* Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential */

/**
 * Provides a singleton object for managing draggable elements.
 */

vpx.drag = {"dragged": null};


/**
 * Given an HTML element, prevent text selection actions from affecting it.
 * Captured events are prevented from propagating, even to the browser itself.
 *
 * @param o     the HTML element to protect from text selections
 */

vpx.drag.initVoidSelection = function (o) {
   vpx.xua.event.listen(o, "mousedown", vpx.drag.voidSelection);
   vpx.xua.event.listen(o, "selectstart", vpx.drag.voidSelection);
   vpx.xua.event.listen(o, "dragstart", vpx.drag.voidSelection);
};

vpx.drag.voidSelection = function (e) {
   // XXX: If mousedown is captured, Mozilla sometimes reports mousemove events
   // as if they originated in the drag context window (which is not always the
   // case) rather than the drag viewport. Ironically, Mozilla also fails to
   // capture mouse movements that escape the handle area by moving over the
   // drag context window. As a result, dragging suspends until the mouse
   // returns to the draggable area.
   if (e.type != "mousedown" || areStrictlyEqual(vpx.xua.event.getSource(e), e.target)) {
      vpx.xua.event.cancel(e);
   }
};


/**
 * Given the handle for a collection of draggable elements, calculate their
 * total area.
 *
 * @param o     an HTML element suitable for initiating drag operations for the
 *              draggable object. This parameter is optional; if it is not
 *              specified, the active handle for the current drag operation
 *              will be used.
 * @return      an object with properties for the position (<return>.pos) and
 *              dimensions (<return>.dim) of the draggable area
 */

vpx.drag.getDraggableArea = function (o) {
   o = o || vpx.drag.dragged;
   var parts = o.dragSpec.parts;
   var x, y, z, w, h;

   for (var i = 0; i < parts.length; i++) {
      var p = vpx.xua.getPosition(parts[i]);
      var d = vpx.xua.getDimension(parts[i]);
      var diff = 0;
      
      if (! isDefined(x)) {
	 x = p.x;
	 y = p.y;
	 z = p.z;
	 w = d.w;
	 h = d.h;
	 continue;
      }

      if (! isDefined(z) || p.z > z) {
	 z = p.z;
      }

      if (p.x <= x) {
	 if (d.w > w + (x - p.x)) {
            // p.x . . .
	    //
            //   x   .
	    w = d.w;
	 } else {
            // p.x . .
	    //
            //   x   . .
	    w += (x - p.x);
	 }

	 x = p.x;
      } else if (p.x > x) {
	 diff = (p.x + d.w) - (x + w);
	 if (diff > 0) {
	    // p.x   . .
	    //
	    //   x . .
	    w += diff;
	 }
      }

      if (p.y <= y) {
	 if (d.h > h + (y - p.y)) {
	    // p.y  y
	    //  .
	    //  .   .
	    //  .
	    h = d.h;
	 } else {
	    // p.y  y
	    //  .
	    //  .   .
	    //      .
	    h += (y - p.y);
	 }

	 y = p.y;
      } else if (p.y > y) {
	 diff = (p.y + d.h) - (y + h);
	 if (diff > 0) {
	    // p.y  y
	    //      .
	    //  .   .
	    //  .
	    h += diff;
	 }
      }
   }

   return {pos:new Position(x, y, z), dim:new Dimension(w, h)};
};


/**
 * Collect the elements of a draggable object, measure its size and prepare it
 * to be dragged.
 *
 * @param handles     a list of HTML elements that can be used to initiate
 *                    drag operations. If an element is empty (does not contain
 *                    any significant characters, not even &nbsp;) Mozilla will
 *                    report mousemove events as if they originated in the drag
 *                    context window instead of the frame where the mousedown
 *                    was received. Ironically, Mozilla also fails to capture
 *                    mouse movements that escape the handle by moving into the
 *                    drag context window. As a result, dragging will suspend
 *                    until the moves back into to the handle area.
 * @param parts       a list of html elements that should be moved during a
 *                    drag operation initiated by one of the handles. This
 *                    parameter is optional; if it is not specified, the list
 *                    of handles will be used.
 * @param context     the window or frame in which drag operations will occur.
 *                    This parameter is optional; if it is not specified, the
 *                    caller's window will be used.
 */

vpx.drag.init = function (handles, parts, context) {
   var o = handles[0];

   o.isDraggable = true;

   o.dragSpec = {};
   o.dragSpec.parts = parts || [];
   o.dragSpec.context = context || self;

   var area = vpx.drag.getDraggableArea(o);
   o.dragSpec.pos = area.pos;
   o.dragSpec.dim = area.dim;

   o.dragCallbacks = {"start":[],"drag":[],"end":[]};

   vpx.xua.event.listen(o, "mousedown", vpx.drag.start);
   vpx.drag.initVoidSelection(o);

   for (var i = 1; i < handles.length; i++) {
      o = handles[i];

      vpx.xua.event.listen(o, "mousedown", vpx.drag.start);
      vpx.drag.initVoidSelection(o);

      extendObject(o, handles[0], ["isDraggable", "dragSpec", "dragCallbacks"]);
   }
};


/**
 * Set the rectangle in which the dragged object will be constrained.
 *
 * @param handles     a list of HTML elements that can be used to initiate
 *                    drag operations for a single draggable object
 * @param min         a Position object representing the top-left corner of the
 *                    bounding rectangle.
 * @param max         a Position object representing the bottom-right corner of 
 *                    the bounding rectangle.
 */

vpx.drag.setBoundary = function (handles, min, max) {
   var o = handles[0];

   o.dragSpec.boundary = (isTypeOf(min, Limit) ? min : new Limit(min, max));

   vpx.drag.ignore(handles, "start", vpx.drag.runSetBoundaryByElement);
};


/**
 * Use an HTML element to determine the rectangle in which the dragged object
 * will be constrained.
 *
 * @param handles     a list of HTML elements that can be used to initiate
 *                    drag operations for a single draggable object
 * @param el          the HTML object used to determine the bounding rectangle
 */

vpx.drag.setBoundaryByElement = function (handles, el) {
   var o = handles[0];
   o.dragSpec.boundaryElement = el;

   vpx.drag.listen(handles, "start", vpx.drag.runSetBoundaryByElement);
   vpx.drag.runSetBoundaryByElement(o);
};

vpx.drag.runSetBoundaryByElement = function (o) {
   o = o || vpx.drag.dragged;
   var el = o.dragSpec.boundaryElement;

   var dim = vpx.xua.getDimension(el);
   var pos = vpx.xua.viewport.getOffset(vpx.xua.getView(el), o.context);

   if (! isView(el)) {
     pos = pos.add(vpx.xua.getPosition(el));
   }

   o.dragSpec.boundary = new Limit(pos, pos.add(dim.toPosition()));
};


/**
 * Prepare to handle mousemove events (i.e., in response to a mousedown event).
 *
 * @param e     the event object
 */

vpx.drag.start = function (e) {
   vpx.drag.end();
   
   
   if (vpx.xua.event.getButton(e) != vpx.xua.event.MOUSE_BUTTON_LEFT) {
      return null;
   }

   var o = vpx.xua.event.getSource(e);
   if (isDefined(o) && isDefined(o.nodeName)) {
      while (o.nodeName.toUpperCase() != "HTML" &&
	 (! isDefined(o.isDraggable) || ! o.isDraggable)) {
	 o = o.parentNode;
      }
   }
   vpx.drag.dragged = o;

   o.dragSpec.viewport = vpx.xua.getView(o);

   var area = vpx.drag.getDraggableArea(o);
   o.dragSpec.pos = area.pos;
   o.dragSpec.dim = area.dim;

   // Original position of the draggable parts.
   o.dragSpec.originalPartsPos = o.dragSpec.pos;

   // Last known mouse event position.
   o.dragSpec.lastMousePos = vpx.xua.event.getPosition(e);

   // Make the event position relative to the top-left corner of the context
   // window, not the event frame.
   if (o.dragSpec.context != o.dragSpec.viewport) {
      o.dragSpec.lastMousePos =
	o.dragSpec.lastMousePos.add(vpx.xua.viewport.getOffset(o.dragSpec.viewport,
	  o.dragSpec.context));
   }

   // Original cursor offset from the top, left corner of the base.
   o.dragSpec.originalMouseOffset =
     o.dragSpec.lastMousePos.subtract(o.dragSpec.originalPartsPos);

   // Sometimes the drag state can become stuck on, requiring a second click to
   // end the drag. Give people a hint by painting a div over the dragged
   // object to reflect its state.
   var shield = o.dragSpec.shield =
     vpx.xua.append(o.dragSpec.context.document.body, vpx.xua.create("div"));

   // XXX: Presentation should be defined in a style sheet.
   vpx.xua.setStyle(shield,
     "border",          "solid 1px #8fadcc",
     "backgroundColor", "#b8cfe6",
     "position",        "absolute");
   vpx.xua.setOpacity(shield, 0.6);

   vpx.xua.setDimension(shield, o.dragSpec.dim);
   vpx.xua.setPosition(shield, o.dragSpec.pos);
   vpx.xua.setZ(shield, (! isDefined(o.dragSpec.pos.z) ? 1 : o.dragSpec.pos.z + 1));

   for (var i = 0; i < o.dragCallbacks.start.length; i++) {
      o.dragCallbacks.start[i]();
   }

   // Drag events are sometimes received by the context document (e.g., top),
   // and sometimes by the mouse event defaultView document.
   vpx.xua.event.listen(o.dragSpec.context.document, "mousemove", vpx.drag.drag);
   vpx.xua.event.listen(o.dragSpec.context.document, "mouseup", vpx.drag.end);
   vpx.xua.event.listen(o.dragSpec.viewport.document, "mousemove", vpx.drag.drag);
   vpx.xua.event.listen(o.dragSpec.viewport.document, "mouseup", vpx.drag.end);

   return false;
};


/**
 * Manage mousemove events for draggable objects. Move all of the current
 * draggable object parts - within a bounding rectangle if one exists.
 *
 * @param e     the event object
 */

vpx.drag.drag = function (e) {
   var o = vpx.drag.dragged;

   o.dragSpec.lastMousePos = vpx.xua.event.getPosition(e);
   
   var view = vpx.xua.getView(vpx.xua.event.getSource(e));

   if (o.dragSpec.context != view) {
      o.dragSpec.lastMousePos =
	 o.dragSpec.lastMousePos.add(vpx.xua.viewport.getOffset(view,
	    o.dragSpec.context));
   }

   // Calculate move distances from the original cursor offset.
   var pos = o.dragSpec.lastMousePos.subtract(o.dragSpec.originalMouseOffset);

   if (isDefined(o.dragSpec.boundary)) {
      pos = vpx.xua.constrainPosition(pos, o.dragSpec.dim, o.dragSpec.boundary);
      window.satus = pos.toString();
   }

   var offset = pos.subtract(o.dragSpec.pos);
   
   var i;
   
   for (i = 0; i < o.dragSpec.parts.length; i++) {
      var p = vpx.xua.getPosition(o.dragSpec.parts[i]);
      vpx.xua.setPosition(o.dragSpec.parts[i], p.add(offset));
   }

   o.dragSpec.pos = pos;
   vpx.xua.setPosition(o.dragSpec.shield, pos);

   for (i = 0; i < o.dragCallbacks.drag.length; i++) {
      o.dragCallbacks.drag[i](e);
   }
};


/**
 * Clean up the dragged object after dragging it (i.e., in response to a
 * mouseup event).
 *
 * @param e     the event object
 */

vpx.drag.end = function (e) {
   if (! isDefined(vpx.drag.dragged)) {
      return;
   }

   var o = vpx.drag.dragged;

   vpx.xua.event.ignore(o.dragSpec.viewport.document, "mousemove", vpx.drag.drag);
   vpx.xua.event.ignore(o.dragSpec.viewport.document, "mouseup", vpx.drag.end);
   vpx.xua.event.ignore(o.dragSpec.context.document, "mousemove", vpx.drag.drag);
   vpx.xua.event.ignore(o.dragSpec.context.document, "mouseup", vpx.drag.end);

   vpx.xua.cut(o.dragSpec.shield);
   o.dragSpec.shield = null;

   for (var i = 0; i < o.dragCallbacks.end.length; i++) {
      o.dragCallbacks.end[i]();
   }

   o = vpx.drag.dragged = null;
};


/**
 * Add a drag event listener to a draggable object. Use this method to add
 * custom behaviors to drag operations.
 *
 * @param handles     a list of HTML elements that can be used to initiate
 *                    drag operations for a single draggable object
 * @param event       the event to listen for (start, drag or end)
 * @param method      the method to invoke when the event occurs
 */

vpx.drag.listen = function (handles, event, method) {
   var o = handles[0];

   if (! isDefined(o.dragCallbacks)) {
      return;
   }

   o.dragCallbacks[event].push(method);
};


/**
 * Remove a drag event listener from a draggable object.
 *
 * @param handles     a list of HTML elements that can be used to initiate
 *                    drag operations for a single draggable object
 * @param event       the event to ignore (start, drag or end)
 * @param method      the method to invoke when the event occurs
 */

vpx.drag.ignore = function (handles, event, method) {
   var o = handles[0];

   if (! isDefined(o.dragCallbacks)) {
      return;
   }

   for (var i = 0; i < o.dragCallbacks[event].length; i++) {
      if (o.dragCallbacks[event][i] == method) {
	 o.dragCallbacks[event].splice(i, 1);
	 break;
      }
   }
};
