/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Enhance an object by using the function as an instance method with access to
 * the <code>this</code> object self-reference.  If extra arguments are passed
 * to this function, they will in turn be passed onto the bound function before
 * any other arguments. (e.g. handleEvent.bind(obj, "foo") translates to
 * obj.handleEvent("foo", ...))
 *
 * @param o     the object to enhance
 * @param args  arguments to prepend to the invocation of the bound function
 * @return      the instance method created
 */
 
Function.prototype.bind = function(o) {
   var m = this;
   if (arguments.length <= 1) {
      // Simple case
      return function () { return m.apply(o, arguments); };
   }

   var ba = [];  // Bound args
   for (var i = 1; i < arguments.length; i++) ba.push(arguments[i]);
   return function () {
      var a = clone(ba, false);
      for (var i = 0; i < arguments.length; i++) a.push(arguments[i]);
      return m.apply(o, a);
   };
};


/**
 * Enhance an object by using the function as an event listener and instance
 * method with access to the <code>this</code> object self-reference.
 *
 * @param o     the object to enhance
 * @return      the instance method created
 */

Function.prototype.bindListener = function(o) {
   var m = this;
   return function (e) { return m.call(o, e || window.event); };
};


/**
 * Allows a method of one object to be executed in the context of a different
 * object (the calling object). When the method is applied, it will use the
 * calling object as the <code>this</code> object self-reference. Although it
 * is simialr to <code>Function.call</code>, <code>apply</code> is able to
 * handle a function whose argument list is not known.
 * <p>
 * This version is only used when <code>Function.apply</code> is not supported
 * natively by a browser.
 *
 * @param thisScope     the calling object
 * @param theseArgs     an argument array for the calling object
 * @return              the result generated by the applied method
 */

if (! isDefined(Function.prototype.apply)) {
   // Adapted from http://youngpup.net/2002/oldblog123
   //
   // BY: Aaron Boodman
   // http://creativecommons.org/licenses/by/2.0/

   Function.prototype.apply = function(thisScope, theseArgs) {
      var list = [];

      thisScope = thisScope || self;
      theseArgs = theseArgs || [];

      for (var i = 0; i < theseArgs.length; i++) {
    list[i] = "theseArgs[" + i + "]";
      }

      thisScope.__apply__ = this;
      var result = eval("thisScope.__apply__(" + list.join(",") + ");");
      delete thisScope.__apply__;
      return result;
   };
}


/**
 * The following extends all functions to allow a monitored function to notify failures, 
 * recover from it and retry the invocation automatically for a limited number of attempts.
 * 
 *
 * A couple of standard use cases are presented:
 *
 * 1) call to function foo with automatic retrial in case of failure:
 *       foo.monitoredInvoke(arg1, arg2, .., argN);
 *    This will invoke foo(args) in retry with 500ms of delay, the maximum 3 times (default)
 * 
 * 2) customize above scenario with other delay and maxiumum retrial attemps:
 *       foo.setRetrialDelay(200); // 200ms delay
 *       foo.setMaxRetrials(5);    // retry 5 times at most
 *       foo.monitoredInvoke(arg1, arg2, .., argN);
 *
 * 3) customize above scenario by specifying another context whereon to apply the function:
 *       foo.setContext(contextObj);
 *       foo.monitoredInvoke(arg1, arg2, .., argN);
 * 
 * 4) perform alternative code when first invocation attemps failes (while retrial
 *    attemps are silently performed in the background):
 *       if ( foo.monitoredInvoke(arg1, arg2, .., argN) ) {
 *             // first invocation has failed. perform alternative action..<b> 
 *       }
 *
 * 5) set an invocation listener (that implements vpx.ext.InvocationListener) to the monitored function:
 *       var myInvocationListener = new InvocationListenerImpl();
 *       myInvocationListener.invocationSucceeded = function () {
 *          // invocation has succeded.. 
 *       };
 *       myInvocationListener.invocationFailed = function () {
 *          // invocation failed.. perform recovery code here
 *       };
 *       foo.setInvocationListener(myInvocationListener);
 *       foo.monitoredInvoke(arg1, arg2, .., argN);
 */


/**
 * Sets the delay between two attempts for executing the function call
 *
 * @param delay
 *    the delay between two attempts
 */
 
Function.prototype.setRetrialDelay = function(delay) {
   this.delay = delay;
};


/**
 * Sets the maximum number of retry attempts for this function 
 *
 * @param maxRetrials
 *    the maximum number of retrial attempts
 */

Function.prototype.setMaxRetrials = function(maxRetrials) {
   this.maxRetrials = maxRetrials;
};


/**
 * Sets the context for this function 
 *
 * @param l vpx.ext.InvocationListener
 *    the listener object that provides callback for succeded and failed invocation
 */

Function.prototype.setInvocationListener = function(l) {
   if( isDefined(l) && 
       isFunction(l.invocationSucceeded) &&
       isFunction(l.invocationFailed) )
   {
      this.invocationListener = l;
   } else {
      throw new Error("Function#setInvocationListener: supplied " +
                      "object does not implement vpx.ext.InvocationListener.");
   }
};


/**
 * Sets the context for this function 
 *
 * @param context Object
 *    The context in which to invoke the function
 */

Function.prototype.setContext = function(context) {
   this.context = context;
};


/**
 * private helper function for enhancing an object by 
 *
 * @param context
 *    the originating context (Window instance)
 * @param delay
 *    time in ms between two attempts
 * @param maxRetrials
 *    maximum number if retrial attempts
 * @parem args
 *    the argumtents to be passed to the function
 * @parem iter
 *    iteration count. starts at 0 and goes through maxIter
 */
 
Function.prototype._doRetry = function(context, delay, maxRetrials, args, iter) {
   // retry until success of maxRetrials reached
   try {
      this.apply(this, args);
      if ( isDefined(this.invocationListener) ) {
         this.invocationListener.invocationSucceeded();
      }
   } catch (e) {
      if (iter < maxRetrials) {
         window.setTimeout(this._doRetry.bind(this, context, delay, maxRetrials, args, iter + 1), delay);
      
      } else {
         if ( isDefined(this.invocationListener) ) {
            this.invocationListener.invocationFailed();
         }
	
         // expose the thrown exception to the client
         throw e;
      }
   }
};


/**
 * Invokes the given function and monitors the invocation for success or
 * failure.  If the invocation fails, this will automatically retry a predefined
 * number of times.
 *
 * @param args ...
 *    The arguments to the function invocation
 * @return
 *    false if the first invocation succeeded; true if it failed
 */

Function.prototype.monitoredInvoke = function() {
   // default values
   var delay = 500;
   var maxRetrials = 3;
   var context = self;
   // look for customized settings
   if ( isDefined(this.context) ) {
      context = this.context;
   }
   // perform the function call   
   try {
      this.apply(context, arguments);
      if ( isDefined(this.invocationListener) ) {
         this.invocationListener.invocationSucceeded();
      }
      return false;
   } catch (e) {
      // look for other customized settings
      if ( isDefined(this.delay) ) {
         delay = this.delay;
      }
      if ( isDefined(this.maxRetrials) ) {
         maxRetrials = this.maxRetrials;
      }
      window.setTimeout(this._doRetry.bind(this, context, delay, maxRetrials, arguments, 0), delay);
      return true;
   }
};

