/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * The vpx.ext package extends object prototypes to include useful functionality
 * that is not otherwise available, or that is not supported by all browsers. Adds
 * utility functions to the default scope (i.e., self).
 *
 * @version 1.0 (March 30, 2006)
 */

// Ensure package dependencies are satisfied
if (!window.vpx) {
   throw new Error("vpx.ext: Package dependency failed (vpx)");
}

// Create the package namespace for this and all child packages
vpx.ext = {
   event  : {}
};


/**
 * Determine if both of the supplied objects are equal and of the same type.
 *
 * @param o     any object
 * @param x     any other object
 * @return      true if o is strictly equal to x; false otherwise.
 */

function areStrictlyEqual(o, x) {
   return o === x;
}


/**
 * Determine if the supplied object is null.
 *
 * @param o     any object
 * @return      true if o is null; false otherwise.
 *
 * @xxx         Technically, a value is not null if it is undefined. Is there
 *              any reason to return false here if o is undefined?
 */

function isNull(o) {
   return typeof o == 'undefined' || areStrictlyEqual(o, null);
}


/**
 * Determine if the supplied object is defined.
 *
 * @param o     any object
 * @return      true if o is defined; false otherwise.
 *
 * @xxx         Technically, a value is defined if it is null. Is there any
 *              reason to return true here if o is defined to be null?
 */

function isDefined(o) {
   return ! isNull(o);
}


/**
 * Determine if the supplied object is an array.
 *
 * @param o     any object
 * @return      true if o is an array; false otherwise.
 */

function isArray(o) {
   return isDefined(o) && typeof o == 'object' && o.constructor == Array;
}


/**
 * Determine if the supplied object is a boolean.
 *
 * @param o     any object
 * @return      true if o is a boolean; false otherwise.
 */

function isBoolean(o) {
   return typeof o == 'boolean';
}


/**
 * Determine if the supplied object is a DOM HTML element.
 *
 * @param o     any object
 * @return      true if o is an HTML element; false otherwise.
 */

function isElement(o) {
   return isDefined(o.nodeType);
}


/**
 * Determine if the supplied object is a function.
 *
 * @param o     any object
 * @return      true if o is a function; false otherwise.
 */

function isFunction(o) {
   return typeof o == 'function';
}


/**
 * Determine if the supplied object is a number.
 *
 * @param o     any object
 * @return      true if o is a number; false otherwise.
 */

function isNumber(o) {
   return isFinite(o);
}


/**
 * Determine if the supplied object is an object.
 *
 * @param o     any object
 * @return      true if o is an object; false otherwise.
 */

function isObject(o) {
   return (typeof o == 'object' || isFunction(o));
}


/**
 * Determine if the supplied object is a string.
 *
 * @param o     any object
 * @return      true if o is a string; false otherwise.
 */

function isString(o) {
   if (typeof o == 'object' && o != null) {
      return isDefined(o.valueOf) && typeof o.valueOf() == 'string';
   }

   return typeof o == 'string';
}


/**
 * Determine if the supplied object is a window or frame.
 *
 * @param o     any object
 * @return      true if o is a window object; false otherwise.
 */

function isView(o) {
   return (typeof o == 'Window' ) ||
          ( isDefined(o) && isDefined(o.self) && isDefined(o.window) &&
            (o==o.self) && (o==o.window) );
}


/**
 * Determine if the supplied object is the number zero.
 *
 * @param o     any object
 * @return      true if o is the number zero; false otherwise.
 */

function isZero(o) {
   areStrictlyEqual(o, 0);
}


/**
 * Determine if the supplied object can be cloned safely.
 *
 * @param o     any object
 * @return      true if o can be cloned without too much recursion; false
 *              otherwise.
 */

function isCloneable(o) {
   return (! isNull(o) && ! isView(o) && ! isElement(o) && ! isFunction(o) && isObject(o));
}


/**
 * Return a copy of the supplied object.
 *
 * @param o        any object
 * @param deep     if false or not defined, the new object's properties and
 *                 methods will refer to the cloned object's properties and
 *                 methods rather than copying them.
 * @return         true if o is the number zero; false otherwise.
 */

function clone(o, deep) {
   if (! isCloneable(o)) {
      return o;
   }

   var r = new o.constructor();

   for (var p in o) {
      if (deep) {
         r[p] = clone(o[p], deep);
      } else {
         r[p] = o[p];
      }
   }

   return r;
}


/**
 * Determine if the reference object o is the same type as the supplied
 * constructor.
 *
 * @param o               any object
 * @param constructor     a constructor function
 * @return                true if o's constructor equals constructor
 */

function isTypeOf(o, constructor) {
   return o.constructor === constructor;
}


/**
 * Allows object instances to extend their capabilities dynamically.
 *
 * @param o           the object to extend
 * @param x           an object from which extended properties should be
 *                    referenced. Unless one of the two optional parameters is
 *                    specified, all of o's properties will be used to extend
 *                    the capabilities of this object instance.
 * @param include     an array of property names that should be used to
 *                    extend this object instance. This parameter is optional;
 *                    if it is specified, only the listed properties will be
 *                    used to extend this object instance.
 * @param exclude     an array of property names that should not be used to
 *                    extend this object instance. This parameter is optional;
 *                    if it is specified, the listed properties will not be
 *                    used to extend this object instance. This parameter will
 *                    be ignored if the include parameter is also specified.
 * @return            the object extended
 */

function extendObject(o, x, include, exclude) {
   var i;
   if (isDefined(include)) {
      for (i = 0; i < include.length; i++) {
         var p = include[i];

         if (! isDefined(x[p])) {
            continue;
         }

         o[p] = x[p];
      }
   } else {
      var hash = {};
      if (isDefined(exclude)) {
         for (i = 0; i < exclude.length; i++) {
            hash[exclude[i]] = true;
         }
      }

      for (p in x) {
         if (hash[p]) {
            continue;
         }

         o[p] = x[p];
      }
   }

   return o;
};


/**
 * Compares two objects using logical equality comparison.  The base
 * implementation of this function simply tests true equality.  Subclasses
 * of <code>Object</code> may implement an equals() method to provide their
 * own implementations.
 *
 * @param o     The reference object
 * @param x     The object against which to compare
 * @return      true if the two objects are equal; false otherwise
 */

function arePolyEqual(o, x) {
   if (isDefined(o) && isDefined(o.equals)) {
      return o.equals(x);
   } else {
      return o === x;
   }
}


/**
 * Construct a new Position object.
 *
 * @param x     a point on the x axis
 * @param y     a point on the y axis
 * @param z     a point on the z axis
 * @return      a new Position instance
 */

function Position(x, y, z) {
   this.x = x;
   this.y = y;
   this.z = z;

   this.toString = function() {
      return this.x + ',' + this.y + (z ? ',' + z : '');
   };

   this.add = function() {
      var p;

      if (arguments.length == 1) {
         p = arguments[0];
      } else {
         p = new Position(arguments[0], arguments[1]);
      }

      return new Position(this.x + p.x, this.y + p.y);
   };

   this.subtract = function() {
      var p;

      if (arguments.length == 1) {
         p = arguments[0];
      } else {
         p = new Position(arguments[0], arguments[1]);
      }

      return new Position(this.x - p.x, this.y - p.y);
   };

   this.toDimension = function () {
      return new Dimension(this.x, this.y);
   };
}


/**
 * Construct a new Limit object.
 *
 * @param min     the lower bound
 * @param max     the upper bound
 * @return        a new Limit instance
 */

function Limit(min, max) {
   this.min = min;
   this.max = max;

   this.toString = function () {
      return this.min + " - " + this.max;
   };
}


/**
 * Construct a new Dimension object.
 *
 * @param w     width
 * @param h     height
 * @return      a new Dimension instance
 */

function Dimension(w, h) {
   this.w = w;
   this.h = h;

   this.toString = function() {
      return this.w + 'x' + this.h;
   };

   this.add = function() {
      var d;

      if (arguments.length == 1) {
         d = arguments[0];
      } else {
         d = new Dimension(arguments[0], arguments[1]);
      }

      return new Dimension(this.w + d.w, this.h + d.h);
   };

   this.subtract = function() {
      var d;

      if (arguments.length == 1) {
         d = arguments[0];
      } else {
         d = new Dimension(arguments[0], arguments[1]);
      }

      return new Dimension(this.w - d.w, this.h - d.h);
   };

   this.toPosition = function () {
      return new Position(this.w, this.h);
   };
}
