/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public abstract class AbstractButton
 * extends Component
 * implements ItemSelectable
 *
 * Defines common behaviors for buttons and menu items.
 */

/**
 * Abstract constructor, used to provide common instantiation functionality to
 * concrete children of this class.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the button
 */
vpx.gui.AbstractButton = function(view) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }
   if (!this._concrete) {
      throw new Error("vpx.gui.AbstractButton: cannot instantiate abstract class");
   }

   // super(view)
   vpx.gui.Component.call(this, view);
};

// AbstractButton extends vpx.gui.Component
vpx.gui.AbstractButton.prototype = new vpx.gui.Component(vpx.ABSTRACT_PASS);
vpx.gui.AbstractButton.prototype.constructor = vpx.gui.AbstractButton;

// Shorthand for brevity's sake
var _c = vpx.gui.AbstractButton;     // Class
var _i = _c.prototype;               // Instance
_i._concrete = false;                // vpx system flag for concrete classes

// Instance variables
_i.model                = null;      // protected vpx.gui.ButtonModel
_i.text                 = "";        // private String
_i.defaultIcon          = null;      // private vpx.gui.Icon
_i.pressedIcon          = null;      // private vpx.gui.Icon
_i.disabledIcon         = null;      // private vpx.gui.Icon

_i.selectedIcon         = null;      // private vpx.gui.Icon
_i.disabledSelectedIcon = null;      // private vpx.gui.Icon

_i.rolloverIcon         = null;      // private vpx.gui.Icon
_i.rolloverSelectedIcon = null;      // private vpx.gui.Icon

_i.rolloverEnabled      = false;     // private boolean

_i.iconTextGap          = 4;         // private int

_i.rolloverEnabledSet   = false;     // private boolean
_i.iconTextGapSet       = false;     // private boolean

/**
 * Combined listeners: ActionListener, ChangeListener, ItemListener.  This is
 * used so that we may catch events that out model throws and wrap them in
 * order to mutate the <code>source</code> property to be that of the button
 * (as opposed to the button model).
 */
_i.handler = null;                   // private Handler

/**
 * The button model's <code>changeListener</code>.
 */
_i.changeListener = null;            // protected vpx.core.event.ChangeListener

/**
 * The button model's <code>ActionListener</code>.
 */
_i.actionListener = null;            // protected vpx.gui.event.ActionListener

/**
 * The button model's <code>ItemListener</code>.
 */
_i.itemListener = null;              // protected vpx.gui.event.ItemListener

/**
 * Only one <code>ChangeEvent</code> is needed per button instance since the
 * event's only state is the source property.  The source of events generated
 * is always "this".
 */
_i.changeEvent = null;               // protected transient vpx.core.event.ChangeEvent

/** Identifies a change in the button model. */
_c.MODEL_CHANGED_PROPERTY = "model";

/** Identifies a change in the button's text. */
_c.TEXT_CHANGED_PROPERTY = "text";

// Text positioning and alignment
/** Identifies a change in the button's margins. */
_c.MARGIN_CHANGED_PROPERTY = "margin";
/** Identifies a change in the button's vertical alignment. */
_c.VERTICAL_ALIGNMENT_CHANGED_PROPERTY = "verticalAlignment";
/** Identifies a change in the button's horizontal alignment. */
_c.HORIZONTAL_ALIGNMENT_CHANGED_PROPERTY = "horizontalAlignment";

/** Identifies a change in the button's vertical text position. */
_c.VERTICAL_TEXT_POSITION_CHANGED_PROPERTY = "verticalTextPosition";
/** Identifies a change in the button's horizontal text position. */
_c.HORIZONTAL_TEXT_POSITION_CHANGED_PROPERTY = "horizontalTextPosition";

// Paint options
/**
 * Identifies a change to having the border drawn,
 * or having it not drawn.
 */
_c.BORDER_PAINTED_CHANGED_PROPERTY = "borderPainted";
/**
 * Identifies a change to having the border highlighted when focused,
 * or not.
 */
_c.FOCUS_PAINTED_CHANGED_PROPERTY = "focusPainted";
/**
 * Identifies a change from rollover enabled to disabled or back
 * to enabled.
 */
_c.ROLLOVER_ENABLED_CHANGED_PROPERTY = "rolloverEnabled";
/**
 * Identifies a change to having the button paint the content area.
 */
_c.CONTENT_AREA_FILLED_CHANGED_PROPERTY = "contentAreaFilled";

// Icons
/** Identifies a change to the icon that represents the button. */
_c.ICON_CHANGED_PROPERTY = "icon";

/**
 * Identifies a change to the icon used when the button has been
 * pressed.
 */
_c.PRESSED_ICON_CHANGED_PROPERTY = "pressedIcon";
/**
 * Identifies a change to the icon used when the button has
 * been selected.
 */
_c.SELECTED_ICON_CHANGED_PROPERTY = "selectedIcon";

/**
 * Identifies a change to the icon used when the cursor is over
 * the button.
 */
_c.ROLLOVER_ICON_CHANGED_PROPERTY = "rolloverIcon";
/**
 * Identifies a change to the icon used when the cursor is
 * over the button and it has been selected.
 */
_c.ROLLOVER_SELECTED_ICON_CHANGED_PROPERTY = "rolloverSelectedIcon";

/**
 * Identifies a change to the icon used when the button has
 * been disabled.
 */
_c.DISABLED_ICON_CHANGED_PROPERTY = "disabledIcon";
/**
 * Identifies a change to the icon used when the button has been
 * disabled and selected.
 */
_c.DISABLED_SELECTED_ICON_CHANGED_PROPERTY = "disabledSelectedIcon";

/**
 * Return the default icon.
 *
 * @return vpx.gui.Icon
 *    The default <code>Icon</code>
 */
_i.getIcon = function() {
    return this.defaultIcon;
};

/**
 * Returns the rollover icon for the button.
 *
 * @return vpx.gui.Icon
 *    The <code>rolloverIcon</code> property
 */
_i.getRolloverIcon = function() {
    return this.rolloverIcon;
};

/**
 * Returns the pressed icon for the button.
 *
 * @return vpx.gui.Icon
 *    The <code>pressedIcon</code> property
 */
_i.getPressedIcon = function() {
    return this.pressedIcon;
};

/**
 * Returns the rollover selection icon for the button.
 *
 * @return vpx.gui.Icon
 *    The <code>rolloverSelectedIcon</code> property
 */
_i.getRolloverSelectedIcon = function() {
   return this.rolloverSelectedIcon;
};

/**
 * Returns the selected icon for the button.
 *
 * @return vpx.gui.Icon
 *    The <code>selectedIcon</code> property
 */
_i.getSelectedIcon = function() {
    return this.selectedIcon;
};

/**
 *
 * @param
 *    ...
 * @return vpx.gui.Icon
 *    ...
 */
_i.getDisabledIcon = function() {
   if (this.disabledIcon == null) {
      //?? XXX Use L&F to create dynamic disabled icon from getIcon() (using transparency)
      if (this.disabledIcon != null) {
         this.firePropertyChange(c.DISABLED_ICON_CHANGED_PROPERTY, null, this.disabledIcon);
      }
   }
   return this.disabledIcon;
};

/**
 *
 * @param
 *    ...
 * @return vpx.gui.Icon
 *    ...
 */
_i.getDisabledSelectedIcon = function() {
};

/**
 *
 * @param defaultIcon vpx.gui.Icon
 *    ...
 * @return void
 *    ...
 */
_i.setIcon = function(defaultIcon) {
   var oldValue = this.defaultIcon;
   this.defaultIcon = defaultIcon;

   /* If the default icon has really changed and we had
    * generated the disabled icon for this component,
    * (i.e. setDisabledIcon() was never called) then
    * clear the disabledIcon field.
    */
   //if (defaultIcon != oldValue && (this.disabledIcon instanceof UIResource)) {
   //   this.disabledIcon = null;
   //}

   var c = vpx.gui.AbstractButton;
   if (defaultIcon != oldValue) {
      this.firePropertyChange(c.ICON_CHANGED_PROPERTY, oldValue, defaultIcon);
      this._firePlacementChanged();
   }
};

/**
 *
 * @param rolloverIcon vpx.gui.Icon
 *    ...
 * @return void
 *    ...
 */
_i.setRolloverIcon = function(rolloverIcon) {
    var oldValue = this.rolloverIcon;
    this.rolloverIcon = rolloverIcon;

    var cls = vpx.gui.AbstractButton;
    this.firePropertyChange(cls.ROLLOVER_ICON_CHANGED_PROPERTY, oldValue, rolloverIcon);

    this.setRolloverEnabled(true);
    if (rolloverIcon != oldValue) {
        // No way to determine whether we are currently in
        // a rollover state, so repaint regardless
        //??repaint();
    }
};

/**
 * Sets the pressed icon for the button.
 *
 * @param pressedIcon vpx.gui.Icon
 *    The icon used as the "pressed" image
 */
_i.setPressedIcon = function(pressedIcon) {
    var oldValue = this.pressedIcon;
    this.pressedIcon = pressedIcon;

    var cls = vpx.gui.AbstractButton;
    this.firePropertyChange(cls.PRESSED_ICON_CHANGED_PROPERTY, oldValue, pressedIcon);

    if (pressedIcon != oldValue) {
        if (this.getModel().isPressed()) {
            //??repaint();
        }
    }
};

/**
 *
 * @param rolloverSelectedIcon vpx.gui.Icon
 *    ...
 * @return void
 *    ...
 */
_i.setRolloverSelectedIcon = function(rolloverSelectedIcon) {
};

/**
 * Sets the selected icon for the button.
 *
 * @param selectedIcon vpx.gui.Icon
 *    The icon used as the "selected" image
 */
_i.setSelectedIcon = function(selectedIcon) {
    var oldValue = this.selectedIcon;
    this.selectedIcon = selectedIcon;

    /* If the default selected icon has really changed and we had
     * generated the disabled selected icon for this component,
     * (i.e. setDisabledSelectedIcon() was never called) then
     * clear the disabledSelectedIcon field.
     */
    if (selectedIcon != oldValue && this.disabledSelectedIcon instanceof UIResource) {
        this.disabledSelectedIcon = null;
    }

    var cls = vpx.gui.AbstractButton;
    this.firePropertyChange(cls.SELECTED_ICON_CHANGED_PROPERTY, oldValue, selectedIcon);
    if (selectedIcon != oldValue) {
        if (this.isSelected()) {
            //??repaint();
        }
    }
};

/**
 *
 * @param disabledIcon vpx.gui.Icon
 *    ...
 * @return void
 *    ...
 */
_i.setDisabledIcon = function(disabledIcon) {
   var oldValue = this.disabledIcon;
   this.disabledIcon = disabledIcon;

   var c = vpx.gui.AbstractButton;
   if (disabledIcon != oldValue) {
      this.firePropertyChange(c.DISABLED_ICON_CHANGED_PROPERTY, oldValue, disabledIcon);
   }
};

/**
 *
 * @param disabledSelectedIcon vpx.gui.Icon
 *    ...
 * @return void
 *    ...
 */
_i.setDisabledSelectedIcon = function(disabledSelectedIcon) {
};

/**
 * Returns the button's text.
 *
 * @return String
 *    The button's text
 */
_i.getText = function() {
    return this.text;
};

/**
 * Sets the button's text.
 *
 * @param text String
 *    The string used to set the text
 */
_i.setText = function(text) {
    var oldValue = this.text;
    this.text = text;

    if (text == null || oldValue == null || text != oldValue) {
       var cls = vpx.gui.AbstractButton;
       this.firePropertyChange(cls.TEXT_CHANGED_PROPERTY, oldValue, text);
       this._firePlacementChanged();
    }
};

/**
 *
 * @param
 *    ...
 * @return int
 *    ...
 */
_i.getIconTextGap = function() {
    return this.iconTextGap;
};

/**
 *
 * @param iconTextGap int
 *    ...
 * @return void
 *    ...
 */
_i.setIconTextGap = function(iconTextGap) {
    var oldValue = this.iconTextGap;
    this.iconTextGap = iconTextGap;
    this.iconTextGapSet = true;
    this.firePropertyChange("iconTextGap", oldValue, iconTextGap);
    if (iconTextGap != oldValue) {
        //??revalidate();
        //??repaint();
    }
};

/**
 * Gets the <code>rolloverEnabled</code> property.
 *
 * @return boolean
 *    The value of the <code>rolloverEnabled</code> property
 */
_i.isRolloverEnabled = function() {
    return this.rolloverEnabled;
};

/**
 * Sets the <code>rolloverEnabled</code> property, which must be
 * <code>true</code> for rollover effects to occur. The default value for the
 * <code>rolloverEnabled</code> property is <code>false</code>.
 *
 * @param b boolean
 *    if <code>true</code>, rollover effects should be painted
 */
_i.setRolloverEnabled = function(b) {
    var oldValue = this.rolloverEnabled;
    this.rolloverEnabled = b;
    this.rolloverEnabledSet = true;
    var cls = vpx.gui.AbstractButton;
    this.firePropertyChange(cls.ROLLOVER_ENABLED_CHANGED_PROPERTY, oldValue, this.rolloverEnabled);
    if (b != oldValue) {
        //??repaint();
    }
};

/**
 * Returns the state of the button. True if the toggle button is selected,
 * false if it's not.
 *
 * @return boolean
 *    true if the toggle button is selected, otherwise false
 */
_i.isSelected = function() {
    return this.model.isSelected();
};

/**
 * Sets the state of the button. Note that this method does not trigger an
 * <code>actionEvent</code>. Call <code>doClick</code> to perform a programatic
 * action change.
 *
 * @param b boolean
 *    true if the button is selected, otherwise false
 */
_i.setSelected = function(b) {
   this.model.setSelected(b);
};

/**
 * Programmatically perform a "click". This does the same thing as if the user
 * had pressed and released the button. The button stays visually "pressed" for
 * <code>pressTime</code> milliseconds.
 *
 * @param pressTime int
 *    The time to "hold down" the button, in milliseconds
 */
_i.doClick = function(pressTime) {
   if (pressTime == null) {
      // mimic no-args doClick()
      pressTime = 68;
   }

   this.model.setArmed(true);
   this.model.setPressed(true);

   // Attempt to mimic Thread.currentThread.sleep(pressTime)
   if (pressTime > 0) {
      var onResume = function() {
         this.model.setPressed(false);
         this.model.setArmed(false);
      };
      window.setTimeout(onResume.bind(this), pressTime);
   } else {
      this.model.setPressed(false);
      this.model.setArmed(false);
   }
};

/**
 *
 * @param
 *    ...
 * @return boolean
 *    ...
 */
_i.isEnabled = function() {
   return this.model.isEnabled();
};

/**
 * Enables (or disables) the button.
 *
 * @param b boolean
 *    true to enable the button, otherwise false
 */
_i.setEnabled = function(b) {
   if (!b && this.model.isRollover()) {
      this.model.setRollover(false);
   }
   //??super.setEnabled(b);
   this.model.setEnabled(b);
};

/**
 * Sets the action command for this button.
 *
 * @param actionCommand String
 *    The action command for this button
 */
_i.setActionCommand = function(actionCommand) {
    this.getModel().setActionCommand(actionCommand);
};

/**
 * Returns the action command for this button.
 *
 * @return String
 *    The action command for this button
 */
_i.getActionCommand = function() {
    var ac = this.getModel().getActionCommand();
    if (ac == null) {
        ac = this.getText();
    }
    return ac;
};

/**
 * Returns the model that this button represents.
 *
 * @return vpx.gui.ButtonModel
 *    The <code>model</code> property
 */
_i.getModel = function() {
    return this.model;
};

/**
 * Sets the model that this button represents.
 *
 * @param newModel
 *    The new <code>ButtonModel</code>
 */
_i.setModel = function(newModel) {
   var oldModel = this.getModel();

   if (oldModel != null) {
      oldModel.removeChangeListener(this.changeListener);
      oldModel.removeActionListener(this.actionListener);
      oldModel.removeItemListener(this.itemListener);
      delete this.changeListener;
      delete this.actionListener;
      delete this.itemListener;
   }

   this.model = newModel;

   if (newModel != null) {
      this.changeListener = this._createChangeListener();
      this.actionListener = this._createActionListener();
      this.itemListener = this._createItemListener();
      newModel.addChangeListener(this.changeListener);
      newModel.addActionListener(this.actionListener);
      newModel.addItemListener(this.itemListener);
   }

   var c = vpx.gui.AbstractButton;
   this.firePropertyChange(c.MODEL_CHANGED_PROPERTY, oldModel, newModel);
};

/**
 * Adds an <code>ItemListener</code> to the <code>checkbox</code>.
 *
 * @param l vpx.gui.event.ItemListener
 *    The <code>ItemListener</code> to be added
 */
_i.addItemListener = function(l) {
   this.listenerList.add("ItemListener", l);
};

/**
 * Removes an <code>ItemListener</code> from the button.
 *
 * @param l vpx.gui.event.ItemListener
 *    The <code>ItemListener</code> to be removed
 */
_i.removeItemListener = function(l) {
   this.listenerList.remove("ItemListener", l);
};

/**
 * Returns an array of all the <code>ItemListener</code>s added
 * to this AbstractButton with addItemListener().
 *
 * @return vpx.gui.event.ItemListener[]
 *    All of the <code>ItemListener</code>s added, or an empty array if no
 *    listeners have been added
 */
_i.getItemListeners = function() {
   return this.listenerList.getListeners("ItemListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type. The event instance is lazily created using the
 * <code>event</code> parameter.
 *
 * @param event vpx.gui.event.ItemEvent
 *    The <code>ItemEvent</code> object
 */
_i._fireItemStateChanged = function(event) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();
   var e = null;

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ItemListener") {
         // Lazily create the event
         if (e == null) {
            var ItemEvent = vpx.gui.event.ItemEvent;
            e = new ItemEvent(this, ItemEvent.ITEM_STATE_CHANGED,
                              this, event.getStateChange());
         }

         // listeners[i + 1] is of type vpx.gui.event.ItemListener
         listeners[i + 1].itemStateChanged(e);
      }
   }
};

/**
 * Adds a <code>ChangeListener</code> to the button.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to be added
 */
_i.addChangeListener = function(l) {
   this.listenerList.add("ChangeListener", l);
};

/**
 * Removes a ChangeListener from the button.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to be removed
 */
_i.removeChangeListener = function(l) {
   this.listenerList.remove("ChangeListener", l);
};

/**
 * Returns an array of all the <code>ChangeListener</code>s added
 * to this AbstractButton with addChangeListener().
 *
 * @return vpx.core.event.ChangeListener[]
 *    All of the <code>ChangeListener</code>s added, or an empty array if no
 *    listeners have been added
 */
_i.getChangeListeners = function() {
   return this.listenerList.getListeners("ChangeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is lazily created.
 */
_i._fireStateChanged = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ChangeListener") {
         // Lazily create the event:
         if (this.changeEvent == null) {
            this.changeEvent = new vpx.core.event.ChangeEvent(this);
         }

         // listeners[i + 1] is of type vpx.core.event.ChangeListener
         listeners[i + 1].stateChanged(this.changeEvent);
      }
   }
};

/**
 * Adds an <code>ActionListener</code> to the button.
 *
 * @param l vpx.gui.event.ActionListener
 *    The <code>ActionListener</code> to be added
 */
_i.addActionListener = function(l) {
   this.listenerList.add("ActionListener", l);
};

/**
 * Removes an <code>ActionListener</code> from the button.
 *
 * @param l vpx.gui.event.ActionListener
 *    The listener to be removed
 */
_i.removeActionListener = function(l) {
   this.listenerList.remove("ActionListener", l);
};

/**
 * Returns an array of all the <code>ActionListener</code>s added
 * to this AbstractButton with addActionListener().
 *
 * @return vpx.gui.event.ActionListener[]
 *    All of the <code>ActionListener</code>s added, or an empty array if no
 *    listeners have been added
 */
_i.getActionListeners = function() {
   return this.listenerList.getListeners("ActionListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is created lazily using the
 * <code>event</code> parameter.
 *
 * @param event vpx.gui.event.ActionEvent
 *    The <code>ActionEvent</code> object
 */
_i._fireActionPerformed = function(event) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();
   var e = null;

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ActionListener") {
         // Lazily create the event
         if (e == null) {
            var actionCmd = event.getActionCommand();
            if (actionCmd == null) {
               actionCmd = this.getActionCommand();
            }
            var ActionEvent = vpx.gui.event.ActionEvent;
            e = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, actionCmd);
         }

         // listeners[i + 1] is of type vpx.gui.event.ActionListener
         listeners[i + 1].actionPerformed(e);
      }
   }
};

/**
 * Initializes the button with the specified text and icon.
 *
 * @param text String
 *    The text of the button
 * @param icon vpx.gui.Icon
 *    The icon of the button
 */
_i._init = function(text, icon) {
   if (text != null) {
      this.setText(text);
   }

   if (icon != null) {
      this.setIcon(icon);
   }
};

_i._initUI = function() {
   vpx.gui.Component.prototype._initUI.call(this);
};

/**
 * Subclasses that want to handle <code>ChangeEvents</code> differently can
 * override this to return another <code>ChangeListener</code> implementation.
 *
 * @return vpx.core.event.ChangeListener
 *    The new <code>ChangeListener</code>
 */
_i._createChangeListener = function() {
   return this._getHandler();
};

/**
 * Subclasses that want to handle <code>ActionEvents</code> differently can
 * override this to return another <code>ActionListener</code> implementation.
 *
 * @return vpx.gui.event.ActionListener
 *    The new <code>ActionListener</code>
 */
_i._createActionListener = function() {
   return this._getHandler();
};

/**
 * Subclasses that want to handle <code>ItemEvents</code> differently can
 * override this to return another <code>ItemListener</code> implementation.
 *
 * @return vpx.gui.event.ItemListener
 *    The new <code>ItemListener</code>
 */
_i._createItemListener = function() {
   return this._getHandler();
};

/**
 * class Handler
 * extends Object
 * implements vpx.gui.event.ActionListener, vpx.core.event.ChangeListener,
 *            vpx.gui.event.ItemListener
 *
 * Listeners that are automatically added to button model.
 *
 * @version 1.0 (Oct 17, 2005)
 */

/**
 * Constructs a new Handler.
 *
 * @param button vpx.gui.AbstractButton
 *    The friend button that this handler represents
 */
_c.Handler = function(button) {
   this.button = button;
};

/**
 * <code>ChangeListener</code> implementation.
 *
 * @param e vpx.core.event.ChangeEvent
 *    The <code>ChangeEvent</code>
 */
_c.Handler.prototype.stateChanged = function(e) {
   this.button._fireStateChanged();
};

/**
 * <code>ActionListener</code> implementation.
 *
 * @param e vpx.gui.event.ActionEvent
 *    The <code>ActionEvent</code>
 */
_c.Handler.prototype.actionPerformed = function(e) {
   this.button._fireActionPerformed(e);
};

/**
 * <code>ItemListener</code> implementation.
 *
 * @param e vpx.gui.event.ItemEvent
 *    The <code>ItemEvent</code>
 */
_c.Handler.prototype.itemStateChanged = function(e) {
   this.button._fireItemStateChanged(e);
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Gets the internal combined listener (ActionListener, ChangeListener,
 * and ItemListener).
 *
 * @return vpx.gui.AbstractButton.Handler
 *    The combined listener
 */
_i._getHandler = function() {
   if (this.handler == null) {
      this.handler = new vpx.gui.AbstractButton.Handler(this);
   }
   return this.handler;
};
