/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * public class ButtonGroup
 * extends Object
 *
 * This class is used to create a multiple-exclusion scope for a set of
 * buttons. Creating a set of buttons with the same <code>ButtonGroup</code>
 * object means that turning "on" one of those buttons turns off all other
 * buttons in the group.
 * <p/>
 * A <code>ButtonGroup</code> can be used with any set of objects that inherit
 * from <code>AbstractButton</code>. Typically a button group contains
 * instances of <code>RadioButton</code>, <code>RadioButtonMenuItem</code>,
 * or <code>ToggleButton</code>. It wouldn't make sense to put an instance of
 * <code>MenuItem</code> in a button group because <code>MenuItem</code>
 * doesn't implement the selected state.
 * <p/>
 * Initially, all buttons in the group are unselected. Once any button is
 * selected, one button is always selected in the group. There is no way to
 * turn a button programmatically to "off", in order to clear the button group.
 * To give the appearance of "none selected", add an invisible radio button to
 * the group and then programmatically select that button to turn off all the
 * displayed radio buttons. For example, a normal button with the label "none"
 * could be wired to select the invisible radio button.
 * <p/>
 * VPX GUI button groups are based on Java Swing button groups. For examples
 * and further information on using Swing button groups, see
 * <a href="http://java.sun.com/docs/books/tutorial/uiswing/components/button.html#radiobutton">How to Use Radio Buttons</a>,
 * a section in <em>The Java Tutorial</em>.
 *
 * @version 1.0 (Oct 14, 2005)
 */

/**
 * Creates a new <code>ButtonGroup</code>.
 */
vpx.gui.ButtonGroup = function() {
   this.buttons = [];
};

// Shorthand for brevity's sake
var _c = vpx.gui.ButtonGroup;         // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.buttons = null;                    // protected vpx.gui.AbstractButton[]
_i.selection = null;                  // protected vpx.gui.ButtonModel

/**
 * Adds the button to the group.
 *
 * @param b vpx.gui.AbstractButton
 *    The button to be added
 */
_i.add = function(b) {
   if (b == null) {
      return;
   }
   this.buttons.push(b);

   if (b.isSelected()) {
      if (this.selection == null) {
         this.selection = b.getModel();
      } else {
         b.setSelected(false);
      }
   }

   b.getModel().setGroup(this);
};

/**
 * Removes the button from the group.
 *
 * @param b vpx.gui.AbstractButton
 *    The button to be removed
 */
_i.remove = function(b) {
   if (b == null) {
      return;
   }
   var i = this.buttons.indexOf(b);
   if (i >= 0) {
      this.buttons.splice(i, 1);
   }
   if(b.getModel() == this.selection) {
      delete this.selection;
      this.selection = null;
   }
   b.getModel().setGroup(null);
};

/**
 * Returns all the buttons that are participating in this group.
 *
 * @return vpx.gui.AbstractButton[]
 *    An array of the buttons in this group
 */
_i.getElements = function() {
   var res = new Array(this.buttons.length);
   for (var i = 0; i < this.buttons.length; i++) {
      res[i] = this.buttons[i];
   }
   return res;
};

/**
 * Returns the model of the selected button.
 *
 * @return vpx.gui.ButtonModel
 *    The selected button model
 */
_i.getSelection = function() {
   return this.selection;
};

/**
 * Sets the selected value for the <code>ButtonModel</code>. Only one button
 * in the group may be selected at a time.
 *
 * @param m vpx.gui.ButtonModel
 *    The <code>ButtonModel</code>
 * @param b boolean
 *    <code>true</code> if this button is to be selected, otherwise
 *    <code>false</code>
 */
_i.setSelected = function(m, b) {
   if (b && m != null && m != this.selection) {
      var oldSelection = this.selection;
      this.selection = m;
      if (oldSelection != null) {
         oldSelection.setSelected(false);
      }
      m.setSelected(true);
   }
};

/**
 * Returns whether a <code>ButtonModel</code> is selected.
 *
 * @param m vpx.gui.ButtonModel
 *    The <code>ButtonModel</code> to test
 * @return boolean
 *    <code>true</code> if the button is selected, otherwise returns
 *    <code>false</code>
 */
_i.isSelected = function(m) {
   return (m == this.selection);
};

/**
 * Returns the number of buttons in the group.
 *
 * @return int
 *    The button count
 */
_i.getButtonCount = function() {
   if (this.buttons == null) {
      return 0;
   }
   return this.buttons.length;
};
