/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * package vpx.gui
 * public interface ButtonModel
 * extends ItemSelectable
 *
 * State Model for buttons. This model is used for check boxes and radio
 * buttons, which are special kinds of buttons, as well as for normal buttons.
 * For check boxes and radio buttons, pressing the mouse selects the button.
 * For normal buttons, pressing the mouse "arms" the button. Releasing the
 * mouse over the button then initiates a <i>button</i> press, firing its
 * action event. Releasing the mouse elsewhere disarms the button.
 * <p/>
 * In use, a UI will invoke <code>setSelected(boolean)</code> when a mouse
 * click occurs over a check box or radio button. It will invoke
 * <code>setArmed(boolean)</code> when the mouse is pressed over a regular
 * button and invoke <code>setPressed(boolean)</code> when the mouse is
 * released. If the mouse travels outside the button in the meantime,
 * <code>setArmed(false)</code> will tell the button not to fire when it sees
 * <code>setPressed</code>. (If the mouse travels back in, the button will be
 * rearmed.)
 *
 * @version 1.0 (Sept 27, 2005)
 */

// This file simply defines an interface. It should not be loaded into HTML
throw new Error("Interface vpx.gui.ButtonModel cannot be instantiated");

/**
 * Indicates partial commitment towards pressing the button.
 *
 * @return boolean
 *    true if the button is armed and ready to be pressed
 */
vpx.gui.ButtonModel.prototype.isArmed = function() {};

/**
 * Indicates if the button has been selected. Only needed for certain types of
 * buttons - such as radio buttons and check boxes.
 *
 * @return boolean
 *    true if the button is selected
 */
vpx.gui.ButtonModel.prototype.isSelected = function() {};

/**
 * Indicates if the button can be selected or pressed by an input device
 * (such as a mouse pointer). (Check boxes are selected, regular buttons are
 * "pressed".)
 *
 * @return boolean
 *    true if the button is enabled, and therefore selectable (or pressable)
 */
vpx.gui.ButtonModel.prototype.isEnabled = function() {};

/**
 * Indicates if button has been pressed.
 *
 * @return boolean
 *    true if the button has been pressed
 */
vpx.gui.ButtonModel.prototype.isPressed = function() {};

/**
 * Indicates that the mouse is over the button.
 *
 * @return boolean
 *    true if the mouse is over the button
 */
vpx.gui.ButtonModel.prototype.isRollover = function() {};

/**
 * Marks the button as "armed". If the mouse button is released while it is
 * over this item, the button's action event fires. If the mouse button is
 * released elsewhere, the event does not fire and the button is disarmed.
 *
 * @param b boolean
 *    true to arm the button so it can be selected
 */
vpx.gui.ButtonModel.prototype.setArmed = function(b) {};

/**
 * Selects or deselects the button.
 *
 * @param b boolean
 *    true selects the button; false deselects the button
 */
vpx.gui.ButtonModel.prototype.setSelected = function(b) {};

/**
 * Enables or disables the button.
 *
 * @param b boolean
 *    true to enable the button
 */
vpx.gui.ButtonModel.prototype.setEnabled = function(b) {};

/**
 * Sets the button to pressed or unpressed.
 *
 * @param b boolean
 *    true to set the button to "pressed"
 */
vpx.gui.ButtonModel.prototype.setPressed = function(b) {};

/**
 * Sets or clears the button's rollover state
 *
 * @param b boolean
 *    true to turn on rollover
 */
vpx.gui.ButtonModel.prototype.setRollover = function(b) {};

/**
 * Sets the actionCommand string that gets sent as part of the event when the
 * button is pressed.
 *
 * @param s String
 *    The String that identifies the generated event
 */
vpx.gui.ButtonModel.prototype.setActionCommand = function(s) {};

/**
 * Returns the action command for this button.
 *
 * @return String
 *    The String that identifies the generated event
 */
vpx.gui.ButtonModel.prototype.getActionCommand = function() {};

/**
 * Identifies the group this button belongs to -- needed for radio buttons,
 * which are mutually exclusive within their group.
 *
 * @param group vpx.gui.ButtonGroup
 *    The ButtonGroup this button belongs to
 */
vpx.gui.ButtonModel.prototype.setGroup = function(group) {};

/**
 * Adds an ActionListener to the button.
 *
 * @param l vpx.gui.event.ActionListener
 *    The listener to add
 */
vpx.gui.ButtonModel.prototype.addActionListener = function(l) {};

/**
 * Removes an ActionListener from the button.
 *
 * @param l vpx.gui.event.ActionListener
 *    The listener to remove
 */
vpx.gui.ButtonModel.prototype.removeActionListener = function(l) {};

/**
 * Adds an ItemListener to the button.
 *
 * @param l vpx.gui.event.ItemListener
 *    The listener to add
 */
vpx.gui.ButtonModel.prototype.addItemListener = function(l) {};

/**
 * Removes an ItemListener from the button.
 *
 * @param l vpx.gui.event.ItemListener
 *    The listener to remove
 */
vpx.gui.ButtonModel.prototype.removeItemListener = function(l) {};

/**
 * Adds a ChangeListener to the button.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to add
 */
vpx.gui.ButtonModel.prototype.addChangeListener = function(l) {};

/**
 * Removes a ChangeListener from the button.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to remove
 */
vpx.gui.ButtonModel.prototype.removeChangeListener = function(l) {};
