/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Abstract constructor, used to provide common instantiation functionality to
 * concrete children of this class.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the component
 */
vpx.gui.Component = function(view) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }
   if (!this._concrete) {
      throw new Error("vpx.gui.Component: cannot instantiate abstract class");
   }

   this.view = view;
   this.listenerList = new vpx.core.event.EventListenerList();
   this.component = [];
};

// Shorthand for brevity's sake
var _c = vpx.gui.Component;           // Class
var _i = _c.prototype;                // Instance
_i._concrete = false;                 // vpx system flag for concrete classes

// Instance variables
_i.view = null;                       // protected [DOM Level 2 Views]AbstractView
_i.changeSupport = null;              // private vpx.gui.event.PropertyChangeSupport
_i.ui = null;                         // protected vpx.gui.plaf.ComponentUI
_i.parent = null;                     // private vpx.gui.Component
_i.visible = true;                    // protected boolean
_i.listenerList  = null;              // protected vpx.core.event.EventListenerList
_i.ncomponents = 0;                   // private int
_i.component = null;                  // private vpx.gui.Component[]
_i.boundHandlers = null;              // private transient Object
_i.componentEvent = null;             // private vpx.gui.event.ComponentEvent
_i.prefSize = null;                   // private vpx.gui.Dimension
_i.prefSizeSet = false;               // private boolean
_i.attrs = null;                      // private vpx.core.HashMap

// Class constants

/** Identifies a change in the component's visibility. */
_c.VISIBILITY_PROPERTY = "visibility";

/** Identifies a change in the component's preferred size. */
_c.PREF_SIZE_PROPERTY = "preferredSize";

/**
 * Gets the context in which this component was created. This context will
 * determine where this component's native peer will reside.
 *
 * @return [DOM Level 2 Views]AbstractView
 *    The context in which this component was created
 */
_i.getView = function() {
   return this.view;
};

/**
 * Gets the parent of this component.
 *
 * @return vpx.gui.Component
 *    The parent container of this component
 */
_i.getParent = function() {
   return this.parent;
};

/**
 * Gets the number of components in this container.
 *
 * @return int
 *    Te number of components in this panel.
 */
_i.getComponentCount = function() {
   return this.ncomponents;
};

/**
 * Gets the nth component in this container.
 *
 * @param n int
 *    The index of the component to get.
 * @return vpx.gui.Component
 *    The n<sup>th</sup> component in this container.
 * @exception Error
 *    If the n<sup>th</sup> value does not exist.
 */
_i.getComponentAt = function(n) {
   if ((n < 0) || (n >= this.ncomponents)) {
      throw new Error("vpx.gui.Component#getComponentAt(): no such child: " + n);
   }
   return this.component[n];
};

/**
 * Gets all the components in this container.
 *
 * @return vpx.gui.Component[]
 *    An array of all the components in this container
 */
_i.getComponents = function() {
   var a = new Array(this.ncomponents);
   for (var i = 0; i < this.ncomponents; i++) {
      a[i] = this.component[i];
   }
   return a;
};

/**
 * Appends the specified component to the end of this container component.
 *
 * @param c vpx.gui.Component
 *    The component to be added
 * @return vpx.gui.Component
 *    The component argument
 */
_i.add = function(c) {
   this.ui.appendToPeer(c);

   // Reparent the component
   if (c.parent != null) {
      c.parent.remove(c);
   }

   // Add component to the list
   this.component[this.ncomponents++] = c;
   c.parent = this;
   c._firePlacementChanged();

   if (this.listenerList.getListenerCountByType("ContainerListener") > 0) {
      // No need to create event if no one's interested
      var cls = vpx.gui.event.ContainerEvent;
      var e = new cls(this, cls.COMPONENT_ADDED, c);
      this._fireContentsChanged(e);
   }
};

/**
 * Removes the specified component from this container.
 *
 * @param c vpx.gui.Component
 *    The component to be removed
 */
_i.remove = function(c) {
   if (c.parent == this)  {
      /* Search backwards, expect that more recent additions
       * are more likely to be removed.
       */
      for (var i = this.ncomponents - 1; i >= 0; i--) {
         if (this.component[i] == c) {
            this.removeAt(i);
            c._firePlacementChanged();
         }
      }
   }
};

/**
 *
 */
_i.removeAll = function() {
   var nlstnrs = this.listenerList.getListenerCountByType("ContainerListener");

   while (this.ncomponents > 0) {
      var c = this.component.pop();
      this.ncomponents--;

      this.ui.removeFromPeer(c);

      c.parent = null;
      c._firePlacementChanged();

      if (nlstnrs > 0) {
         // No need to create event if no one's interested
         var cls = vpx.gui.event.ContainerEvent;
         var e = new cls(this, cls.COMPONENT_REMOVED, c);
         this._fireContentsChanged(e);
      }
   }
};

/**
 * Removes the component, specified by <code>i</code>, from this container.
 *
 * @param i int
 *    The index of the component to be removed
 */
_i.removeAt = function(i) {
   if (i < 0  || i >= this.ncomponents) {
      throw new Error("vpx.gui.Component#removeAt(): index out of bounds: " + i);
   }
   var c = this.component[i];

   this.ui.removeFromPeer(c);

   c.parent = null;
   c._firePlacementChanged();
   this.component.splice(i, 1);
   this.ncomponents--;

   if (this.listenerList.getListenerCountByType("ContainerListener") > 0) {
      // No need to create event if no one's interested
      var cls = vpx.gui.event.ContainerEvent;
      var e = new cls(this, cls.COMPONENT_REMOVED, c);
      this._fireContentsChanged(e);
   }
};

/**
 * Gets this component's ui delegate.
 *
 * @return vpx.gui.plaf.ComponentUI
 *    The ui delegate for this component
 */
_i.getUI = function() {
   return this.ui;
};

/**
 * Determines whether this component should be visible when its parent is
 * visible. Components are initially visible.
 *
 * @return boolean
 *    <code>true</code> if the component is visible, <code>false</code>
 *    otherwise
 */
_i.isVisible = function() {
   return this.visible;
};

/**
 * Shows or hides this component depending on the value of parameter
 * <code>b</code>.
 *
 * @param b boolean
 *    If <code>true</code>, shows this component; otherwise, hides this
 *    component
 */
_i.setVisible = function(b) {
   if (b == this.visible) {
      // No-op
      return;
   }

   this.visible = b;
   this.firePropertyChange(vpx.gui.Component.VISIBILITY_PROPERTY, !b, b);
};

/**
 * Determines whether this component is showing on screen. This means that the
 * component must be visible, and it must be in a container that is visible and
 * showing.
 *
 * @return boolean
 *    <code>true</code> if the component is showing, <code>false</code>
 *    otherwise
 */
_i.isShowing = function() {
   var peer = this.getPeer();
   if (this.visible && (peer != null)) {
      if (this.parent == null) {
         return (peer.parentNode != null);
      } else {
         return this.parent.isShowing();
      }
   }
   return false;
};

/**
 * Sets the preferred size of this component to a constant value.  Subsequent
 * calls to <code>getPreferredSize</code> will always return this value.
 * Setting the preferred size to <code>null</code> restores the default
 * behavior.
 *
 * @param s vpx.gui.Dimension
 *    The new preferred size, or null
 */
_i.setPreferredSize = function(s) {
   var old;
   // If the preferred size was set, use it as the old value, otherwise
   // use null to indicate we didn't previously have a set preferred
   // size.
   if (this.prefSizeSet) {
      old = this.prefSize;
   } else {
      old = null;
   }
   this.prefSize = s;
   this.prefSizeSet = (s != null);
   this.firePropertyChange(vpx.gui.Component.PREF_SIZE_PROPERTY, old, s);
};

/**
 * Returns true if the preferred size has been set to a non-<code>null</code>
 * value; otherwise returns false.
 *
 * @return boolean
 *    true if <code>setPreferredSize</code> has been invoked with a non-null
 *    value.
 */
_i.isPreferredSizeSet = function() {
   return this.prefSizeSet;
};

/**
 * Gets the preferred size of this component.
 *
 * @return vpx.gui.Dimension
 *    A dimension object indicating this component's preferred size
 */
_i.getPreferredSize = function() {
   var dim = this.prefSize;
   if (dim == null || !this.isPreferredSizeSet()) {
      //?? XXX Ask peer for preferred size?
      //?? this.prefSize = (this.peer != null) ? this.peer.preferredSize() : ...
      dim = this.prefSize;
   }
   return new vpx.gui.Dimension(dim);
};

/**
 * Adds the specified container listener to receive container events from this
 * component. If l is null, no exception is thrown, and no action is performed.
 *
 * @param l vpx.gui.event.ContainerListener
 *    The container listener
 */
_i.addContainerListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.add("ContainerListener", l);
};

/**
 * Removes the specified container listener so it no longer receives container
 * events from this component. If l is null, no exception is thrown, and no
 * action is performed.
 *
 * @param l vpx.gui.event.ContainerListener
 *    The container listener
 */
_i.removeContainerListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.remove("ContainerListener", l);
};

/**
 * Returns an array of all the container listeners registered on this component.
 *
 * @return vpx.gui.event.ContainerListener[]
 *    All of this container's <code>ContainerListener</code>s or an empty array
 *    if no container listeners are currently registered
 */
_i.getContainerListeners = function() {
   return this.listenerList.getListeners("ContainerListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.
 *
 * @param e vpx.gui.event.ContainerEvent
 *    The <code>ContainerEvent</code> to deliver to listeners
 */
_i._fireContentsChanged = function(e) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   var id = e.getId();
   var cls = vpx.gui.event.ContainerEvent;

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ContainerListener") {
         // listeners[i + 1] is of type vpx.gui.event.ContainerListener
         if (id == cls.COMPONENT_ADDED) {
            listeners[i + 1].componentAdded(e);
         } else if (id == cls.COMPONENT_REMOVED) {
            listeners[i + 1].componentRemoved(e);
         } else {
            throw new Error("vpx.gui.Component#_fireContentsChanged(): invalid id: " + id);
         }
      }
   }
};

/**
 * Adds the specified component listener to receive component events from this
 * component. If l is null, no exception is thrown, and no action is performed.
 *
 * @param l vpx.gui.event.ComponentListener
 *    The component listener
 */
_i.addComponentListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.add("ComponentListener", l);
};

/**
 * Removes the specified component listener so it no longer receives component
 * events from this component. If l is null, no exception is thrown, and no
 * action is performed.
 *
 * @param l vpx.gui.event.ComponentListener
 *    The component listener
 */
_i.removeComponentListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.remove("ComponentListener", l);
};

/**
 * Returns an array of all the component listeners registered on this component.
 *
 * @return vpx.gui.event.ComponentListener[]
 *    All of this component's <code>ComponentListener</code>s or an empty array
 *    if no component listeners are currently registered
 */
_i.getComponentListeners = function() {
   return this.listenerList.getListeners("ComponentListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The component event is created lazily.
 */
_i._firePlacementChanged = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ComponentListener") {
         if (this.componentEvent == null) {
            this.componentEvent = new vpx.gui.event.ComponentEvent(this);
         }

         // listeners[i + 1] is of type vpx.gui.event.ComponentListener
         listeners[i + 1].placementChanged(this.componentEvent);
      }
   }
};

/**
 * Adds the specified mouse listener to receive mouse events from this
 * component. If l is null, no exception is thrown, and no action is performed.
 *
 * @param l vpx.gui.event.MouseListener
 *    The mouse listener
 */
_i.addMouseListener = function(l) {
   if (l == null) {
      return;
   }

   var peer = this.ui.getClickablePeer();
   if (peer == null) {
      throw new Error("vpx.gui.Component#addMouseListener(): no peer");
   }

   if (this.boundHandlers == null) {
      this._createBoundHandlers();
   }

   if (this.listenerList.getListenerCountByType("MouseListener") == 0) {
      var handler = this.boundHandlers.mouseAction;
      vpx.xua.event.listen(peer, "click", handler);
      vpx.xua.event.listen(peer, "mousedown", handler);
      vpx.xua.event.listen(peer, "mouseup", handler);
      vpx.xua.event.listen(peer, "mouseover", handler);
      vpx.xua.event.listen(peer, "mouseout", handler);
   }

   this.listenerList.add("MouseListener", l);
};

/**
 * Removes the specified mouse listener so it no longer receives mouse
 * events from this component. If l is null, no exception is thrown, and no
 * action is performed.
 *
 * @param l vpx.gui.event.MouseListener
 *    The mouse listener
 */
_i.removeMouseListener = function(l) {
   if (l == null) {
      return;
   }

   this.listenerList.remove("MouseListener", l);

   if (this.listenerList.getListenerCountByType("MouseListener") == 0) {
      var peer = this.ui.getClickablePeer();
      var handler = this.boundHandlers.mouseAction;
      vpx.xua.event.ignore(peer, "click", handler);
      vpx.xua.event.ignore(peer, "mousedown", handler);
      vpx.xua.event.ignore(peer, "mouseup", handler);
      vpx.xua.event.ignore(peer, "mouseover", handler);
      vpx.xua.event.ignore(peer, "mouseout", handler);
   }
};

/**
 * Returns an array of all the mouse listeners registered on this component.
 *
 * @return vpx.gui.event.MouseListener[]
 *    All of this component's <code>MouseListener</code>s or an empty array
 *    if no mouse listeners are currently registered
 */
_i.getMouseListeners = function() {
   return this.listenerList.getListeners("MouseListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.
 *
 * @param e [DOM Level 2 Events]MouseEvent
 *    The native DOM <code>MouseEvent</code>. This event will be converted to a
 *    <code>vpx.gui.event.MouseEvent</code> object before being passed to
 *    any listeners
 */
_i._fireMouseActionPerformed = function(e) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Wrap the native event in a lightweight event
   var c = vpx.gui.event.MouseEvent;
   var event = c.fromDOMEvent(this, e);

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "MouseListener") {
         // listeners[i + 1] is of type vpx.gui.event.MouseListener
         var l = listeners[i + 1];
         switch (event.getId()) {
         case c.MOUSE_CLICKED:
            l.mouseClicked(event);
            break;
         case c.MOUSE_PRESSED:
            l.mousePressed(event);
            break;
         case c.MOUSE_RELEASED:
            l.mouseReleased(event);
            break;
         case c.MOUSE_ENTERED:
            l.mouseEntered(event);
            break;
         case c.MOUSE_EXITED:
            l.mouseExited(event);
            break;
         default:
            throw new Error("Component#_fireMouseActionPerformed(): Unknown event id");
            break;
         }
      }
   }

   delete event;
   vpx.xua.event.cancel(e);
   return false;
};

/**
 * Adds a PropertyChangeListener to the listener list. The listener is
 * registered for all bound properties of this class, including the following:
 * <ul>
 *    <li>this Component's font ("font")</li>
 *    <li>this Component's background color ("background")</li>
 *    <li>this Component's foreground color ("foreground")</li>
 *    <li>this Component's focusability ("focusable")</li>
 *    <li>this Component's preferred size ("preferredSize")</li>
 *    <li>this Component's minimum size ("minimumSize")</li>
 *    <li>this Component's maximum size ("maximumSize")</li>
 *    <li>this Component's name ("name")</li>
 * </ul>
 * Note that if this <code>Component</code> is inheriting a bound property,
 * then no event will be fired in response to a change in the inherited
 * property.
 * <p/>
 * If <code>listener</code> is <code>null</code>, no exception is thrown, and
 * no action is performed.
 *
 * @param l vpx.gui.event.PropertyChangeListener
 *    The property change listener to be added
 */
_i.addGlobalPropertyChangeListener = function(l) {
   if (!l) {
      return;
   }
   if (this.changeSupport == null) {
      this.changeSupport = new vpx.gui.event.PropertyChangeSupport(this);
   }
   this.changeSupport.addGlobalPropertyChangeListener(l);
};

/**
 * Removes a PropertyChangeListener from the listener list. This method should
 * be used to remove PropertyChangeListeners that were registered for all
 * bound properties of this class.
 * <p/>
 * If listener is null, no exception is thrown, and no action is performed.
 *
 * @param l vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be removed
 */
_i.removeGlobalPropertyChangeListener = function(l) {
   if (!l || this.changeSupport == null) {
      return;
   }
   this.changeSupport.removeGlobalPropertyChangeListener(l);
};

/**
 * Returns an array of all the property change listeners registered on this
 * component.
 *
 * @return vpx.gui.event.PropertyChangeListener[]
 *    All of this component's <code>PropertyChangeListener</code>s, or an empty
 *    array if no property change listeners are currently registered
 */
_i.getGlobalPropertyChangeListeners = function() {
   if (this.changeSupport == null) {
      return [];
   }
   return this.changeSupport.getGlobalPropertyChangeListeners();
};

/**
 * Adds a PropertyChangeListener to the listener list for a specific property.
 * The specified property may be user-defined, or one of the following:
 * <ul>
 *    <li>this Component's font ("font")</li>
 *    <li>this Component's background color ("background")</li>
 *    <li>this Component's foreground color ("foreground")</li>
 *    <li>this Component's focusability ("focusable")</li>
 *    <li>this Component's preferred size ("preferredSize")</li>
 *    <li>this Component's minimum size ("minimumSize")</li>
 *    <li>this Component's maximum size ("maximumSize")</li>
 *    <li>this Component's name ("name")</li>
 * </ul>
 * Note that if this <code>Component</code> is inheriting a bound property,
 * then no event will be fired in response to a change in the inherited
 * property.
 * <p/>
 * If <code>propertyName</code> or <code>listener</code> is <code>null</code>,
 * no exception is thrown, and no action is taken.
 *
 * @param property String
 *    One of the property names listed above
 * @param l vpx.gui.event.PropertyChangeListener
 *    The property change listener to be added
 */
_i.addSpecificPropertyChangeListener = function(property, l) {
   if (!property || !l) {
      return;
   }
   if (this.changeSupport == null) {
      this.changeSupport = new vpx.gui.event.PropertyChangeSupport(this);
   }
   this.changeSupport.addSpecificPropertyChangeListener(property, l);
};

/**
 * Removes a <code>PropertyChangeListener</code> from the listener list for a
 * specific property. This method should be used to remove
 * <code>PropertyChangeListener</code>s that were registered for a specific
 * bound property.
 * <p/>
 * If <code>propertyName</code> or <code>listener</code> is <code>null</code>,
 * no exception is thrown, and no action is taken.
 *
 * @param property String
 *    A valid property name
 * @param l vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be removed
 */
_i.removeSpecificPropertyChangeListener = function(property, l) {
   if (!property || !l || this.changeSupport == null) {
      return;
   }
   this.changeSupport.removeSpecificPropertyChangeListener(property, l);
};

/**
 * Returns an array of all the listeners which have been associated with the
 * named property.
 *
 * @param property String
 *    A valid property name
 * @return vpx.gui.event.PropertyChangeListener[]
 *    All of the <code>PropertyChangeListener</code>s associated with the named
 *    property; if no such listeners have been added or if
 *    <code>property</code> is <code>null</code>, an empty array is
 *    returned
 */
_i.getSpecificPropertyChangeListeners = function(property) {
   if (!property || this.changeSupport == null) {
      return [];
   }
   return this.changeSupport.getSpecificPropertyChangeListeners(property);
};

/**
 * Support for reporting bound property changes for Object properties.  This
 * method can be called when a bound property has changed, and it will send the
 * appropriate PropertyChangeEvent to any registered PropertyChangeListeners.
 *
 * @param property String
 *    The property whose value has changed
 * @param oldValue Object
 *    The property's previous value
 * @param newValue Object
 *    The property's new value
 */
_i.firePropertyChange = function(property, oldValue, newValue) {
   if (this.changeSupport == null || (oldValue && newValue && arePolyEqual(oldValue, newValue))) {
      return;
   }
   this.changeSupport.firePropertyChange(property, oldValue, newValue);
};

/**
 * Gets the native peer of the component. The peer implements the component's
 * behavior.
 *
 * @return HTMLElement
 *    The native peer of the component
 */
_i.getPeer = function() {
   return this.ui.getPeer();
};

/**
 * Gets the location of this component within the context of the given view.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context of the desired location
 * @return vpx.gui.Point
 *    The location of this component relative to the given view, or
 *    <code>null</code> if this component has no peer
 */
_i.getLocation = function(view) {
   var peer = this.getPeer();

   if (peer == null) {
      return null;
   }

   var TRANS_X = 1;
   var TRANS_Y = 2;
   var p = new vpx.gui.Point(0, 0);

   p.translate(vpx.xua.getX(peer), vpx.xua.getY(peer));

   var v = vpx.xua.getView(peer);
   if (v == null) {
      //?? XXX What does this mean?
      return p;
   }

   for (; v != v.top && v != view; v = v.parent) {
      var frm = v.frameElement;
      if (frm) {
         p.translate(vpx.xua.getX(frm), vpx.xua.getY(frm));

         var add = [
            frm.style.borderLeftWidth, TRANS_X,
            frm.style.borderTopWidth, TRANS_Y
         ];

         for (var i = add.length - 2; i >= 0; i -= 2) {
            if (add[i] != null && add[i] != "") {
               var match = add[i].match(/[0-9]+/);
               if (match != null) {
                  switch (add[i + 1]) {
                  case TRANS_X:
                     p.translate(parseInt(match[0]), 0);
                     break;
                  case TRANS_Y:
                     p.translate(0, parseInt(match[0]));
                     break;
                  default:
                     throw new Error("Component#getLocation(): bad translation array");
                     break;
                  }
               }
            }
         }
      }
   }

   if (v != view) {
      throw new Error("Component#getLocation(): view not in component hierarchy");
   }

   return p;
};

/**
 * Returns the object bound with the specified name in this component, or
 * <code>null</code> if no object is bound under the name.
 *
 * @param name String
 *    A string specifying the name of the object
 * @return Object
 *    The object with the specified name
 */
_i.getAttribute = function(name) {
   if (this.attrs == null) {
      return null;
   }
   return this.attrs.get(name);
};

/**
 * Returns an <code>Array</code> of <code>String</code> objects containing the
 * names of all the objects bound to this component.
 *
 * @return String[]
 *    An array of String objects specifying the names of all the objects bound
 *    to this component
 */
_i.getAttributeNames = function() {
   if (this.attrs == null) {
      return [];
   }
   return this.attrs.keyArray();
};

/**
 * Binds an object to this component, using the name specified. If an object of
 * the same name is already bound to the component, the object is replaced.
 * <p/>
 * After this method executes, the component then notifies any
 * <code>ComponentAttributeListener</code>s that have registered interest on
 * this component.
 * <p/>
 * If the value passed in is null, this has the same effect as calling
 * <code>removeAttribute()</code>.
 *
 * @param name String
 *    The name to which the object is bound; cannot be null
 * @param value Object
 *    The object to be bound
 */
_i.setAttribute = function(name, value) {
   if (this.attrs == null) {
      // Delay object instantiation
      this.attrs = new vpx.core.HashMap();
   }

   var e = null;
   if (this.listenerList.getListenerCountByType("ComponentAttributeListener") > 0) {
      // No need to create event if no one's interested
      var c = vpx.gui.event.ComponentBindingEvent;
      var eId, eVal;
      if (this.attrs.containsKey(name)) {
         eId = c.ATTRIBUTE_REPLACED;
         eVal = this.attrs.get(name);
      } else {
         eId = c.ATTRIBUTE_ADDED;
         eVal = value;
      }
      e = new c(this, eId, name, eVal);
   }

   this.attrs.put(name, value);

   if (e != null) {
      this._fireAttributeChanged(e);
   }
};

/**
 * Removes the object bound with the specified name from this component. If the
 * component does not have an object bound with the specified name, this method
 * does nothing.
 * <p/>
 * After this method executes, the container then notifies any
 * <code>ComponentAttributeListener</code>s that have registered interest on
 * this component.
 *
 * @param name String
 *    The name of the object to remove from this component
 */
_i.removeAttribute = function(name) {
   if (this.attrs == null) {
      return;
   }
   var oldValue = this.attrs.remove(name);

   if (this.listenerList.getListenerCountByType("ComponentAttributeListener") > 0) {
      // No need to create event if no one's interested
      var c = vpx.gui.event.ComponentBindingEvent;
      var id = c.ATTRIBUTE_REMOVED;
      var e = new c(this, id, name, oldValue);
      this._fireAttributeChanged(e);
   }
};

/**
 * Adds the specified attribute listener to receive component attribute events
 * from this component. If l is null, no exception is thrown, and no action is
 * performed.
 *
 * @param l vpx.gui.event.ComponentAttributeListener
 *    The component attribute listener
 */
_i.addAttributeListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.add("ComponentAttributeListener", l);
};

/**
 * Removes the specified attribute listener so it no longer receives component
 * attribute events from this component. If l is null, no exception is thrown,
 * and no action is performed.
 *
 * @param l vpx.gui.event.ComponentAttributeListener
 *    The component attribute listener
 */
_i.removeAttributeListener = function(l) {
   if (l == null) {
      return;
   }
   this.listenerList.remove("ComponentAttributeListener", l);
};

/**
 * Returns an array of all the attribute listeners registered on this component.
 *
 * @return vpx.gui.event.ComponentAttributeListener[]
 *    All of this component's <code>ComponentAttributeListener</code>s or an
 *    empty array if no component attribute listeners are currently registered
 */
_i.getAttributeListeners = function() {
   return this.listenerList.getListeners("ComponentAttributeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.
 *
 * @param e vpx.gui.event.ComponentBindingEvent
 *    The event to fire
 */
_i._fireAttributeChanged = function(e) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   var c = vpx.gui.event.ComponentBindingEvent;

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ComponentAttributeListener") {
         // listeners[i + 1] is of type vpx.gui.event.ComponentAttributeListener
         var l = listeners[i + 1];
         switch (event.getId()) {
         case c.ATTRIBUTE_ADDED:
            l.attributeAdded(e);
            break;
         case c.ATTRIBUTE_REMOVED:
            l.attributeRemoved(e);
            break;
         case c.ATTRIBUTE_REPLACED:
            l.attributeReplaced(e);
            break;
         default:
            throw new Error("Component#_fireAttributeChanged(): Unknown event id");
            break;
         }
      }
   }
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.Component]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


_i._initUI = function() {
   var constructor = eval("vpx.gui.plaf." + this.uiClassID);
   if (!constructor) {
      throw new Error("Class vpx.gui.plaf." + this.uiClassID + " not found");
   }
   this.ui = new constructor();
   this.ui.installUI(this);
};

_i._createBoundHandlers = function() {
   this.boundHandlers = {
      mouseAction : this._fireMouseActionPerformed.bind(this)
   };
};
