/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class DefaultButtonModel
 * extends Object
 * implements ButtonModel
 *
 * The default implementation of a <code>Button</code> component's data model.
 *
 * @version 1.0 (Sept 27, 2005)
 */

/**
 * Constructs a default <code>ButtonModel</code>.
 */
vpx.gui.DefaultButtonModel = function() {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   this.listenerList = new vpx.core.event.EventListenerList();
   this.setEnabled(true);
};

// Shorthand for brevity's sake
var _c = vpx.gui.DefaultButtonModel;  // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.stateMask     = 0;      // protected int
_i.actionCommand = null;   // protected String
_i.group         = null;   // protected vpx.gui.ButtonGroup
_i.changeEvent   = null;   // protected transient vpx.core.event.ChangeEvent
_i.listenerList  = null;   // protected vpx.core.event.EventListenerList

/**
 * Indicates partial commitment towards choosing the button.
 */
_c.ARMED    = 1 << 0;

/**
 * Indicates that the button has been selected. Only needed for certain types
 * of buttons - such as RadioButton or Checkbox.
 */
_c.SELECTED = 1 << 1;

/**
 * Indicates that the button has been "pressed" (typically, when the mouse is
 * released).
 */
_c.PRESSED  = 1 << 2;

/**
 * Indicates that the button can be selected by an input device (such as a
 * mouse pointer).
 */
_c.ENABLED  = 1 << 3;

/**
 * Indicates that the mouse is over the button.
 */
_c.ROLLOVER = 1 << 4;

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#isArmed()
 */
_i.isArmed = function() {
   var cls = vpx.gui.DefaultButtonModel;
   return (this.stateMask & cls.ARMED) != 0;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#isSelected()
 */
_i.isSelected = function() {
   var cls = vpx.gui.DefaultButtonModel;
   return (this.stateMask & cls.SELECTED) != 0;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#isEnabled()
 */
_i.isEnabled = function() {
   var cls = vpx.gui.DefaultButtonModel;
   return (this.stateMask & cls.ENABLED) != 0;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#isPressed()
 */
_i.isPressed = function() {
   var cls = vpx.gui.DefaultButtonModel;
   return (this.stateMask & cls.PRESSED) != 0;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#isRollover()
 */
_i.isRollover = function() {
   var cls = vpx.gui.DefaultButtonModel;
   return (this.stateMask & cls.ROLLOVER) != 0;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setArmed(boolean)
 */
_i.setArmed = function(b) {
   if (this.isArmed() == b || !this.isEnabled()) {
      return;
   }

   var cls = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= cls.ARMED;
   } else {
      this.stateMask &= ~(cls.ARMED);
   }

   this._fireStateChanged();
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setSelected(boolean)
 */
_i.setSelected = function(b) {
   if (this.isSelected() == b) {
      return;
   }

   var cls = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= cls.SELECTED;
   } else {
      this.stateMask &= ~(cls.SELECTED);
   }

   var ItemEvent = vpx.gui.event.ItemEvent;
   var event = new ItemEvent(this,
                             ItemEvent.ITEM_STATE_CHANGED,
                             this,
                             b ?  ItemEvent.SELECTED : ItemEvent.DESELECTED);
   this._fireItemStateChanged(event);

   this._fireStateChanged();
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setEnabled(boolean)
 */
_i.setEnabled = function(b) {
   if(this.isEnabled() == b) {
      return;
   }

   var cls = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= cls.ENABLED;
   } else {
      this.stateMask &= ~(cls.ENABLED);
      // unarm and unpress, just in case
      this.stateMask &= ~(cls.ARMED);
      this.stateMask &= ~(cls.PRESSED);
      this.stateMask &= ~(cls.ROLLOVER);
   }

   this._fireStateChanged();
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setPressed(boolean)
 */
_i.setPressed = function(b) {
   if(this.isPressed() == b || !this.isEnabled()) {
      return;
   }

   var cls = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= cls.PRESSED;
   } else {
      this.stateMask &= ~(cls.PRESSED);
   }

   if(!this.isPressed() && this.isArmed()) {
      //?? XXX Handle 'modifiers' and 'when'? Might require EventQueue
      var ActionEvent = vpx.gui.event.ActionEvent;
      var event = new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
                                  this.getActionCommand());
      this._fireActionPerformed(event);
   }

   this._fireStateChanged();
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setRollover(boolean)
 */
_i.setRollover = function(b) {
   if(this.isRollover() == b || !this.isEnabled()) {
      return;
   }

   var cls = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= cls.ROLLOVER;
   } else {
      this.stateMask &= ~(cls.ROLLOVER);
   }

   this._fireStateChanged();
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setActionCommand(String)
 */
_i.setActionCommand = function(s) {
   this.actionCommand = s;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#getActionCommand()
 */
_i.getActionCommand = function() {
   return this.actionCommand;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#setGroup(vpx.gui.ButtonGroup)
 */
_i.setGroup = function(g) {
   this.group = g;
};

/**
 * Returns the group that this button belongs to. Normally used with radio
 * buttons, which are mutually exclusive within their group.
 *
 * @return vpx.gui.ButtonGroup
 *    A <code>ButtonGroup</code> that this button belongs to
 */
_i.getGroup = function() {
   return this.group;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#addActionListener(vpx.gui.event.ActionListener)
 */
_i.addActionListener = function(l) {
   this.listenerList.add("ActionListener", l);
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#removeActionListener(vpx.gui.event.ActionListener)
 */
_i.removeActionListener = function(l) {
   this.listenerList.remove("ActionListener", l);
};

/**
 * Returns an array of all the action listeners
 * registered on this <code>DefaultButtonModel</code>.
 *
 * @return vpx.gui.event.ActionListener[]
 *    All of this model's <code>ActionListener</code>s, or an empty array if no
 *    action listeners are currently registered
 */
_i.getActionListeners = function() {
   return this.listenerList.getListeners("ActionListener");
};

/**
 * Notifies all listeners that have registered interest for
 * notification on this event type.
 *
 * @param e vpx.gui.event.ActionEvent
 *    The <code>ActionEvent</code> to deliver to listeners
 */
_i._fireActionPerformed = function(e) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ActionListener") {
         // listeners[i + 1] is of type vpx.gui.event.ActionListener
         listeners[i + 1].actionPerformed(e);
      }
   }
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#addItemListener(vpx.gui.event.ItemListener)
 */
_i.addItemListener = function(l) {
   this.listenerList.add("ItemListener", l);
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#removeItemListener(vpx.gui.event.ItemListener)
 */
_i.removeItemListener = function(l) {
   this.listenerList.remove("ItemListener", l);
};

/**
 * Returns an array of all the item listeners registered on this
 * <code>DefaultButtonModel</code>.
 *
 * @return vpx.gui.event.ItemListener[]
 *    All of this model's <code>ItemListener</code>s, or an empty array if no
 *    item listeners are currently registered
 */
_i.getItemListeners = function() {
   return this.listenerList.getListeners("ItemListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.
 *
 * @param e vpx.gui.event.ItemEvent
 *    The <code>ItemEvent</code> to deliver to listeners
 */
_i._fireItemStateChanged = function(e) {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ItemListener") {
         // listeners[i + 1] is of type vpx.gui.event.ItemListener
         listeners[i + 1].itemStateChanged(e);
      }
   }
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#addChangeListener(vpx.core.event.ChangeListener)
 */
_i.addChangeListener = function(l) {
   this.listenerList.add("ChangeListener", l);
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.ButtonModel#removeChangeListener(vpx.core.event.ChangeListener)
 */
_i.removeChangeListener = function(l) {
   this.listenerList.remove("ChangeListener", l);
};

/**
 * Returns an array of all the change listeners registered on this
 * <code>DefaultButtonModel</code>.
 *
 * @return vpx.core.event.ChangeListener[]
 *    All of this model's <code>ChangeListener</code>s, or an empty array if no
 *    change listeners are currently registered
 */
_i.getChangeListeners = function() {
   return this.listenerList.getListeners("ChangeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type. The event instance is created lazily.
 */
_i._fireStateChanged = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ChangeListener") {
         // Lazily create the event:
         if (this.changeEvent == null) {
            this.changeEvent = new vpx.core.event.ChangeEvent(this);
         }

         // listeners[i + 1] is of type vpx.core.event.ChangeListener
         listeners[i + 1].stateChanged(this.changeEvent);
      }
   }
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   var str = "[Object vpx.gui.DefaultButtonModel";

   var states = [];
   if (this.isArmed()) {
      states.push("ARMED");
   }
   if (this.isSelected()) {
      states.push("SELECTED");
   }
   if (this.isPressed()) {
      states.push("PRESSED");
   }
   if (this.isEnabled()) {
      states.push("ENABLED");
   }
   if (this.isRollover()) {
      states.push("ROLLOVER");
   }
   if (states.length > 0) {
      str += ",state=" + states.join("|");
   }
   delete states;

   str += "]";
   return str;
};
