/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * The <code>Dimension</code> class encapsulates the width and height of a
 * component (in integer precision) in a single object. The class is associated
 * with certain properties of components. Several methods defined by the
 * <code>Component</code> class return a <code>Dimension</code> object.
 * <p/>
 * Normally the values of <code>width</code> and <code>height</code> are
 * non-negative integers. The constructors that allow you to create a dimension
 * do not prevent you from setting a negative value for these properties.
 * If the value of <code>width</code> or <code>height</code> is negative, the
 * behavior of some methods defined by other objects is undefined.
 *
 * @version 1.0 (Oct 19, 2005)
 */

/**
 * public Dimension()
 *
 * Creates an instance of <code>Dimension</code> with a width of zero and a
 * height of zero.
 */

/**
 * public Dimension(Dimension d)
 *
 * Creates an instance of <code>Dimension</code> whose width and height are the
 * same as the specified dimension's.
 *
 * @param d vpx.gui.Dimension
 *    the specified dimension for the <code>width</code> and
 *    <code>height</code> values
 */

/**
 * public Dimension(int w, int h)
 *
 * Constructs a <code>Dimension</code> and initializes it to the specified
 * width and specified height.
 *
 * @param w int
 *    The width
 * @param h int
 *    The height
 */

vpx.gui.Dimension = function() {
   switch (arguments.length) {
   case 0:
      // Dimension()
      this.width = 0;
      this.height = 0;
      break;

   case 1:
      // Dimension(d)
      var d = arguments[0];
      this.width = d.width;
      this.height = d.height;
      break;

   case 2:
      // Dimension(int w, int h)
      var w = arguments[0];
      var h = arguments[1];
      this.width = w;
      this.height = h;
      break;

   default:
      throw new Error("No such constructor: vpx.gui.Dimension(arg, arg, arg)");
      break;
   }
};

// Shorthand for brevity's sake
var _c = vpx.gui.Dimension;        // Class
var _i = _c.prototype;             // Instance
_i._concrete = true;               // vpx system flag for concrete classes

// Instance variables
_i.width = null;                   // public int
_i.height = null;                  // public int

/**
 * Returns the width of this dimension.
 *
 * @return int
 *    The width of this dimension
 */
_i.getWidth = function() {
   return this.width;
};

/**
 * Returns the height of this dimension.
 *
 * @return int
 *    The height of this dimension
 */
_i.getHeight = function() {
   return this.height;
};

/**
 * Checks whether two dimension objects have equal values.
 *
 * @param obj Object
 *    The object to compare
 * @return boolean
 *    true if the two objects are logically equal; false otherwise
 * @see Object#equals(Object)
 */
_i.equals = function(obj) {
   if (obj instanceof vpx.gui.Dimension) {
      return (this.width == obj.width) && (this.height == obj.height);
   }
   return false;
};

/**
 * Returns a string representation of the values of this <code>Dimension</code>
 * object's <code>height</code> and <code>width</code> fields. This method is
 * intended to be used only for debugging purposes, and the content and format
 * of the returned string may vary between implementations. The returned string
 * may be empty but may not be <code>null</code>.
 *
 * @return String
 *    A string representation of this <code>Dimension</code> object
 */
_i.toString = function() {
   return "[Object vpx.gui.Dimension; width=" + this.width + ",height=" + this.height + "]";
};
