/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class Film
 * extends vpx.gui.Component
 *
 * A film is a frame that is designed to cover a view, including native
 * controls that are otherwise "uncoverable".
 *
 * @version 1.0 (Oct 25, 2005)
 */

/**
 * Constructs a new Film.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the film
 */
vpx.gui.Film = function(view) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }
   if (!this._concrete) {
      throw new Error("vpx.gui.Film: cannot instantiate abstract class");
   }

   // super(view)
   vpx.gui.Component.call(this, view);

   this._initUI();

   // Film elements get automatically added to the view's document body
   var cntnr = view.document.body;
   cntnr.appendChild(this._getCoverPeer());
   cntnr.appendChild(this.getPeer());
};

// Film extends vpx.gui.Component
vpx.gui.Film.prototype = new vpx.gui.Component(vpx.ABSTRACT_PASS);
vpx.gui.Film.prototype.constructor = vpx.gui.Film;

// Shorthand for brevity's sake
var _c = vpx.gui.Film;                // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.uiClassID = "FilmUI";              // private static final String
_i.visible = false;                   // protected boolean
_i.ui = null;                         // protected vpx.gui.plaf.FilmUI

/**
 * Only one <code>ChangeEvent</code> is needed per film instance since the
 * event's only state is the source property.  The source of events generated
 * is always "this".
 */
_i.changeEvent = null;                // protected transient vpx.core.event.ChangeEvent

/**
 * Tells whether or not the film is currently visible.  Note that the film may
 * be visible but set to be transparent.
 *
 * @return boolean
 *    true if this film is visible; false if it is invisible
 */
_i.isVisible = function() {
   return this.visible;
};

/**
 * Sets the visibility of this film.
 *
 * @param b boolean
 *    true to make the film visible; false to make it invisible
 */
_i.setVisible = function(b) {
   // Is it a no-op?
   if (b == this.visible) {
      return;
   }
   this.visible = b;

   var cls = vpx.gui.Component;
   if (b) {
      this.firePropertyChange(cls.VISIBILITY_PROPERTY, false, true);
   } else {
      this.firePropertyChange(cls.VISIBILITY_PROPERTY, true, false);
   }
   this._fireStateChanged();
};

/**
 * Adds a <code>ChangeListener</code> to the film.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to be added
 */
_i.addChangeListener = function(l) {
   this.listenerList.add("ChangeListener", l);
};

/**
 * Removes a ChangeListener from the film.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to be removed
 */
_i.removeChangeListener = function(l) {
   this.listenerList.remove("ChangeListener", l);
};

/**
 * Returns an array of all the <code>ChangeListener</code>s added
 * to this Film with addChangeListener().
 *
 * @return vpx.core.event.ChangeListener[]
 *    All of the <code>ChangeListener</code>s added, or an empty array if no
 *    listeners have been added
 */
_i.getChangeListeners = function() {
   return this.listenerList.getListeners("ChangeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is lazily created.
 */
_i._fireStateChanged = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ChangeListener") {
         // Lazily create the event:
         if (this.changeEvent == null) {
            this.changeEvent = new vpx.core.event.ChangeEvent(this);
         }

         // listeners[i + 1] is of type vpx.core.event.ChangeListener
         listeners[i + 1].stateChanged(this.changeEvent);
      }
   }
};

/**
 * Gets the native peer of the component that's responsible for covering native
 * elements.  This peer is different than the standard peer in that its purpose
 * is strictly for covering elements -- no components will ever be added to it
 * as children.
 *
 * @return HTMLElement
 *    The native "cover peer" of the component
 */
_i._getCoverPeer = function() {
   return this.ui.coverPeer;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.Film]";
};
