/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class MediaTracker
 * extends Object
 */

/**
 * Creates a media tracker to track images.
 */
vpx.gui.MediaTracker = function() {
   this.active = {};
};

// Shorthand for brevity's sake
var _c = vpx.gui.MediaTracker;     // Class
var _i = _c.prototype;             // Instance
_i._concrete = true;               // vpx system flag for concrete classes

// Class constants
_c.LOADING  = 1;                   // public static final int
_c.ABORTED  = 2;                   // public static final int
_c.ERRORED  = 4;                   // public static final int
_c.COMPLETE = 8;                   // public static final int

_c.DONE     = (_c.ABORTED | _c.ERRORED | _c.COMPLETE);

// Class variables
_c.library  = {};                  // private static Hashtable
_c.buffer   = null;                // private static HTMLElement

// Instance variables
_i.active   = null;                // private Hashtable

/**
 * Adds an image to the list of images being tracked by this media tracker. The
 * image will eventually be rendered at its default (unscaled) size.
 *
 * @param src String
 *    The URI "src" of the image to be tracked
 * @param id int
 *    An identifier used to track this image
 * @return HTMLImageElement
 *    A newly created Image object that corresponds to the given src
 */
_i.addImage = function(src, id) {
   var cls = vpx.gui.MediaTracker;
   var img;
   if (cls.library[src]) {
      img = cls.library[src].cloneNode(false);
      img.onload = null;
      img.onabort = null;
      img.onerror = null;
   } else {
      img = cls.library[src] = document.createElement("img");
      img.src = src;
   }
   this.active[id] = {
      status : cls.LOADING,
      img    : img
   };
   return img;
};

_i.removeImage = function(id) {
   if (!this.active[id]) {
      return;
   }
   delete this.active[id].status;
   delete this.active[id].img;
   delete this.active[id];
};

_i.waitForId = function(id, onDone) {
   var node = this.active[id];
   if (!node) {
      throw new Error("vpx.gui.MediaTracker#waitForId(): invalid id: " + id);
   }

   var cls = vpx.gui.MediaTracker;
   if ((node.status & cls.DONE) != 0) {
      onDone();
      return;
   }

   var img = node.img;
   img.onload = this._handleImgDone.bind(this, id, cls.COMPLETE, onDone);
   img.onabort = this._handleImgDone.bind(this, id, cls.ABORTED, onDone);
   img.onerror = this._handleImgDone.bind(this, id, cls.ERRORED, onDone);

   if (!cls.buffer) {
      cls.buffer = cls._createBuffer();
   }
   cls.buffer.appendChild(img);
};

_i.statusId = function(id) {
   var node = this.active[id];
   if (!node) {
      throw new Error("vpx.gui.MediaTracker#statusId(): invalid id: " + id);
   }
   return node.status;
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 *
 */
_i._handleImgDone = function(id, status, callback) {
   var node = this.active[id];
   if (!node) {
      return;
   }

   var img = node.img;
   img.onload = null;
   img.onabort = null;
   img.onerror = null;

   node.status = status;
   callback();
};

/**
 * Returns a component in which to draw buffered images.
 *
 * @return HTMLElement
 *    The component in which the images will be buffered
 */
_c._createBuffer = function() {
   if (!document.body) {
      throw new Error("vpx.gui.MediaTracker#_createBuffer(): document not loaded");
   }
   var comp = document.createElement("div");
   comp.style.position = "absolute";
   comp.style.visibility = "hidden";
   if (document.body.firstChild) {
      document.body.insertBefore(comp, document.body.firstChild);
   } else {
      document.body.appendChild(comp);
   }
   return comp;
};
