/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class MenuBar
 * extends vpx.gui.Component
 * implements vpx.gui.MenuElement
 *
 * An implementation of a menu bar. You add <code>Menu</code> objects to the
 * menu bar to construct a menu. When the user selects a <code>Menu</code>
 * object, its associated <code>PopupMenu</code> is displayed, allowing the
 * user to select one of the <code>MenuItems</code> on it.
 * <p/>
 * VPX Menus are based on Java Swing menus.  For information and examples of
 * using Swing JMenus, see
 * <a href="http://java.sun.com/doc/books/tutorial/uiswing/components/menu.html">How to Use Menus</a>,
 * a section in <em>The Java Tutorial.</em>
 *
 * @version 1.0 (Oct 12, 2005)
 * @see vpx.gui.Menu
 * @see vpx.gui.PopupMenu
 * @see vpx.gui.MenuItem
 */

/**
 * Creates a new menubar.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the menubar
 */
vpx.gui.MenuBar = function(view) {
   // super(view)
   vpx.gui.Component.call(this, view);

   this.setSelectionModel(new vpx.gui.DefaultSingleSelectionModel());
   this._initUI();
};

// MenuBar extends vpx.gui.Component
vpx.gui.MenuBar.prototype = new vpx.gui.Component(vpx.ABSTRACT_PASS);
vpx.gui.MenuBar.prototype.constructor = vpx.gui.MenuBar;

// Shorthand for brevity's sake
var _c = vpx.gui.MenuBar;             // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes
_i._IMPL_vpx_gui_MenuElement = true;  // vpx system flag for interface impl

// Instance variables
_i.uiClassID = "MenuBarUI";           // private static final String
_i.selectionModel = null;             // private transient vpx.gui.SingleSelectionModel

// Class constants
_c.SELECTION_MODEL_PROPERTY = "selectionModel";  // public static final String

/**
 * Returns the menu at the specified position in the menu bar.
 *
 * @param i int
 *    An integer giving the position in the menu bar, where 0 is the first
 *    position
 * @return vpx.gui.Menu
 *    The <code>Menu</code> at that position, or <code>null</code> if there is
 *    no <code>Menu</code> at that position (ie. if it is a
 *    <code>MenuItem</code>)
 */
_i.getMenu = function(i) {
   var c = this.getComponentAt(i);
   if (c instanceof vpx.gui.Menu) {
      return c;
   }
   return null;
};

/**
 * Returns the number of items in the menu bar.
 *
 * @return int
 *    The number of items in the menu bar
 */
_i.getMenuCount = function() {
   return this.getComponentCount();
};

/**
 * Returns the index of the specified component.
 *
 * @param c vpx.gui.Component
 *    The <code>Component</code> to find
 * @return int
 *    An integer giving the component's position, where 0 is first; or -1 if it
 *    can't be found
 */
_i.getComponentIndex = function(c) {
   var count = this.getComponentCount();
   var a = this.getComponents();
   for (var i = 0; i < count; i++) {
      if (a[i] == c) {
         return i;
      }
   }
   return -1;
};

/**
 * Sets the currently selected component, producing a change to the selection
 * model.
 *
 * @param sel vpx.gui.Component
 *    The <code>Component</code> to select
 */
_i.setSelected = function(sel) {
   var i = this.getComponentIndex(sel);
   this.selectionModel.setSelectedIndex(i);
};

/**
 * Returns true if the menu bar currently has a component selected.
 *
 * @return boolean
 *    true if a selection has been made, else false
 */
_i.isSelected = function() {
   return this.selectionModel.isSelected();
};

/**
 * Returns the model object that handles single selections.
 *
 * @return vpx.gui.SingleSelectionModel
 *    The <code>selectionModel</code> property
 */
_i.getSelectionModel = function() {
   return this.selectionModel;
};

/**
 * Sets the model object to handle single selections.
 *
 * @param model vpx.gui.SingleSelectionModel
 *    The <code>SingleSelectionModel</code> to use
 */
_i.setSelectionModel = function(model) {
   var oldValue = this.selectionModel;
   this.selectionModel = model;
   var cls = vpx.gui.MenuBar;
   this.firePropertyChange(cls.SELECTION_MODEL_PROPERTY, oldValue, model);
};

/**
 * Implemented to be a <code>MenuElement</code>. Does nothing.
 *
 * @see vpx.gui.MenuElement#processMouseEvent(MouseEvent, vpx.gui.MenuElement[], vpx.gui.MenuSelectionManager)
 */
_i.processMouseEvent = function(event, path, manager) {
   // Do nothing
};

/**
 * Implemented to be a <code>MenuElement</code>. Does nothing.
 *
 * @see vpx.gui.MenuElement#menuSelectionChanged(boolean)
 */
_i.menuSelectionChanged = function(isIncluded) {
   // Do nothing
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns the menus in this menu
 * bar. This is the reason for implementing the <code>MenuElement</code>
 * interface -- so that the menu bar can be treated the same as other menu
 * elements.
 *
 * @see vpx.gui.MenuElement#getSubElements()
 */
_i.getSubElements = function() {
   var res = [];
   var c = this.getComponentCount();

   for (var i = 0; i < c; i++) {
      var m = this.getComponentAt(i);
      if (m._IMPL_vpx_gui_MenuElement) {
         res.push(m);
      }
   }

   return res;
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns this object.
 *
 * @see vpx.gui.MenuElement#getComponent()
 */
_i.getComponent = function() {
   return this;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.MenuBar]";
};
