/* Copyright 2006 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * public class MenuButton
 * extends vpx.gui.AbstractButton
 * implements vpx.gui.MenuElement
 *
 * An implementation of a "push" button that brings up a menu when activated.
 * <p/>
 * VPX Buttons are based on Java Swing buttons. For information and examples of
 * using Swing buttons, see
 * <a href="http://java.sun.com/docs/books/tutorial/uiswing/components/button.html">How to Use Buttons, Check Boxes, and Radio Buttons</a>,
 * a section in <em>The Java Tutorial.</em>
 *
 * @version 1.0 (Apr 18, 2006)
 */

/**
 * Creates a button with initial text and an icon.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the menubar
 * @param text String
 *    The text of the button
 * @param icon vpx.gui.Icon
 *    The Icon image to display on the button
 */
vpx.gui.MenuButton = function MenuButton(view, text, icon) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super(view)
   vpx.gui.AbstractButton.call(this, view);

   // Create the model
   this.setModel(new vpx.gui.DefaultButtonModel());

   // initialize
   this._initUI();
   this._init(text, icon);
};

// MenuButton extends vpx.gui.AbstractButton
vpx.gui.MenuButton.prototype = new vpx.gui.AbstractButton(vpx.ABSTRACT_PASS);
vpx.gui.MenuButton.prototype.constructor = vpx.gui.MenuButton;

// Shorthand for brevity's sake
var _c = vpx.gui.MenuButton;           // Class
var _i = _c.prototype;                 // Instance
_i._concrete = true;                   // vpx system flag for concrete classes
_i._IMPL_vpx_gui_MenuElement = true;   // vpx system flag for interface impl
_i._IMPL_vpx_gui_PopupInvoker = true;  // vpx system flag for interface impl

// Instance variables
_i.uiClassID = "MenuButtonUI";         // private static final String
_i.menuEvent = null;                   // protected transient vpx.gui.event.MenuEvent

/**
 * private vpx.gui.PopupMenu
 *
 * The popup menu portion of the button.
 */
_i.popupMenu = null;

/**
 * private vpx.core.event.ChangeListener
 *
 * The button's model listener.  Default is <code>null</code>.
 */
_i.menuChangeListener = null;

/**
 * private vpx.gui.Point
 *
 * Location of the popup component. Location is <code>null</code> if it was not
 * customized by <code>setMenuLocation</code>
 */
_i.customMenuLocation = null;

/**
 * Appends a menu item to the end of this button.  Returns the menu item added.
 *
 * @param menuItem vpx.gui.MenuItem
 *    The <code>Menuitem</code> to be added
 * @return vpx.gui.MenuItem
 *    The <code>MenuItem</code> added
 */
_i.add = function(menuItem) {
   this._ensurePopupMenuCreated();
   return this.popupMenu.add(menuItem);
};

/**
 * Appends a new separator to the end of the button.
 */
_i.addSeparator = function() {
    this._ensurePopupMenuCreated();
    this.popupMenu.addSeparator();
};

/**
 * Inserts the specified <code>Menuitem</code> at a given position.
 *
 * @param mi vpx.gui.MenuItem
 *    The <code>Menuitem</code> to add
 * @param pos int
 *    An integer specifying the position at which to add the new
 *    <code>Menuitem</code>
 * @return vpx.gui.MenuItem
 *    The new menu item
 * @exception Error
 *    If the value of <code>pos</code> < 0
 */
_i.insert = function(mi, pos) {
   if (pos < 0) {
      throw new Error("vpx.gui.MenuButton#insert(): index less than zero");
   }
   this._ensurePopupMenuCreated();
   this.popupMenu.insert(mi, pos);
   return mi;
};

/**
 * Inserts a separator at the specified position.
 *
 * @param index int
 *    An integer specifying the position at which to insert the menu separator
 * @exception Error
 *    If the value of <code>index</code> < 0
 */
_i.insertSeparator = function(index) {
   if (index < 0) {
      throw new Error("vpx.gui.MenuButton#insertSeparator(): index less than zero.");
   }

   this._ensurePopupMenuCreated();
   this.popupMenu.insert(new vpx.gui.Separator(this.popupMenu.getView()), index);
};

/**
 * Returns the popupmenu associated with this button.  If there is no popup
 * menu, this will create one.
 *
 * @return vpx.gui.PopupMenu
 *    The popup menu associated with this menu
 */
_i.getPopupMenu = function() {
   this._ensurePopupMenuCreated();
   return this.popupMenu;
};

/**
 * Returns true if the button's popup window is visible.
 *
 * @return boolean
 *    true if the menu is visible, false otherwise
 */
_i.isPopupMenuVisible = function() {
   this._ensurePopupMenuCreated();
   return this.popupMenu.isVisible();
};

/**
 * Sets the visibility of the button's popup.  If the button is not enabled,
 * this method will have no effect.
 *
 * @param b boolean
 *    true to make the menu visible, false to hide it
 */
_i.setPopupMenuVisible = function(b) {
   var isVisible = this.isPopupMenuVisible();
   if (b != isVisible && (this.isEnabled() || !b)) {
      this._ensurePopupMenuCreated();
      if (b && this.isShowing()) {
         // Set location of popupMenu (pulldown or pullright)
         var p = this._getCustomMenuLocation();
         if (p == null) {
            p = this._getPopupMenuOrigin();
         }
         this.getPopupMenu().show(this, p.x, p.y);
      } else {
         this.getPopupMenu().setVisible(false);
      }
   }
};

/**
 * Removes the menu item at the specified index from this button.
 *
 * @param pos int
 *    The position of the item to be removed
 * @exception Error
 *    If the value of <code>pos</code> < 0, or if <code>pos</code> is greater
 *    than the number of menu items
 */
_i.removeAt = function(pos) {
   if (pos < 0) {
      throw new Error("vpx.gui.MenuButton#removeAt(): index less than zero");
   }
   if (pos > this.getMenuComponentCount()) {
      throw new Error("vpx.gui.MenuButton#removeAt(): index greater than the number of items");
   }
   if (this.popupMenu != null) {
      this.popupMenu.removeAt(pos);
   }
};

/**
 * Removes the specified menu item from this button.  If there is no popup
 * menu, this method will have no effect.
 *
 * @param item vpx.gui.MenuItem
 *    The <code>MenuItem</code> to be removed from the menu
 */
_i.remove = function(item) {
   if (this.popupMenu != null) {
      this.popupMenu.remove(item);
   }
};

/**
 * Removes all menu items from this button.
 */
_i.removeAll = function() {
   if (this.popupMenu != null) {
      this.popupMenu.removeAll();
   }
};

/**
 * Returns the number of components in the button.
 *
 * @return int
 *    An integer containing the number of components in the button
 */
_i.getMenuComponentCount = function() {
   var componentCount = 0;
   if (this.popupMenu != null) {
      componentCount = this.popupMenu.getComponentCount();
   }
   return componentCount;
};

/**
 * Returns the component at position <code>n</code>.
 *
 * @param n int
 *    The position of the component to be returned
 * @return vpx.gui.Component
 *    The component requested, or <code>null</code> if there is no popup menu
 */
_i.getMenuComponent = function(n) {
   if (this.popupMenu != null) {
      return this.popupMenu.getComponentAt(n);
   }

   return null;
};

/**
 * Returns an array of <code>Component</code>s of the button's subcomponents.
 * Note that this returns all <code>Component</code>s in the popup menu,
 * including separators.
 *
 * @return vpx.gui.Component[]
 *    An array of <code>Component</code>s or an empty array if there is no
 *    popup menu
 */
_i.getMenuComponents = function() {
   if (this.popupMenu != null) {
      return this.popupMenu.getComponents();
   }

   return [];
};

/**
 * Sets the data model for this button.
 *
 * @param newModel vpx.gui.ButtonModel
 *    The <code>ButtonModel</code>
 */
_i.setModel = function(newModel) {
   var oldModel = this.getModel();

   // super.setModel(newModel)
   var spr = vpx.gui.AbstractButton.prototype;
   spr.setModel.call(this, newModel);

   if (oldModel != null && this.menuChangeListener != null) {
      oldModel.removeChangeListener(this.menuChangeListener);
      delete this.menuChangeListener;
   }

   this.model = newModel;

   if (newModel != null) {
      this.menuChangeListener = this._createMenuChangeListener();
      newModel.addChangeListener(this.menuChangeListener);
   }
};

/**
 * Returns true if the button is currently selected (the menu is open).
 *
 * @return boolean
 *    true if the button is selected, else false
 */
_i.isSelected = function() {
    return this.getModel().isSelected();
};

/**
 * Sets the selection status of the button (whether the menu is open).
 *
 * @param b boolean
 *    true to select (open) the button; false to de-select the button
 */
_i.setSelected = function(b) {
   var model = this.getModel();

   if (b != model.isSelected()) {
      model.setSelected(b);
   }
};

/**
 * Sets the location of the popup component.
 *
 * @param x int
 *    The x coordinate of the popup's new position
 * @param y int
 *    The y coordinate of the popup's new position
 */
_i.setMenuLocation = function(x, y) {
    if (this.customMenuLocation != null) {
       delete this.customMenuLocation;
    }
    this.customMenuLocation = new vpx.gui.Point(x, y);
    if (this.popupMenu != null) {
       this.popupMenu.setLocation(x, y);
    }
};

/**
 * Computes the origin for the <code>MenuButton</code>'s popup menu.
 *
 * @return vpx.gui.Point
 *    A <code>Point</code> in the coordinate space of the menu that should be
 *    used as the origin of the <code>Menu</code>'s popup menu
 */
_i._getPopupMenuOrigin = function() {
   var peer = this.getPeer();
   return new vpx.gui.Point(0, peer.offsetHeight);
};

/**
 * Adds a listener for menu events. If l is null, no exception is thrown, and
 * no action is performed.
 *
 * @param l vpx.gui.event.MenuListener
 *    The listener to be added
 */
_i.addMenuListener = function(l) {
	if (l == null) {
      return;
	}
   this.listenerList.add("MenuListener", l);
};

/**
 * Removes a listener for menu events. If l is null, no exception is thrown,
 * and no action is performed.
 *
 * @param l vpx.gui.event.MenuListener
 *    The listener to be removed
 */
_i.removeMenuListener = function(l) {
	if (l == null) {
      return;
	}
   this.listenerList.remove("MenuListener", l);
};

/**
 * Returns an array of all the <code>MenuListener</code>s added
 * to this Menu with addMenuListener().
 *
 * @return vpx.gui.event.MenuListener[]
 *    All of the <code>MenuListener</code>s added or an empty array if no
 *    listeners have been added
 */
_i.getMenuListeners = function() {
   return this.listenerList.getListeners("MenuListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is created lazily.
 */
_i._fireMenuSelected = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "MenuListener") {
         // Lazily create the event:
         if (this.menuEvent == null) {
            this.menuEvent = new vpx.gui.event.MenuEvent(this);
         }

         // listeners[i + 1] is of type vpx.gui.event.MenuListener
         listeners[i + 1].menuSelected(this.menuEvent);
      }
   }
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is created lazily.
 */
_i._fireMenuDeselected = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "MenuListener") {
         // Lazily create the event:
         if (this.menuEvent == null) {
            this.menuEvent = new vpx.gui.event.MenuEvent(this);
         }

         // listeners[i + 1] is of type vpx.gui.event.MenuListener
         listeners[i + 1].menuDeselected(this.menuEvent);
      }
   }
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.  The event instance is created lazily.
 */
_i._fireMenuCanceled = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "MenuListener") {
         // Lazily create the event:
         if (this.menuEvent == null) {
            this.menuEvent = new vpx.gui.event.MenuEvent(this);
         }

         // listeners[i + 1] is of type vpx.gui.event.MenuListener
         listeners[i + 1].menuCanceled(this.menuEvent);
      }
   }
};

/**
 * Implemented to be a <code>MenuElement</code>. Messaged when the menu
 * selection changes to activate or deactivate this button.
 *
 * @see vpx.gui.MenuElement#menuSelectionChanged(boolean)
 */
_i.menuSelectionChanged = function(isIncluded) {
   this.setSelected(isIncluded);
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns an array of
 * <code>MenuElement</code>s containing the submenu for this button component.
 * If popup menu is <code>null</code>, returns an empty array. This method is
 * required to conform to the <code>MenuElement</code> interface.
 *
 * @see vpx.gui.MenuElement#getSubElements()
 */
_i.getSubElements = function() {
   if (this.popupMenu == null) {
      return [];
   }
   return [this.popupMenu];
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.MenuButton]";
};


/**
 * class MenuChangeListener
 * extends Object
 * implements vpx.core.event.ChangeListener
 *
 * Listener that automatically gets registered with this button's model
 * when the model is set.  It will keep track of the selection state of the
 * button and fire corresponding <code>MenuEvent</code>s.
 *
 * @version 1.0 (Apr 18, 2006)
 */

/**
 * Constructs a new MenuChangeListener.
 *
 * @param menu vpx.gui.Menu
 *    The friend menu to listen to
 */
_c.MenuChangeListener = function(menu) {
    this.isSelected = false;
    this.menu = menu;
};

// Instance variables
_c.MenuChangeListener.prototype.isSelected = false;

/**
 * (non-Javadoc)
 *
 * @see vpx.core.event.ChangeListener#stateChanged(vpx.core.event.ChangeEvent)
 */
_c.MenuChangeListener.prototype.stateChanged = function(e) {
    var model = e.getSource();
    var modelSelected = model.isSelected();

    if (modelSelected != this.isSelected) {
        if (modelSelected == true) {
            this.menu._fireMenuSelected();
        } else {
            this.menu._fireMenuDeselected();
        }
        this.isSelected = modelSelected;
    }
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Return the customized location of the popup component.
 *
 * @return vpx.gui.Point
 *    This popup menu's custom location, or <code>null</code> if
 *    <code>setCustomMenuLocation</code> has not been called
 */
_i._getCustomMenuLocation = function() {
   return this.customMenuLocation;
};

/**
 * Creates the button's popup menu if needed.
 */
_i._ensurePopupMenuCreated = function() {
    if (this.popupMenu == null) {
        this.popupMenu = new vpx.gui.PopupMenu(vpx.getTle());
        this.popupMenu.setInvoker(this);
    }
};

/**
 * Creates this button's menu change listener.
 */
_i._createMenuChangeListener = function() {
   return new vpx.gui.MenuButton.MenuChangeListener(this);
};
