/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class MenuItem
 * extends vpx.gui.AbstractButton
 * implements vpx.gui.MenuElement
 *
 * An implementation of an item in a menu. A menu item is essentially a button
 * sitting in a list. When the user selects the "button", the action
 * associated with the menu item is performed. A <code>MenuItem</code>
 * contained in a <code>PopupMenu</code> performs exactly that function.
 */

/**
 * Creates a <code>MenuItem</code> with the specified text and icon.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the menubar
 * @param text String
 *    The text of the <code>MenuItem</code>
 * @param icon vpx.gui.Icon
 *    The icon of the <code>MenuItem</code>
 * @param model vpx.gui.ButtonModel
 *    The menu item's button model. If unspecified, this will default to a new
 *    <code>DefaultButtonModel</code>
 */
vpx.gui.MenuItem = function(view, text, icon, model) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super(view)
   vpx.gui.AbstractButton.call(this, view);

   if (arguments.length < 4 || model == null) {
      model = new vpx.gui.DefaultButtonModel();
   }
   this.setModel(model);
   this._initUI();
   this._init(text, icon);
};

// MenuItem extends vpx.gui.AbstractButton
vpx.gui.MenuItem.prototype = new vpx.gui.AbstractButton(vpx.ABSTRACT_PASS);
vpx.gui.MenuItem.prototype.constructor = vpx.gui.MenuItem;

// Shorthand for brevity's sake
var _c = vpx.gui.MenuItem;               // Class
var _i = _c.prototype;                   // Instance
_i._concrete = true;                     // vpx system flag for concrete classes
_i._IMPL_vpx_gui_MenuElement = true;     // vpx system flag for interface impl

// Instance variables
_i.uiClassID = "MenuItemUI";             // private static final String

/**
 * Identifies the menu item as "armed". If the mouse button is
 * released while it is over this item, the menu's action event
 * will fire. If the mouse button is released elsewhere, the
 * event will not fire and the menu item will be disarmed.
 *
 * @param b boolean
 *    true to arm the menu item so it can be selected
 */
_i.setArmed = function(b) {
   var model = this.getModel();

   if(model.isArmed() != b) {
      model.setArmed(b);
   }
};

/**
 * Returns whether the menu item is "armed".
 *
 * @return boolean
 *    true if the menu item is armed and can be selected
 */
_i.isArmed = function() {
   return this.getModel().isArmed();
};

/**
 * Enables or disables the menu item.
 *
 * @param b boolean
 *    true to enable the item
 */
_i.setEnabled = function(b) {
   // Make sure we aren't armed!
   if (b == false) {
      this.setArmed(false);
   }

   // super.setEnabled(b)
   var spr = vpx.gui.AbstractButton.prototype;
   spr.setEnabled.call(this, b);
};

/**
 * Implemented to be a <code>MenuElement</code>. Does nothing.
 *
 * @see vpx.gui.MenuElement#processMouseEvent(MouseEvent, vpx.gui.MenuElement[], vpx.gui.MenuSelectionManager)
 */
_i.processMouseEvent = function(event, path, manager) {
   // Do nothing
};

/**
 * Implemented to be a <code>MenuElement</code>. Called by the
 * <code>MenuSelectionManager</code> when the <code>MenuElement</code> is
 * selected or unselected.
 *
 * @see vpx.gui.MenuElement#menuSelectionChanged(boolean)
 */
_i.menuSelectionChanged = function(isIncluded) {
   this.setArmed(isIncluded);
};

/**
 * Implemented to be a <code>MenuElement</code>. This method returns an array
 * containing the sub-menu components for this menu component.
 *
 * @see vpx.gui.MenuElement#getSubElements()
 */
_i.getSubElements = function() {
   return [];
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns this object.
 *
 * @see vpx.gui.MenuElement#getComponent()
 */
_i.getComponent = function() {
   return this;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.MenuItem]";
};
