/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class MenuSelectionManager
 * extends Object
 *
 * A MenuSelectionManager owns the selection in menu hierarchy.
 *
 * @version 1.0 (Oct 13, 2005)
 */

/**
 * Creates a new <code>MenuSelectionManager</code>.
 */
vpx.gui.MenuSelectionManager = function() {
   this.selection = [];
   this.listenerList = new vpx.core.event.EventListenerList();

   this.film = new vpx.gui.Film(vpx.getTle());
   this.addChangeListener(this._getHandler());
   this.film.addMouseListener(this._getHandler());
};

// Shorthand for brevity's sake
var _c = vpx.gui.MenuSelectionManager; // Class
var _i = _c.prototype;                 // Instance
_i._concrete = true;                   // vpx system flag for concrete classes

// Static class variables
_c.instance = null;                    // private static vpx.gui.MenuSelectionManager

// Instance variables
_i.selection = null;                   // private Array
_i.changeEvent   = null;               // protected transient vpx.core.event.ChangeEvent
_i.listenerList = null;                // protected vpx.core.event.EventListenerList
_i.film = null;                        // protected vpx.gui.Film
_i.handler = null;                     // protected vpx.gui.MenuSelectionManager.Handler

/**
 * Returns the default menu selection manager.
 *
 * @return vpx.gui.MenuSelectionManager
 *    A MenuSelectionManager object
 */
_c.defaultManager = function() {
   var c = vpx.gui.MenuSelectionManager;
   if (c.instance == null) {
      // First time reference to the class
      c._staticInit();
   }
   return c.instance;
};

/**
 * Changes the selection in the menu hierarchy.  The elements in the array are
 * sorted in order from the root menu element to the currently selected menu
 * element.
 * <p/>
 * Note that this method is public but is used by the look and feel engine and
 * should not be called by client applications.
 *
 * @param path vpx.gui.MenuElement[]
 *    An array of <code>MenuElement</code> objects specifying the selected path
 */
_i.setSelectedPath = function(path) {
   var i;
   var currentSelectionCount = this.selection.length;
   var firstDifference = 0;

   if (path == null) {
      path = [];
   }

   for (i = 0; i < path.length; i++) {
      if (i < currentSelectionCount && this.selection[i] == path[i]) {
         firstDifference++;
      } else {
         break;
      }
   }

   for (i = currentSelectionCount - 1; i >= firstDifference; i--) {
      var me = this.selection[i];
      this.selection.splice(i, 1);
      me.menuSelectionChanged(false);
   }

   for (i = firstDifference; i < path.length; i++) {
      if (path[i] != null) {
         this.selection.push(path[i]);
         path[i].menuSelectionChanged(true);
      }
   }

   this._fireStateChanged();
};

/**
 * Returns the path to the currently selected menu item
 *
 * @return vpx.gui.MenuElement[]
 *    An array of MenuElement objects representing the selected path
 */
_i.getSelectedPath = function() {
   var res = new Array(this.selection.length);
   for (var i = 0; i < this.selection.length; i++) {
      res[i] = this.selection[i];
   }
   return res;
};

/**
 * Tell the menu selection to close and unselect all the menu components. Call
 * this method when a choice has been made.
 */
_i.clearSelectedPath = function() {
   if (this.selection.length > 0) {
      this.setSelectedPath(null);
   }
};

/**
 * Gets this menu selection manager's <code>Film</code> element.
 *
 * @return vpx.gui.Film
 *    This menu selection manager's <code>Film</code> element
 */
_i.getFilm = function() {
   return this.film;
};

/**
 * Adds a <code>ChangeListener</code> to this manager.
 *
 * @param l vpx.core.event.ChangeListener
 *    The <code>ChangeListener</code> object
 */
_i.addChangeListener = function(l) {
   this.listenerList.add("ChangeListener", l);
};

/**
 * Removes a <code>ChangeListener</code> from this manager.
 *
 * @param l vpx.core.event.ChangeListener
 *    The <code>ChangeListener</code> object
 */
_i.removeChangeListener = function(l) {
   this.listenerList.remove("ChangeListener", l);
};

/**
 * Returns an array of all the <code>ChangeListener</code>s added to this
 * MenuSelectionManager with addChangeListener().
 *
 * @return vpx.core.event.ChangeListener[]
 *    All of this model's <code>ChangeListener</code>s, or an empty array if no
 *    change listeners are currently registered
 */
_i.getChangeListeners = function() {
   return this.listenerList.getListeners("ChangeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type. The event instance is created lazily.
 */
_i._fireStateChanged = function() {
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ChangeListener") {
         // Lazily create the event:
         if (this.changeEvent == null) {
            this.changeEvent = new vpx.core.event.ChangeEvent(this);
         }

         // listeners[i + 1] is of type vpx.core.event.ChangeListener
         listeners[i + 1].stateChanged(this.changeEvent);
      }
   }
};

/**
 * Gets the built-in event handler for this menu selection manager.
 *
 * @return vpx.gui.MenuSelectionManager.Handler
 *    The built-in event handler for this menu selection manager
 */
_i._getHandler = function() {
   if (this.handler == null) {
      this.handler = new vpx.gui.MenuSelectionManager.Handler(this);
   }
   return this.handler;
};

/**
 * class Handler
 * extends Object
 * implements vpx.core.event.ChangeListener, vpx.gui.event.MouseListener
 *
 * The internal menu selection event listener. This listener will manage the
 * visibility of the manager's <code>Film</code> component and will also
 * handle mouse clicks on the film to clear the manager's selection path.
 *
 * @version 1.0 (Oct 17, 2005)
 */

/**
 * Constructs a new Handler.
 *
 * @param mgr vpx.gui.MenuSelectionManager
 *    The friend selection manager that this listener represents
 */
_c.Handler = function(mgr) {
   this.mgr = mgr;
};

// Instance variables
_c.Handler.prototype.isSelected = false;

/**
 * Handles a state change. This monitors whether anything is selected and turns
 * on and off a film that is placed over the document to catch mouse clicks.
 * The film becomes active when there is a current selection and de-activates
 * itself when the selection is <code>null</code>. When active, the film
 * catches all mouse clicks that occur on the open document and clears the
 * selection when any click occurs (thus automatically de-activating the film
 * as well).
 *
 * @param e vpx.core.event.ChangeEvent
 *    The <code>ChangeEvent</code> object
 * @see vpx.core.event.ChangeListener#stateChanged(vpx.core.event.ChangeEvent)
 */
_c.Handler.prototype.stateChanged = function(e) {
   var mgrSelected = (this.mgr.selection.length > 0);

   if (mgrSelected != this.isSelected) {
      this.mgr.film.setVisible(mgrSelected);
      this.isSelected = mgrSelected;
   }
};

/**
 * Listens on this selection manager's film.  This clears the manager's
 * selected path whenever the film receives a click event.
 *
 * @see vpx.gui.event.MouseListener#mouseClicked(vpx.gui.event.MouseEvent)
 */
_c.Handler.prototype.mouseClicked = function(e) {
   this.mgr.clearSelectedPath();
};

/**
 * Listens on this selection manager's film.  Does nothing.
 *
 * @see vpx.gui.event.MouseListener#mouseClicked(vpx.gui.event.MouseEvent)
 */
_c.Handler.prototype.mousePressed = function(e) {
   // Do nothing
};

/**
 * Listens on this selection manager's film.  Does nothing.
 *
 * @see vpx.gui.event.MouseListener#mouseClicked(vpx.gui.event.MouseEvent)
 */
_c.Handler.prototype.mouseReleased = function(e) {
   // Do nothing
};

/**
 * Listens on this selection manager's film.  Does nothing.
 *
 * @see vpx.gui.event.MouseListener#mouseClicked(vpx.gui.event.MouseEvent)
 */
_c.Handler.prototype.mouseEntered = function(e) {
   // Do nothing
};

/**
 * Listens on this selection manager's film.  Does nothing.
 *
 * @see vpx.gui.event.MouseListener#mouseClicked(vpx.gui.event.MouseEvent)
 */
_c.Handler.prototype.mouseExited = function(e) {
   // Do nothing
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Static initializer block.  Sets up a singleton manager instance and
 * registers a change listener on it to handle docment film (to catch mouse
 * clicks).
 */
_c._staticInit = function() {
   var c = vpx.gui.MenuSelectionManager;
   c.instance = new c();
};
