/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class PopupMenu
 * extends vpx.gui.Component
 * implements vpx.gui.MenuElement
 *
 * An implementation of a popup menu -- a small window that pops up and
 * displays a series of choices. A <code>PopupMenu</code> is used for the menu
 * that appears when the user selects an item on the menu bar. It is also used
 * for "pull-right" menu that appears when the user selects a menu item that
 * activates it. Finally, a <code>PopupMenu</code> can also be used anywhere
 * else you want a menu to appear. For example, when the user right-clicks in a
 * specified area.
 * <p/>
 * VPX Menus are based on Java Swing menus.  For information and examples of
 * using Swing JMenus, see
 * <a href="http://java.sun.com/doc/books/tutorial/uiswing/components/menu.html">How to Use Menus</a>,
 * a section in <em>The Java Tutorial.</em>
 *
 * @version 1.0 (Sep 29, 2005)
 */

/**
 * Creates a new popup menu
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the popup menu
 */
vpx.gui.PopupMenu = function(view) {
   // super(view)
   vpx.gui.Component.call(this, view);

   this.setSelectionModel(new vpx.gui.DefaultSingleSelectionModel());
   this._initUI();
   this.setLocation(0, 0);
};

// PopupMenu extends vpx.gui.Component
vpx.gui.PopupMenu.prototype = new vpx.gui.Component(vpx.ABSTRACT_PASS);
vpx.gui.PopupMenu.prototype.constructor = vpx.gui.PopupMenu;

// Shorthand for brevity's sake
var _c = vpx.gui.PopupMenu;           // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes
_i._IMPL_vpx_gui_MenuElement = true;  // vpx system flag for interface impl

// Instance variables
_i.uiClassID = "PopupMenuUI";         // private static final String
_i.visible = false;                   // protected boolean
_i.invoker = null;                    // protected transient Component
_i.invokerBasePos = null;             // protected transient vpx.gui.Point
_i.selectionModel = null;             // private vpx.gui.SingleSelectionModel
_i.desiredLocationX = null;           // private int
_i.desiredLocationY = null;           // private int

// Class constants

/** Identifies a change in the component's location. */
_c.LOCATION_PROPERTY = "location";

/**
 *
 * @param c vpx.gui.Component
 *    ...
 * @return vpx.gui.Component
 *    ...
 */
_i.add = function(c) {
   // super.add(c)
   var spr = vpx.gui.Component.prototype;
   spr.add.call(this, c);
   return c;
};

/**
 *
 */
_i.addSeparator = function() {
   this.add(new vpx.gui.Separator(this.getView()));
};

/**
 * Returns the index of the specified component.
 *
 * @param c vpx.gui.Component
 *    The <code>Component</code> to find
 * @return int
 *    The index of the component, where 0 is the first; or -1 if the component
 *    is not found
 */
_i.getComponentIndex = function(c) {
   var count = this.getComponentCount();
   var a = this.getComponents();
   for (var i = 0; i < count; i++) {
      if (a[i] == c) {
         return i;
      }
   }
   return -1;
};

/**
 *
 * @param
 *    ...
 * @return void
 *    ...
 */
_i.pack = function() {
   if(this.popup != null) {
      var pref = this.getPreferredSize();

      if (pref == null || pref.width != this.getWidth() ||
          pref.height != this.getHeight()) {
         this.popup = this.getPopup();
      } else {
         this.validate();
      }
   }
};

/**
 *
 * @param component vpx.gui.Component
 *    ...
 * @param index int
 *    ...
 */
_i.insert = function(component, index) {
   var i;

   if (index < 0) {
      throw new Error("vpx.gui.PopupMenu#insert(): index less than zero");
   }

   var nItems = this.getComponentCount();
   var tempItems = [];

   /* Remove the item at index, nitems-index times
      storing them in a temporary vector in the
      order they appear on the menu.
   */
   for (i = index; i < nItems; i++) {
      tempItems.push(this.getComponentAt(index));
      this.removeAt(index);
   }

   this.add(component);

   /* Add the removed items back to the menu, they are
      already in the correct order in the temp vector.
   */
   for (i = 0; i < tempItems.length; i++) {
      this.add(tempItems[i]);
   }
};

/**
 *
 * @param pos int
 *    ...
 */
_i.removeAt = function(pos) {
   if (pos < 0) {
      throw new Error("vpx.gui.PopupMenu#insert(): index less than zero");
   }
   if (pos > this.getComponentCount() -1) {
      throw new Error("vpx.gui.PopupMenu#insert(): index greater than the number of items");
   }
   // super.removeAt(pos)
   var spr = vpx.gui.Component.prototype;
   spr.removeAt.call(this, pos);
};

/**
 *
 * @param x int
 *    ...
 * @param y int
 *    ...
 */
_i.setLocation = function(x, y) {
   var oldX = this.desiredLocationX;
   var oldY = this.desiredLocationY;

   this.desiredLocationX = x;
   this.desiredLocationY = y;

   if(x != oldX || y != oldY) {
      var oldPoint = new vpx.gui.Point(oldX, oldY);
      var newPoint = new vpx.gui.Point(x, y);
      var c = vpx.gui.PopupMenu;
      this.firePropertyChange(c.LOCATION_PROPERTY, oldPoint, newPoint);
   }
};

/**
 * Returns the component which is the 'invoker' of this
 * popup menu.
 *
 * @return vpx.gui.Component
 *    The <code>Component</code> in which the popup menu is displayed
 */
_i.getInvoker = function() {
   return this.invoker;
};

/**
 * Sets the invoker of this popup menu -- the component in which the popup menu
 * menu is to be displayed.
 *
 * @param invoker vpx.gui.Component
 *    The <code>Component</code> in which the popup menu is displayed
 */
_i.setInvoker = function(invoker) {
   var oldInvoker = this.invoker;
   this.invoker = invoker;

   if (oldInvoker != this.invoker) {
      //?? XXX What if the document body has not yet fully loaded?
      this.getView().document.body.appendChild(this.getPeer());
      this.invokerBasePos = invoker.getLocation(this.getView());
      delete this.desiredLocationX;
      delete this.desiredLocationY;
   }
};

/**
 * Displays the popup menu at the position x,y in the coordinate space of the
 * component invoker.
 *
 * @param invoker vpx.gui.Component
 *    The component in whose space the popup menu is to appear
 * @param x int
 *    The x coordinate in invoker's coordinate space at which the popup menu is
 *    to be displayed
 * @param y int
 *    The y coordinate in invoker's coordinate space at which the popup menu is
 *    to be displayed
 */
_i.show = function(invoker, x, y) {
   this.setInvoker(invoker);
   this.setLocation(x, y);
   this.setVisible(true);
};

/**
 *
 * @param d vpx.gui.Dimension
 *    ...
 */
_i.setPopupSize = function(d) {
   var oldSize = this.getPreferredSize();

   this.setPreferredSize(d);
   if (this.popup != null) {
      var newSize = this.getPreferredSize();

      if (!oldSize.equals(newSize)) {
         this.popup = this.getPopup();
      }
   }
};

/**
 * Sets the currently selected component,  This will result in a change to the
 * selection model.
 *
 * @param sel vpx.gui.Component
 *    The <code>Component</code> to select
 */
_i.setSelected = function(sel) {
   var model = this.getSelectionModel();
   var i = this.getComponentIndex(sel);
   model.setSelectedIndex(i);
};

/**
 * Returns the model object that handles single selections.
 *
 * @return vpx.gui.SingleSelectionModel
 *    The <code>selectionModel</code> property
 */
_i.getSelectionModel = function() {
   return this.selectionModel;
};

/**
 * Sets the model object to handle single selections.
 *
 * @param model vpx.gui.SingleSelectionModel
 *    The new <code>SingleSelectionModel</code>
 */
_i.setSelectionModel = function(model) {
   this.selectionModel = model;
};

/**
 *
 * @param b boolean
 *    ...
 * @return void
 *    ...
 */
_i.setVisible = function(b) {
   // Is it a no-op?
   if (b == this.visible) {
      return;
   }

/*
   // if closing, first close all Submenus
   if (b == false) {

      // 4234793: This is a workaround because JPopupMenu.firePopupMenuCanceled is
      // a protected method and cannot be called from BasicPopupMenuUI directly
      // The real solution could be to make
      // firePopupMenuCanceled public and call it directly.
      var doCanceled = getClientProperty("JPopupMenu.firePopupMenuCanceled");
      if (doCanceled != null && doCanceled == Boolean.TRUE) {
         putClientProperty("JPopupMenu.firePopupMenuCanceled", Boolean.FALSE);
         firePopupMenuCanceled();
      }
      getSelectionModel().clearSelection();

   } else {
      // This is a popup menu with MenuElement children,
      // set selection path before popping up!
      if (_isPopupMenu()) {
         if (getSubElements().length > 0) {
            MenuElement me[] = new MenuElement[2];
            me[0]=(MenuElement)this;
            me[1]=getSubElements()[0];
            MenuSelectionManager.defaultManager().setSelectedPath(me);
         } else {
            MenuElement me[] = new MenuElement[1];
            me[0]=(MenuElement)this;
            MenuSelectionManager.defaultManager().setSelectedPath(me);
         }
      }
   }*/

   var cls = vpx.gui.Component;
   if(b) {
      //??firePopupMenuWillBecomeVisible();
      this.visible = true;
      this.firePropertyChange(cls.VISIBILITY_PROPERTY, false, true);
   } else {
      //??firePopupMenuWillBecomeInvisible();
      //??popup.hide();
      this.visible = false;
      this.firePropertyChange(cls.VISIBILITY_PROPERTY, true, false);
      // 4694797: When popup menu is made invisible, selected path
      // should be cleared
      //if (_isPopupMenu()) {
      //   MenuSelectionManager.defaultManager().clearSelectedPath();
      //}
   }
};

/**
 * Implemented to be a <code>MenuElement</code>. Does nothing.
 *
 * @see vpx.gui.MenuElement#processMouseEvent(MouseEvent, vpx.gui.MenuElement[], vpx.gui.MenuSelectionManager)
 */
_i.processMouseEvent = function(event, path, manager) {
   // Do nothing
};

/**
 * Implemented to be a <code>MenuElement</code>. Messaged when the menubar
 * selection changes to activate or deactivate this menu.
 *
 * @see vpx.gui.MenuElement#menuSelectionChanged(boolean)
 */
_i.menuSelectionChanged = function(isIncluded) {
   if (this.invoker != null && this.invoker._IMPL_vpx_gui_PopupInvoker) {
      var m = this.invoker;
      if (isIncluded) {
         m.setPopupMenuVisible(true);
      } else {
         m.setPopupMenuVisible(false);
      }
   }
   if (this._isPopupMenu() && !isIncluded) {
      this.setVisible(false);
   }
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns an array of
 * <code>MenuElement</code>s containing the submenu for this menu component.
 * It will only return items conforming to the <code>MenuElement</code>
 * interface. If popup menu is <code>null</code>, returns an empty array.
 *
 * @see vpx.gui.MenuElement#getSubElements()
 */
_i.getSubElements = function() {
   var res = [];
   var c = this.getComponentCount();

   for (var i = 0; i < c; i++) {
      var m = this.getComponentAt(i);
      if (m._IMPL_vpx_gui_MenuElement) {
         res.push(m);
      }
   }

   return res;
};

/**
 * Implemented to be a <code>MenuElement</code>. Returns this object.
 *
 * @see vpx.gui.MenuElement#getComponent()
 */
_i.getComponent = function() {
   return this;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.PopupMenu]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Returns true if the popup menu is a standalone popup menu rather than the
 * submenu of a <code>Menu</code>.
 *
 * @return boolean
 *    true if this menu is a standalone popup menu, otherwise false
 */
_i._isPopupMenu = function() {
   return  (this.invoker != null && !this.invoker._IMPL_vpx_gui_PopupInvoker);
};
