/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * public class SplitButton
 * extends vpx.gui.AbstractButton
 *
 * An implementation of a "split" button.  Split buttons are push buttons that
 * have both a standard push button section and a pulldown menu section.  If
 * the user clicks on the main section, the button's action will fire.  If the
 * user clicks on the pulldown section, the button's menu will be activated.
 * <p/>
 * VPX Buttons are based on Java Swing buttons. For information and examples of
 * using Swing buttons, see
 * <a href="http://java.sun.com/docs/books/tutorial/uiswing/components/button.html">How to Use Buttons, Check Boxes, and Radio Buttons</a>,
 * a section in <em>The Java Tutorial.</em>
 *
 * @version 1.0 (Apr 17, 2006)
 */

/**
 * Creates a split button with initial text and an icon.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the button
 * @param text String
 *    The text of the button
 * @param icon vpx.gui.Icon
 *    The Icon image to display on the button
 * @param model vpx.gui.ButtonModel
 *    The button's button model. If unspecified, this will default to a new
 *    <code>DefaultButtonModel</code>
 */
vpx.gui.SplitButton = function SplitButton(view, text, icon, model) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super(view)
   vpx.gui.AbstractButton.call(this, view);

   // Create the model
   if (arguments.length < 4 || model == null) {
      model = new vpx.gui.DefaultButtonModel();
   }
   this.setModel(model);

   // Create the associated MenuButton
   var mbIcon = new vpx.gui.ImageIcon(view, "splitButtonPulldownArrow", null);
   this.menuButton = new vpx.gui.MenuButton(view, null, mbIcon);
   this.menuButton.setAttribute("splitButton", this);

   this.setRolloverEnabled(true);
   this.menuButton.setRolloverEnabled(true);

   // initialize
   this._initUI();
   this._init(text, icon);
};

// SplitButton extends vpx.gui.AbstractButton
vpx.gui.SplitButton.prototype = new vpx.gui.AbstractButton(vpx.ABSTRACT_PASS);
vpx.gui.SplitButton.prototype.constructor = vpx.gui.SplitButton;

// Shorthand for brevity's sake
var _c = vpx.gui.SplitButton;          // Class
var _i = _c.prototype;                 // Instance
_i._concrete = true;                   // vpx system flag for concrete classes

// Instance variables
_i.uiClassID = "SplitButtonUI";        // private static final String
_i.menuButton = null;                  // private vpx.gui.MenuButton

/**
 * Appends a menu item to the end of this button.  Returns the menu item added.
 *
 * @param menuItem vpx.gui.MenuItem
 *    The <code>Menuitem</code> to be added
 * @return vpx.gui.MenuItem
 *    The <code>MenuItem</code> added
 */
_i.add = function(menuItem) {
   return this.menuButton.add(menuItem);
};

/**
 * Appends a new separator to the end of the button.
 */
_i.addSeparator = function() {
    this.menuButton.addSeparator();
};

/**
 * Inserts the specified <code>Menuitem</code> at a given position.
 *
 * @param mi vpx.gui.MenuItem
 *    The <code>Menuitem</code> to add
 * @param pos int
 *    An integer specifying the position at which to add the new
 *    <code>Menuitem</code>
 * @return vpx.gui.MenuItem
 *    The new menu item
 * @exception Error
 *    If the value of <code>pos</code> < 0
 */
_i.insert = function(mi, pos) {
   return this.menuButton.insert(mi, pos);
};

/**
 * Inserts a separator at the specified position.
 *
 * @param index int
 *    An integer specifying the position at which to insert the menu separator
 * @exception Error
 *    If the value of <code>index</code> < 0
 */
_i.insertSeparator = function(index) {
   this.menuButton.insertSeparator(index);
};

/**
 * Removes the menu item at the specified index from this button.
 *
 * @param pos int
 *    The position of the item to be removed
 * @exception Error
 *    If the value of <code>pos</code> < 0, or if <code>pos</code> is greater
 *    than the number of menu items
 */
_i.removeAt = function(pos) {
   this.menuButton.removeAt(pos);
};

/**
 * Removes the specified menu item from this button.  If there is no popup
 * menu, this method will have no effect.
 *
 * @param item vpx.gui.MenuItem
 *    The <code>MenuItem</code> to be removed from the menu
 */
_i.remove = function(item) {
   this.menuButton.remove(item);
};

/**
 * Removes all menu items from this button.
 */
_i.removeAll = function() {
   this.menuButton.removeAll();
};

/**
 * Gets this button's built-in menu button.
 *
 * @return vpx.gui.MenuButton
 *    This button's built-in menu button
 */
_i.getMenuButton = function() {
   return this.menuButton;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.gui.SplitButton]";
};
