/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class ToggleButton
 * extends vpx.gui.AbstractButton
 *
 * An implementation of a two-state button. The <code>RadioButton</code> and
 * <code>CheckBox</code> classes (yet to be implemented) are subclasses of this
 * class.
 * <p/>
 * VPX GUI is based on Java Swing.  For information and examples of using Swing
 * toggle buttons, see
 * <a href="http://java.sun.com/docs/books/tutorial/uiswing/components/button.html">How to Use Buttons, Check Boxes, and Radio Buttons</a>,
 * a section in <em>The Java Tutorial</em>.
 *
 * @version 1.0 (Oct 14, 2005)
 */

/**
 * Creates a toggle button with the specified text, image, and selection state.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the button
 * @param text String
 *    The text of the toggle button
 * @param icon vpx.gui.Icon
 *    The image that the button should display
 * @param selected boolean
 *    if true, the button is initially selected; otherwise, the button is
 *    initially unselected
 */
vpx.gui.ToggleButton = function(view, text, icon, selected) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super(view)
   vpx.gui.AbstractButton.call(this, view);

   // Create the model
   this.setModel(new vpx.gui.ToggleButton.ToggleButtonModel());

   this.model.setSelected(selected);

   // initialize
   this._initUI();
   this._init(text, icon);
};

// ToggleButton extends vpx.gui.AbstractButton
vpx.gui.ToggleButton.prototype = new vpx.gui.AbstractButton(vpx.ABSTRACT_PASS);
vpx.gui.ToggleButton.prototype.constructor = vpx.gui.ToggleButton;

// Shorthand for brevity's sake
var _c = vpx.gui.ToggleButton;        // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.uiClassID = "ToggleButtonUI";      // private static final String

/**
 * public static class ToggleButtonModel
 * extends vpx.gui.DefaultButtonModel
 *
 * The ToggleButton model
 *
 * @version 1.0 (Oct 14, 2005)
 */

/**
 * Constructs a new ToggleButton Model.
 */
_c.ToggleButtonModel = function() {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super()
   vpx.gui.DefaultButtonModel.call(this);
};

// ToggleButtonModel extends vpx.gui.DefaultButtonModel
_c.ToggleButtonModel.prototype = new vpx.gui.DefaultButtonModel(vpx.ABSTRACT_PASS);
_c.ToggleButtonModel.prototype.constructor = _c.ToggleButtonModel;

/**
 * Sets the selected state of the button.
 *
 * @param b boolean
 *    true selects the toggle button, false deselects the toggle button.
 */
_c.ToggleButtonModel.prototype.setSelected = function(b) {
   var c;

   var group = this.getGroup();
   if (group != null) {
      // Use the group model instead
      group.setSelected(this, b);
      b = group.isSelected(this);
   }

   if (this.isSelected() == b) {
      return;
   }

   c = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= c.SELECTED;
   } else {
      this.stateMask &= ~c.SELECTED;
   }

   // Send ChangeEvent
   this._fireStateChanged();

   // Send ItemEvent
   c = vpx.gui.event.ItemEvent;
   var event = new c(this, c.ITEM_STATE_CHANGED,
                     this, this.isSelected() ?  c.SELECTED : c.DESELECTED);
   this._fireItemStateChanged(event);
};

/**
 * Sets the pressed state of the toggle button.
 *
 * @param b boolean
 *    true to press the button; false to unpress it
 */
_c.ToggleButtonModel.prototype.setPressed = function(b) {
   if ((this.isPressed() == b) || !this.isEnabled()) {
      return;
   }

   if (b == false && this.isArmed()) {
      this.setSelected(!this.isSelected());
   }

   var c = vpx.gui.DefaultButtonModel;
   if (b) {
      this.stateMask |= c.PRESSED;
   } else {
      this.stateMask &= ~c.PRESSED;
   }

   this._fireStateChanged();

   if(!this.isPressed() && this.isArmed()) {
      var ActionEvent = vpx.gui.event.ActionEvent;
      var event = new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
                                  this.getActionCommand());
      this._fireActionPerformed(event);
   }
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_c.ToggleButtonModel.prototype.toString = function() {
   return "[Object vpx.gui.ToggleButton.ToggleButtonModel]";
};
