/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 *
 */
vpx.gui.digest.Digester = function() {
};

// Shorthand for brevity's sake
var _c = vpx.gui.digest.Digester;     // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

/**
 * Parses the given url according to the standard VPX GUI XML contract and
 * creates GUI objects as necessary.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @param spec vpx.net.XmlSpec
 *    The xml spec from which to request the XML data
 * @return vpx.gui.Component[]
 *    The components that were created, in the order they were created
 */
_i.parse = function(view, spec) {
   var res = [];

   var req = tle.getRequest(spec.toUrl());
   req.setAsynchronous(false);

   try {
      req.send();
      var resp = req.getResponse();

      var contentType = resp.getContentType();
      if (contentType != "text/xml") {
        throw new Error("Digester#parse(" + spec + "): bad content-type: " + contentType);
     }
      var status = resp.getStatus();
      // Only proceed if HTTP status is "OK"
      if (status == 200) {
         var xml = resp.getXml();
         var root = xml.documentElement;
         if (root == null) {
            // XML was not well-formed
            throw new Error("Digester#parse(): xml was not well-formed " +
                           "while processing " + spec.toUrl());
         }
         root.normalize();

         var vars = this._processVars(xml, root, view);
         res = res.concat(this._processMenuBars(xml, root, view, vars));
         res = res.concat(this._processComponents(xml, root, view, vars));
         delete vars;
      }
   } finally {
      tle.releaseRequest(req);
      return res;
   }
};

/**
 * Gets the variables that were defined in the XML.
 *
 * @param xml [DOM]Document
 *    The XML document returned from the XMLHttpRequest
 * @param root [DOM]Element
 *    The root element of the XML document
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @return Hashtable
 *    Objects, keyed by variable name
 */
_i._processVars = function(xml, root, view) {
   var res = {};
   var nodes = root.getElementsByTagName("var");
   for (var i = 0; i < nodes.length; i++) {
      var node = nodes[i];
      var name = node.getAttribute("name");
      if (name == null || name == "") {
         vpx.log.warn("Digester#_processVars(): invalid node: no 'name'");
         continue;
      }

      try {
         res[name] = this._resolveVar(view, node);
      } catch (ex) {
         vpx.log.warn(ex.message);
         continue;
      }
   }
   return res;
};

/**
 * Resolves an xml node describing a variable to the actual variable itself.
 *
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the variable
 * @return Object
 *    The variable to which the given node resolves
 */
_i._resolveVar = function(view, el) {
   if (el == null || el.firstChild == null) {
      throw new Error("Digester#_resolveVar(): invalid xml node");
   }
   var str = el.firstChild.nodeValue;
   var result;
   switch (el.getAttribute("type")) {

   case "jsVar":
      try {
         result = eval("view." + str);
      } catch (ex) {
         throw new Error("Digester#_resolveVar(): var not found: " + str);
      }
      break;

   case "docId":
      result = view.document.getElementById(str);
      if (!result) {
         throw new Error("Digester#_resolveVar(): invalid doc id: " + str);
      }
      break;

   case "eval":
      try {
         result = eval(str);
      } catch (ex) {
         throw new Error("Digester#_resolveVar(): eval failed: " + str);
      }
      break;

   case "string":
      result = str;
      break;

   case "boolean":
      result = str.toBool();
      break;

   default:
      throw new Error("Digester#_resolveVar(): Unrecognized type");
      break;

   }
   return result;
};

/**
 * Resolves a fully qualified class string & its scope into a class reference.
 *
 * @param localScope [DOM Level 2 Views]AbstractView
 *    The context in which GUI objects are being created
 * @param fqn String
 *    The fully qualified class name
 * @return Class (Function)
 *    The desired class object
 */
_i._resolveClass = function(localScope, fqn) {
   try {
      return eval("localScope." + fqn);
   } catch (ex) {
      // Ignore
   }

   try {
      return eval(fqn);
   } catch (ex) {
      throw new Error("Digester#_resolveClass(): Class not found: " + fqn);
   }
};

/**
 * Processes the XML for menu bars, creating them as necessary.
 *
 * @param xml [DOM]Document
 *    The XML document returned from the XMLHttpRequest
 * @param root [DOM]Element
 *    The root element of the XML document
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @param vars Hashtable
 *    Objects, keyed by variable name
 * @return vpx.gui.MenuBar[]
 *    The menu bars that were created
 */
_i._processMenuBars = function(xml, root, view, vars) {
   var ret = [];

   var barNodes = root.getElementsByTagName("menubar");
   for (var i = 0; i < barNodes.length; i++) {
      var barNode = barNodes[i];

      // Validate the menubar container
      var cntnr = barNode.getAttribute("container");
      if (!vars[cntnr]) {
         vpx.log.warn("Digester#_processMenuBars(): no cntnr: " + cntnr);
         continue;
      }

      // Instantiate the menubar
      var bar = new vpx.gui.MenuBar(view);
      vars[cntnr].appendChild(bar.getPeer());
      ret.push(bar);

      this._addAttributes(bar, barNode);

      // Add the menus to the bar
      var menuNodes = barNode.getElementsByTagName("menu");
      for (var j = 0; j < menuNodes.length; j++) {
         var menuNode = menuNodes[j];
         var menu = new vpx.gui.Menu(view, menuNode.getAttribute("label"));
         bar.add(menu);

         this._addAttributes(menu, menuNode);
         this._addMenuItems(menu, menuNode, vars);
      }
   }

   return ret;
};

/**
 * Processes the XML for gui components, creating them as necessary.
 *
 * @param xml [DOM]Document
 *    The XML document returned from the XMLHttpRequest
 * @param root [DOM]Element
 *    The root element of the XML document
 * @param view [DOM Level 2 Views]AbstractView
 *    The context in which to create the GUI objects
 * @param vars Hashtable
 *    Objects, keyed by variable name
 * @return vpx.gui.Component[]
 *    The components that were created
 */
_i._processComponents = function(xml, root, view, vars) {
   var ret = [];

   var nodes = root.getElementsByTagName("component");
   for (var i = 0; i < nodes.length; i++) {
      var node = nodes[i];

      // Validate the container
      var cntnr = node.getAttribute("container");
      if (!vars[cntnr]) {
         vpx.log.warn("Digester#_processComponents(): no cntnr: " + cntnr);
         continue;
      }

      // Instantiate the component
      try {
         var c = this._resolveClass(view, node.getAttribute("class"));
      } catch (ex) {
         vpx.log.warn(ex.message);
         continue;
      }
      var cNodes = node.getElementsByTagName("constructor");
      if (cNodes == null || cNodes.length == 0) {
         vpx.log.warn("Digester#_processComponents(): No constructor node found");
         continue;
      }
      if (cNodes.length > 1) {
         vpx.log.warn("Digester#_processComponents(): More than one constructor node found");
         continue;
      }
      var args = [];
      for (var j = 0; j < cNodes[0].childNodes.length; j++) {
         var argNode = cNodes[0].childNodes[j];
         if (argNode.nodeType != vpx.xua.ELEMENT_NODE) {
            continue;
         }
         args.push(this._resolveVar(view, argNode));
      }
      var stmt = "new c(";
      for (j = 0; j < args.length; j++) {
         if (j > 0) {
            stmt += ",";
         }
         stmt += "args[" + j + "]";
      }
      stmt += ")";
      var comp = eval(stmt);
      vars[cntnr].appendChild(comp.getPeer());
      ret.push(comp);

      this._addAttributes(comp, node);
      if (comp instanceof vpx.gui.AbstractButton) {
         this._setText(comp, node);
         this._setIcon(comp, node);
         this._setEnabled(comp, node);
         this._setActionCommand(comp, node);
         this._addActionListener(comp, node, vars);
      }
      if (comp instanceof vpx.gui.Menu || comp instanceof vpx.gui.SplitButton) {
         this._addMenuItems(comp, node, vars);
      }
   }

   return ret;
};

/*
 * Sets the text of an abstract button given a node containing attributes of
 * the button.  If the node does not contain a text attribute, the text of the
 * button will be left as is.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the button
 */
_i._setText = function(b, el) {
   var txt = el.getAttribute("text");
   if (txt != null && txt != "") {
      b.setText(txt);
   }
};

/*
 * Sets the icon of an abstract button given a node containing the definition
 * of the icon.  If the node does not contain any child element that defines an
 * icon, the button's icon will be left as is.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param node [DOM Level 1 Core]Node
 *    The xml node whose children may contain an icon definition node
 */
_i._setIcon = function(b, node) {
   var iconNodes = node.getElementsByTagName("icon");
   if (iconNodes == null || iconNodes.length == 0) {
      return;
   }
   if (iconNodes.length > 1) {
      throw new Error("Digester#_setIcon(): Only one icon node is allowed per button");
   }
   var iNode = iconNodes[0];

   var icon;
   var type = iNode.getAttribute("type");
   switch (type) {

   case "image":
      var className = iNode.getAttribute("class");
      var alt = iNode.getAttribute("alt");
      icon = new vpx.gui.ImageIcon(b.getView(), className, alt);
      break;

   default:
      throw new Error("Digester#_setIcon(): Unsupported icon type: " + type);
      break;

   }

   b.setIcon(icon);
};

/*
 * Sets the enabled state of a button given a node containing attributes of
 * the button.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the button
 */
_i._setEnabled = function(b, el) {
   var enabled = true;
   var enabledStr = el.getAttribute("enabled");
   if (enabledStr != null && enabledStr != "") {
      enabled = enabledStr.toBool();
   }
   b.setEnabled(enabled);
};

/*
 * Sets an abstract button's action command given a node containing attributes
 * of the button.  If the node does not contain an action command attribute,
 * no action command will be set.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the button
 */
_i._setActionCommand = function(b, el) {
   var cmd = el.getAttribute("actionCmd");
   if (cmd != null && cmd != "") {
      b.setActionCommand(cmd);
   }
};

/**
 * Binds the list of attributes to a component given a node containing the list
 * of attributes.  If the node does not contain any attribute nodes, the
 * component will not have any attributes bound to it.
 *
 * @param c vpx.gui.Component
 *    The component
 * @param node [DOM Level 1 Core]Node
 *    The xml node that specifies the attributes of the component
 */
_i._addAttributes = function(c, node) {
   var attrsNodes = node.getElementsByTagName("attrs");
   if (attrsNodes == null || attrsNodes.length == 0) {
      return;
   }

   var attrNodes = attrsNodes[0].getElementsByTagName("attr");
   if (attrNodes == null || attrNodes.length == 0) {
      return;
   }

   for (var i = 0; i < attrNodes.length; i++) {
      var nd = attrNodes[i];
      c.setAttribute(nd.getAttribute("name"), nd.getAttribute("value"));
   }
};

/*
 * Adds an action listener to an abstract button given a node containing
 * attributes of the button.  If the node does not contain an action listener
 * attribute, no action listener will be added.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the button
 * @param vars Hashtable
 *    Objects, keyed by variable name
 */
_i._addActionListener = function(b, el, vars) {
   var lstnr = el.getAttribute("actionListener");
   if (lstnr != null && lstnr != "") {
      if (vars[lstnr]) {
         b.addActionListener(vars[lstnr]);
      } else {
         vpx.log.warn("Digester#_addActionListener(): no lstnr: " + lstnr);
      }
   }
};

/*
 * Adds an item listener to an abstract button given a node containing
 * attributes of the button.  If the node does not contain an item listener
 * attribute, no item listener will be added.
 *
 * @param b vpx.gui.AbstractButton
 *    The button
 * @param el [DOM Level 1 Core]Element
 *    The xml node that specifies the attributes of the button
 * @param vars Hashtable
 *    Objects, keyed by variable name
 */
_i._addItemListener = function(b, el, vars) {
   var lstnr = el.getAttribute("itemListener");
   if (lstnr != null && lstnr != "") {
      if (vars[lstnr]) {
         b.addItemListener(vars[lstnr]);
      } else {
         vpx.log.warn("Digester#_addItemListener(): no lstnr: " + lstnr);
      }
   }
};

/*
 * Adds menu items to a menu given an xml node containing definitions of the
 * menu items.  If the node does not contain any child elements that define
 * menu items, no menu items will be added.
 *
 * @param menu vpx.gui.Menu
 *    The menu
 * @param node [DOM Level 1 Core]Node
 *    The xml node whose children specify the menu items
 * @param vars Hashtable
 *    Objects, keyed by variable name
 */
_i._addMenuItems = function(menu, node, vars) {
   var itemNodes = node.getElementsByTagName("items");
   if (itemNodes == null || itemNodes.length == 0) {
      return;
   }
   if (itemNodes.length > 1) {
      throw new Error("Digester#_addMenuItems(): Only one items node allowed per menu");
   }

   var iNode = itemNodes[0];
   var label, item;
   var tle = vpx.getTle();

   for (var i = 0; i < iNode.childNodes.length; i++) {
      var subnode = iNode.childNodes[i];
      if (subnode.nodeType != vpx.xua.ELEMENT_NODE) {
         continue;
      }

      var tag = subnode.nodeName;
      switch (tag) {

      case "item":
         label = subnode.getAttribute("label");
         item = new vpx.gui.MenuItem(tle, label, null /* icon */);
         this._setEnabled(item, subnode);
         this._setActionCommand(item, subnode);
         this._addActionListener(item, subnode, vars);
         menu.add(item);
         break;

      case "separator":
         menu.addSeparator();
         break;

      case "radiogroup":
         var group = new vpx.gui.ButtonGroup();
         var radioNodes = subnode.getElementsByTagName("radiobutton");
         for (var l = 0; l < radioNodes.length; l++) {
            var radioNode = radioNodes[l];
            label = radioNode.getAttribute("label");
            var icon = null; /* TODO */
            var selected = false;
            var selectedStr = radioNode.getAttribute("selected");
            if (selectedStr != null && selectedStr != "") {
               selected = selectedStr.toBool();
            }
            item = new vpx.gui.RadioButtonMenuItem(tle, label, icon, selected);
            group.add(item);
            this._setActionCommand(item, radioNode);
            this._setEnabled(item, radioNode);
            this._addItemListener(item, radioNode, vars);
            menu.add(item);
         }
         break;

      case "header":
         label = subnode.getAttribute("label");
         item = new vpx.gui.HeaderMenuItem(tle, label, null /* icon */);
         menu.add(item);
         break;

      default:
         throw new Error("Unsupported menu item type: " + tag);
         break;

      }
   }
};
