/* Copyright 2006 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class ComponentBindingEvent
 * extends vpx.core.Event
 *
 * A semantic event which indicates that an object was bound or unbound to a
 * component. This high-level event is generated by a Component when an
 * attribute is added, removed, or changed. The event is passed to every
 * <code>ComponentAttributeListener</code> object that registered to receive
 * such events using the component's <code>addAttributeListener</code> method.
 */

/**
 * Constructs a <code>ComponentBindingEvent</code> object.
 * <p/>
 * Note that passing in an invalid <code>id</code> results in unspecified
 * behavior.
 *
 * @param source vpx.gui.Component
 *    The <code>Component</code> object that originated the event
 * @param id int
 *    An integer that identifies the event type
 * @param name String
 *    The name with which the object has been bound or unbound
 * @param value Object
 *    The value of the attribute that has been added, removed or replaced. If
 *    the attribute was added (or bound), this is the value of the attribute.
 *    If the attribute was removed (or unbound), this is the value of the
 *    removed attribute. If the attribute was replaced, this is the old value
 *    of the attribute
 */
vpx.gui.event.ComponentBindingEvent = function(source, id, name, value) {
   vpx.core.Event.call(this, source, id);
   this.name = name;
   this.value = value;
};

// ComponentBindingEvent extends vpx.core.Event
vpx.gui.event.ComponentBindingEvent.prototype = new vpx.core.Event(vpx.ABSTRACT_PASS);
vpx.gui.event.ComponentBindingEvent.prototype.constructor = vpx.gui.event.ComponentBindingEvent;

// Shorthand for brevity's sake
var _c = vpx.gui.event.ComponentBindingEvent; // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.name = null;                       // private String
_i.value = null;                      // private Object

/**
 * This event id indicates that an attribute has been added.
 */
_c.ATTRIBUTE_ADDED = 2401;            // public static final int

/**
 * This event id indicates that an attribute has been removed.
 */
_c.ATTRIBUTE_REMOVED = 2402;          // public static final int

/**
 * This event id indicates that an attribute has been replaced.
 */
_c.ATTRIBUTE_REPLACED = 2403;         // public static final int

/**
 * Returns the component whose attribute has changed.
 *
 * @return vpx.gui.Component
 *    The <code>Component</code> object that originated the event
 */
_i.getComponent = function() {
   return this.source;
};

/**
 * Returns the name with which the attribute is bound to or unbound from the
 * component.
 *
 * @return String
 *    The name with which the object has been bound or unbound
 */
_i.getName = function() {
   return this.name;
};

/**
 * Returns the value of the attribute that has been added, removed or replaced.
 * If the attribute was added (or bound), this is the value of the attribute.
 * If the attribute was removed (or unbound), this is the value of the removed
 * attribute. If the attribute was replaced, this is the old value of the
 * attribute.
 *
 * @return Object
 *    The value of the attribute that has been added, removed or replaced
 */
_i.getValue = function() {
   return this.value;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   var c = vpx.gui.event.ComponentBindingEvent;
   var typeStr;
   switch (this.id) {
   case c.ATTRIBUTE_ADDED:
      typeStr = "ATTRIBUTE_ADDED";
      break;
   case c.ATTRIBUTE_REMOVED:
      typeStr = "ATTRIBUTE_REMOVED";
      break;
   case c.ATTRIBUTE_REPLACED:
      typeStr = "ATTRIBUTE_REPLACED";
      break;
   default:
      typeStr = "unknown type";
      break;
   }

   return typeStr + ",name=" + this.name + ",value=" + value;
};
