/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class InputEvent
 * extends vpx.core.Event
 *
 * The root event class for all component-level input events.
 * <p/>
 * Input events are delivered to listeners before they are processed normally
 * by the source where they originated. This allows listeners and component
 * subclasses to "consume" the event so that the source will not process them
 * in their default manner.  For example, consuming mousePressed events on a
 * Button component will prevent the Button from being activated.
 *
 * @version 1.0 (Oct 13, 2005)
 */

/**
 * Constructs an InputEvent object with the specified source component,
 * modifiers, and type.
 * <p/>
 * Note that passing in an invalid <code>id</code> results in
 * unspecified behavior.
 *
 * @param source vpx.gui.Component
 *    The object where the event originated
 * @param id int
 *    The event type
 * @param modifiers int
 *    Represents the modifier keys and mouse buttons down while the event
 *    occurred
 * @throws Error
 *    If <code>source</code> is null
 */
vpx.gui.event.InputEvent = function(source, id, modifiers) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   var spr = vpx.core.Event;
   spr.call(this, source, id);

   this.modifiers = modifiers;
};

// InputEvent extends vpx.core.Event
vpx.gui.event.InputEvent.prototype = new vpx.core.Event(vpx.ABSTRACT_PASS);
vpx.gui.event.InputEvent.prototype.constructor = vpx.gui.event.InputEvent;

// Shorthand for brevity's sake
var _c = vpx.gui.event.InputEvent;    // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Class constants

/**
 * The Shift key modifier constant.
 */
_c.SHIFT_DOWN_MASK = 1 << 6;          // public static final int

/**
 * The Control key modifier constant.
 */
_c.CTRL_DOWN_MASK = 1 << 7;           // public static final int

/**
 * The Meta key modifier constant.
 */
_c.META_DOWN_MASK = 1 << 8;           // public static final int

/**
 * The Alt key modifier constant.
 */
_c.ALT_DOWN_MASK = 1 << 9;            // public static final int

/**
 * The Mouse Button1 modifier constant.
 */
_c.BUTTON1_DOWN_MASK = 1 << 10;       // public static final int

/**
 * The Mouse Button2 modifier constant.
 */
_c.BUTTON2_DOWN_MASK = 1 << 11;       // public static final int

/**
 * The Mouse Button3 modifier constant.
 */
_c.BUTTON3_DOWN_MASK = 1 << 12;       // public static final int

// Instance variables

/**
 * The state of the modifier mask at the time the input event was fired.
 */
_i.modifiers = 0;                     // private int

/**
 * Returns whether or not the Shift modifier is down on this event.
 *
 * @return boolean
 *    true if the shift modifier is down; false otherwise
 */
_i.isShiftDown = function() {
    var c = vpx.gui.event.InputEvent;
    return (this.modifiers & c.SHIFT_DOWN_MASK) != 0;
};

/**
 * Returns whether or not the Control modifier is down on this event.
 */
_i.isControlDown = function() {
    var c = vpx.gui.event.InputEvent;
    return (this.modifiers & c.CTRL_DOWN_MASK) != 0;
};

/**
 * Returns whether or not the Meta modifier is down on this event.
 */
_i.isMetaDown = function() {
    var c = vpx.gui.event.InputEvent;
    return (this.modifiers & c.META_DOWN_MASK) != 0;
};

/**
 * Returns whether or not the Alt modifier is down on this event.
 */
_i.isAltDown = function() {
    var c = vpx.gui.event.InputEvent;
    return (this.modifiers & c.ALT_DOWN_MASK) != 0;
};

/**
 * Returns the modifier mask for this event. Modifiers represent the state of
 * all modal keys, such as ALT, CTRL, META, and the mouse buttons just after
 * the event occurred.
 * <p/>
 * For example, if the user presses <b>button 1</b> followed by
 * <b>button 2</b>, and then releases them in the same order, the following
 * sequence of events is generated:
 * <pre>
 *    <code>MOUSE_PRESSED</code>:  <code>BUTTON1_DOWN_MASK</code>
 *    <code>MOUSE_PRESSED</code>:  <code>BUTTON1_DOWN_MASK | BUTTON2_DOWN_MASK</code>
 *    <code>MOUSE_RELEASED</code>: <code>BUTTON2_DOWN_MASK</code>
 *    <code>MOUSE_CLICKED</code>:  <code>BUTTON2_DOWN_MASK</code>
 *    <code>MOUSE_RELEASED</code>:
 *    <code>MOUSE_CLICKED</code>:
 * </pre>
 * <p/>
 * It is not recommended to compare the return value of this method using
 * <code>==</code> because new modifiers can be added in the future. For
 * example, the appropriate way to check that SHIFT and BUTTON1 are down, but
 * CTRL is up is demonstrated by the following code:
 * <pre>
 *    var cls = vpx.gui.event.InputEvent;
 *    var onmask = cls.SHIFT_DOWN_MASK | cls.BUTTON1_DOWN_MASK;
 *    var offmask = cls.CTRL_DOWN_MASK;
 *    if (event.getModifiers() & (onmask | offmask) == onmask) {
 *        ...
 *    }
 * </pre>
 * The above code will work even if new modifiers are added.
 *
 * @return int
 *    The modifier mask for this event
 */
_i.getModifiers = function() {
    return this.modifiers;
};

/**
 * Gets the modifier mask from a native DOM event.
 *
 * @param e [DOM Level 2 Events]MouseEvent
 *    The DOM event
 * @return int
 *    The modifier mask for the event
 */
_c.modifiersFromDOMEvent = function(e) {
   var c = vpx.gui.event.InputEvent;
   var mask = 0;

   if (e.shiftKey) {
      mask |= c.SHIFT_DOWN_MASK;
   }
   if (e.ctrlKey) {
      mask |= c.CTRL_DOWN_MASK;
   }
   if (e.metaKey) {
      mask |= c.META_DOWN_MASK;
   }
   if (e.altKey) {
      mask |= c.ALT_DOWN_MASK;
   }

   var button = vpx.xua.event.getButton(e);
   switch (button) {
   case vpx.xua.event.MOUSE_BUTTON_LEFT:
      mask |= c.BUTTON1_DOWN_MASK;
      break;
   case vpx.xua.event.MOUSE_BUTTON_RIGHT:
      mask |= c.BUTTON2_DOWN_MASK;
      break;
   case vpx.xua.event.MOUSE_BUTTON_MIDDLE:
      mask |= c.BUTTON3_DOWN_MASK;
      break;
   default:
      ; // There might not be a button associated with the event
      break;
   }

   return mask;
};
