/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class ItemEvent
 * extends vpx.core.Event
 *
 * A semantic event which indicates that an item was selected or deselected.
 * This high-level event is generated by an ItemSelectable object (such as a
 * List) when an item is selected or deselected by the user. The event is
 * passed to every <code>ItemListener</code> object that registered to receive
 * such events using the component's <code>addItemListener</code> method.
 * <p/>
 * The object that implements the <code>ItemListener</code> interface gets
 * this <code>ItemEvent</code> when the event occurs. The listener is
 * spared the details of processing individual mouse movements and mouse
 * clicks, and can instead process a "meaningful" (semantic) event like
 * "item selected" or "item deselected".
 */

/**
 * Constructs an <code>ItemEvent</code> object.
 * <p/>
 * Note that passing in an invalid <code>id</code> results in unspecified
 * behavior.
 *
 * @param source vpx.gui.ItemSelectable
 *    The <code>ItemSelectable</code> object that originated the event
 * @param id int
 *    An integer that identifies the event type
 * @param item Object
 *    The item affected by the event
 * @param stateChange int
 *    An integer that indicates whether the item was selected or deselected
 * @throws Error
 *    If <code>source</code> is null
 */
vpx.gui.event.ItemEvent = function(source, id, item, stateChange) {
   vpx.core.Event.call(this, source, id);
   this.item = item;
   this.stateChange = stateChange;
};

// ItemEvent extends vpx.core.Event
vpx.gui.event.ItemEvent.prototype = new vpx.core.Event(vpx.ABSTRACT_PASS);
vpx.gui.event.ItemEvent.prototype.constructor = vpx.gui.event.ItemEvent;

// Shorthand for brevity's sake
var _c = vpx.gui.event.ItemEvent;     // Class
var _i = _c.prototype;                // Instance
_i._c = _c;                           // this._c --> static vars (class)
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.item = null;                       // Object
_i.stateChange = null;                // int

/**
 * This event id indicates that an item's state changed.
 */
_c.ITEM_STATE_CHANGED  = 701;         // public static final int

/**
 * This state-change value indicates that an item was selected.
 */
_c.SELECTED            = 1;           // public static final int

/**
 * This state-change-value indicates that a selected item was deselected.
 */
_c.DESELECTED          = 2;           // public static final int

/**
 * Returns the originator of the event.
 *
 * @return vpx.gui.ItemSelectable
 *    The ItemSelectable object that originated the event
 */
_i.getItemSelectable = function() {
   return this.source;
};

/**
 * Returns the item affected by the event.
 *
 * @return Object
 *    The item that was affected by the event
 */
_i.getItem = function() {
   return this.item;
};

/**
 * Returns the type of state change (selected or deselected).
 *
 * @return const int
 *    An integer that indicates whether the item was selected or deselected
 * @see
 *    #SELECTED
 * @see
 *    #DESELECTED
 */
_i.getStateChange = function() {
   return this.stateChange;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   var typeStr;
   switch (this.id) {
   case vpx.gui.event.ItemEvent.ITEM_STATE_CHANGED:
      typeStr = "ITEM_STATE_CHANGED";
      break;
   default:
      typeStr = "unknown type";
      break;
   }

   var stateStr;
   switch (this.stateChange) {
   case vpx.gui.event.ItemEvent.SELECTED:
      stateStr = "SELECTED";
      break;
   case vpx.gui.event.ItemEvent.DESELECTED:
      stateStr = "DESELECTED";
      break;
   default:
      stateStr = "unknown state";
      break;
   }
   return typeStr + ",item=" + this.item + ",stateChange=" + stateStr;
};
