/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class MouseEvent
 * extends vpx.gui.event.InputEvent
 *
 * An event which indicates that a mouse action occurred in a component.
 * A mouse action is considered to occur in a particular component if and only
 * if the mouse cursor is over the unobscured part of the component's bounds
 * when the action happens.
 * Component bounds can be obscurred by the visible component's children or by
 * a menu or by a top-level window.
 * This event is used both for mouse events (click, enter, exit) and mouse
 * motion events (moves and drags), though mouse motion events are currently
 * unimplemented.
 * <p/>
 * This low-level event is generated by a component object for:
 * <ul>
 * <li>Mouse Events
 *   <ul>
 *     <li>a mouse button is pressed
 *     <li>a mouse button is released
 *     <li>a mouse button is clicked (pressed and released)
 *     <li>the mouse cursor enters the unobscured part of component's geometry
 *     <li>the mouse cursor exits the unobscured part of component's geometry
 *   </ul>
 * <li>Mouse Motion Events (Not yet implemented)
 *   <ul>
 *     <li>the mouse is moved
 *     <li>the mouse is dragged
 *   </ul>
 * </ul>
 * <p/>
 * A <code>MouseEvent</code> object is passed to every
 * <code>MouseListener</code> object which is registered to receive the
 * "interesting" mouse events using the component's
 * <code>addMouseListener</code> method. Each such listener object
 * gets a <code>MouseEvent</code> containing the mouse event.
 * <p/>
 * A <code>MouseEvent</code> object is also passed to every
 * <code>MouseMotionListener</code> object which is registered to receive
 * mouse motion events using the component's
 * <code>addMouseMotionListener</code> method. Each such listener object
 * gets a <code>MouseEvent</code> containing the mouse motion event.
 * <p/>
 * When a mouse button is clicked, events are generated and sent to the
 * registered <code>MouseListener</code>s.
 * The state of modal keys can be retrieved using {@link InputEvent#getModifiers}
 * The button mask returned by {@link InputEvent#getModifiers} reflects
 * only the button that changed state, not the current state of all buttons.
 * To get the state of all buttons and modifier keys, use
 * {@link InputEvent#getModifiersEx}.
 * The button which has changed state is returned by {@link MouseEvent#getButton}
 *
 * @version 1.0 (Oct 13, 2005)
 */

/**
 * Constructs a <code>MouseEvent</code> object with the specified source
 * component, type, modifiers, coordinates, and click count.
 * <p/>
 * Note that passing in an invalid <code>id</code> results in unspecified
 * behavior.  Creating an invalid event (such as by using modifier/button
 * values which don't match) results in unspecified behavior.
 *
 * @param source vpx.gui.Component
 *    The <code>Component</code> that originated the event
 * @param id int
 *    The integer that identifies the event
 * @param modifiers int
 *    The modifier keys down during event (e.g. shift, ctrl, alt, meta)
 * @param x int
 *    The horizontal x coordinate for the mouse location
 * @param y int
 *    The vertical y coordinate for the mouse location
 * @param clickCount int
 *    The number of mouse clicks associated with event
 * @param popupTrigger boolean
 *    true if this event is a trigger for a popup menu
 * @param button int
 *    Which of the mouse buttons has changed state. <code>NOBUTTON</code>,
 *    <code>BUTTON1</code>, <code>BUTTON2</code> or <code>BUTTON3</code>.
 * @throws Error
 *    If an invalid <code>button</code> value is passed in
 * @throws Error
 *    If <code>source</code> is null
 */
vpx.gui.event.MouseEvent = function(source, id, modifiers, x, y, clickCount,
                                    popupTrigger, button)
{
   //super(source, id, modifiers)
   var spr = vpx.gui.event.InputEvent;
   spr.call(this, source, id, modifiers);

   this.x = x;
   this.y = y;
   this.clickCount = clickCount;
   this.popupTrigger = popupTrigger;

   var c = vpx.gui.event.MouseEvent;
   if (button < c.NOBUTTON || button > c.BUTTON3) {
      throw new Error("vpx.gui.event.MouseEvent: Invalid button value");
   }
   this.button = button;
};

// MouseEvent extends vpx.gui.event.InputEvent
vpx.gui.event.MouseEvent.prototype = new vpx.gui.event.InputEvent(vpx.ABSTRACT_PASS);
vpx.gui.event.MouseEvent.prototype.constructor = vpx.gui.event.MouseEvent;

// Shorthand for brevity's sake
var _c = vpx.gui.event.MouseEvent;    // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Class constants

/**
 * The "mouse clicked" event. This <code>MouseEvent</code>
 * occurs when a mouse button is pressed and released.
 */
_c.MOUSE_CLICKED = 500;               // public static final int

/**
 * The "mouse pressed" event. This <code>MouseEvent</code>
 * occurs when a mouse button is pushed down.
 */
_c.MOUSE_PRESSED = 501;               // public static final int

/**
 * The "mouse released" event. This <code>MouseEvent</code>
 * occurs when a mouse button is let up.
 */
_c.MOUSE_RELEASED = 502;              // public static final int

/**
 * The "mouse moved" event. This <code>MouseEvent</code>
 * occurs when the mouse position changes.
 * This is not yet implemented but is reserved for future use.
 */
_c.MOUSE_MOVED = 503;                 // public static final int

/**
 * The "mouse entered" event. This <code>MouseEvent</code>
 * occurs when the mouse cursor enters the unobscured part of component's
 * geometry.
 */
_c.MOUSE_ENTERED = 504;               // public static final int

/**
 * The "mouse exited" event. This <code>MouseEvent</code>
 * occurs when the mouse cursor exits the unobscured part of component's
 * geometry.
 */
_c.MOUSE_EXITED = 505;                // public static final int

/**
 * The "mouse dragged" event. This <code>MouseEvent</code>
 * occurs when the mouse position changes while a mouse button is pressed.
 * This is not yet implemented but is reserved for future use.
 */
_c.MOUSE_DRAGGED = 506;               // public static final int

/**
 * The "mouse wheel" event.  This is the only <code>MouseWheelEvent</code>.
 * It occurs when a mouse equipped with a wheel has its wheel rotated.
 * This is not yet implemented but is reserved for future use.
 */
_c.MOUSE_WHEEL = 507;                 // public static final int

/**
 * Indicates no mouse buttons; used by {@link #getButton}.
 */
_c.NOBUTTON = 0;                      // public static final int

/**
 * Indicates mouse button #1; used by {@link #getButton}.
 */
_c.BUTTON1 = 1;                       // public static final int

/**
 * Indicates mouse button #2; used by {@link #getButton}.
 */
_c.BUTTON2 = 2;                       // public static final int

/**
 * Indicates mouse button #3; used by {@link #getButton}.
 */
_c.BUTTON3 = 3;                       // public static final int

// Instance variables

/**
 * The mouse event's x coordinate.
 * The x value is relative to the component that fired the event.
 */
_i.x = -1;                            // private int

/**
 * The mouse event's y coordinate.
 * The y value is relative to the component that fired the event.
 */
_i.y = -1;                            // private int

/**
 * Indicates the number of quick consecutive clicks of a mouse button.
 * clickCount will be valid for only three mouse events:
 * <p/>
 * <code>MOUSE_CLICKED</code>, <code>MOUSE_PRESSED</code> and
 * <code>MOUSE_RELEASED</code>.
 * For the above, the <code>clickCount</code> will be at least 1.
 * For all other events the count will be 0.
 */
_i.clickCount = 0;                    // private int

/**
 * Indicates which, if any, of the mouse buttons has changed state.
 *
 * The only legal values are the following constants:
 * <code>NOBUTTON</code>, <code>BUTTON1</code>, <code>BUTTON2</code> or
 * <code>BUTTON3</code>.
 */
_i.button = _c.NOBUTTON;              // private int

/**
 * A property used to indicate whether a Popup Menu should appear with a
 * certain gestures. If <code>popupTrigger</code> = <code>false</code>,
 * no popup menu should appear.  If it is <code>true</code>
 * then a popup menu should appear.
 */
_i.popupTrigger = false;              // private boolean

/**
 * Returns the horizontal x position of the event relative to the
 * source component.
 *
 * @return x int
 *    An integer indicating horizontal position relative to the component
 */
_i.getX = function() {
   return this.x;
};

/**
 * Returns the vertical y position of the event relative to the
 * source component.
 *
 * @return y int
 *    An integer indicating vertical position relative to the component
 */
_i.getY = function() {
   return this.y;
};

/**
 * Returns the x,y position of the event relative to the source component.
 *
 * @return vpx.gui.Point
 *    A <code>Point</code> object containing the x and y coordinates
 *    relative to the source component
 */
_i.getPoint = function() {
   return new vpx.gui.Point(this.x, this.y);
};

/**
 * Returns the number of mouse clicks associated with this event.
 *
 * @return int
 *    Integer value for the number of clicks
 */
_i.getClickCount = function() {
   return this.clickCount;
};

/**
 * Returns which, if any, of the mouse buttons has changed state.
 *
 * @return int
 *    One of the following constants: <code>NOBUTTON</code>,
 *    <code>BUTTON1</code>, <code>BUTTON2</code> or <code>BUTTON3</code>
 */
_i.getButton = function() {
   return this.button;
};

/**
 * Returns whether or not this mouse event is the popup menu trigger event for
 * the platform.
 * <p/>
 * <b>Note</b>: Popup menus are triggered differently on different systems.
 * Therefore, <code>isPopupTrigger</code> should be checked in both
 * <code>mousePressed</code> and <code>mouseReleased</code> for proper
 * cross-platform functionality.
 *
 * @return boolean
 *    true if this event is the popup menu trigger for this platform
 */
_i.isPopupTrigger = function() {
   return this.popupTrigger;
};

/**
 * Creates a new lightweight <code>MouseEvent</code> object from a native DOM
 * event.
 *
 * @param source vpx.gui.Component
 *    The <code>Component</code> that originated the event
 * @param e [DOM Level 2 Events]MouseEvent
 *    The native DOM event
 */
_c.fromDOMEvent = function(source, e) {
   var MouseEvent = vpx.gui.event.MouseEvent;
   var InputEvent = vpx.gui.event.InputEvent;
   var id = MouseEvent.idFromDOMEvent(e);
   var mod = InputEvent.modifiersFromDOMEvent(e);
   var x = e.clientX;
   var y = e.clientY;
   //?? TODO clickCount and popupTrigger are hard-coded
   var clickCount = 1;
   var popupTrigger = false;
   var button = MouseEvent.buttonFromDOMEvent(e);
   return new MouseEvent(source, id, mod, x, y, clickCount, popupTrigger, button);
};

/**
 * Gets the event id from a DOM Level 2 event type.
 *
 * @param e [DOM Level 2 Events]MouseEvent
 *    The native DOM event
 * @return int
 *    A valid mouse event id
 * @exception Error
 *    If type is not recognized
 */
_c.idFromDOMEvent = function(e) {
   var type = e.type;
   var c = vpx.gui.event.MouseEvent;
   if (type == "click") {
      return c.MOUSE_CLICKED;
   } else if (type == "mousedown") {
      return c.MOUSE_PRESSED;
   } else if (type == "mouseup") {
      return c.MOUSE_RELEASED;
   } else if (type == "mouseover") {
      return c.MOUSE_ENTERED;
   } else if (type == "mouseout") {
      return c.MOUSE_EXITED;
   } else {
      throw new Error("MouseEvent#idFromDOMEvent(): invalid type: " + type);
   }
};

/**
 * Gets the event button from a DOM Level 2 event button.
 *
 * @param e [DOM Level 2 Events]MouseEvent
 *    The native DOM event
 * @return int
 *    A valid mouse button
 * @exception Error
 *    If <code>button</code> is not recognized
 */
_c.buttonFromDOMEvent = function(e) {
   var button = vpx.xua.event.getButton(e);
   var c = vpx.gui.event.MouseEvent;
   switch (button) {
   case vpx.xua.event.MOUSE_BUTTON_LEFT:
      return c.BUTTON1;
   case vpx.xua.event.MOUSE_BUTTON_RIGHT:
      return c.BUTTON2;
   case vpx.xua.event.MOUSE_BUTTON_MIDDLE:
      return c.BUTTON3;
   case vpx.xua.event.MOUSE_BUTTON_NONE:
      return c.NOBUTTON;
   default:
      throw new Error("MouseEvent#()buttonFromDOMEvent: invalid button: " + button);
   }
};

/**
 * Returns a <code>String</code> describing the modifier keys and mouse buttons
 * that were down during the event, such as "Shift", or "Ctrl+Shift".
 *
 * @param modifiers int
 *    A modifier mask describing the modifier keys and mouse buttons that were
 *    down during the event
 * @return String
 *    A text description of the combination of modifier keys and mouse buttons
 *    that were down during the event
 */
_i.getMouseModifiersText = function(modifiers) {
   var s = "";
   var c = vpx.gui.event.InputEvent;
   if ((modifiers & c.ALT_DOWN_MASK) != 0) {
      s += "Alt+";
   }
   if ((modifiers & c.META_DOWN_MASK) != 0) {
      s += "Meta+";
   }
   if ((modifiers & c.CTRL_DOWN_MASK) != 0) {
      s += "Ctrl+";
   }
   if ((modifiers & c.SHIFT_DOWN_MASK) != 0) {
      s += "Shift+";
   }
   if ((modifiers & c.BUTTON1_DOWN_MASK) != 0) {
      s += "Button1+";
   }
   if ((modifiers & c.BUTTON2_DOWN_MASK) != 0) {
      s += "Button2+";
   }
   if ((modifiers & c.BUTTON3_DOWN_MASK) != 0) {
      s += "Button3+";
   }
   if (s.length > 0) {
      s = s.slice(0, s.length - 1);
   }
   return s;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   var str;
   var c = vpx.gui.event.MouseEvent;
   switch (this.id) {
   case c.MOUSE_PRESSED:
      str = "MOUSE_PRESSED";
      break;
   case c.MOUSE_RELEASED:
      str = "MOUSE_RELEASED";
      break;
   case c.MOUSE_CLICKED:
      str = "MOUSE_CLICKED";
      break;
   case c.MOUSE_ENTERED:
      str = "MOUSE_ENTERED";
      break;
   case c.MOUSE_EXITED:
      str = "MOUSE_EXITED";
      break;
   case c.MOUSE_MOVED:
      str = "MOUSE_MOVED";
      break;
   case c.MOUSE_DRAGGED:
      str = "MOUSE_DRAGGED";
      break;
   case c.MOUSE_WHEEL:
      str = "MOUSE_WHEEL";
      break;
   default:
      str = "unknown type";
      break;
   }

   str += ",(" + this.x + "," + this.y + ")";
   str += ",button=" + this.getButton();
   if (this.getModifiers() != 0) {
      str += ",modifiers=" + this.getMouseModifiersText(this.modifiers);
   }
   str += ",clickCount=" + this.clickCount;

   return str;
};
