/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class PropertyChangeEvent
 * extends vpx.core.Event
 *
 * A "PropertyChange" event gets delivered whenever a component changes a
 * "bound" or "constrained" property.  A PropertyChangeEvent object is sent as
 * an argument to the PropertyChangeListener and VetoableChangeListener
 * methods.
 * <p/>
 * Normally, PropertyChangeEvents are accompanied by the name and the old
 * and new value of the changed property.  If the new value is a primitive
 * type (such as int or boolean) it must be wrapped as the corresponding Object
 * type (such as Integer or Boolean).
 * <p/>
 * Null values may be provided for the old and the new values if their
 * true values are not known.
 * <p/>
 * An event source may send a null object as the name to indicate that an
 * arbitrary set of if its properties have changed.  In this case the
 * old and new values should also be null.
 */

/**
 * Constructs a <code>PropertyChangeEvent</code> object.
 *
 * @param source Object
 *    The object that is the source of the event
 * @param propertyName String
 *    The programatic name of the property that was changed
 * @param oldValue Object
 *    The old value of the property
 * @param newValue Object
 *    The new value of the property
 * @throws Error
 *    If <code>source</code> is null
 */
vpx.gui.event.PropertyChangeEvent = function(source, propertyName, oldValue, newValue) {
   var id = vpx.gui.event.PropertyChangeEvent.PROPERTY_CHANGED;
   vpx.core.Event.call(this, source, id);
   this.propertyName = propertyName;
   this.newValue = newValue;
   this.oldValue = oldValue;
};

// PropertyChangeEvent extends vpx.core.Event
vpx.gui.event.PropertyChangeEvent.prototype = new vpx.core.Event(vpx.ABSTRACT_PASS);
vpx.gui.event.PropertyChangeEvent.prototype.constructor = vpx.gui.event.PropertyChangeEvent;

// Shorthand for brevity's sake
var _c = vpx.gui.event.PropertyChangeEvent;  // Class
var _i = _c.prototype;                       // Instance
_i._c = _c;                                  // this._c --> static vars (class)
_i._concrete = true;                         // vpx system flag for concrete classes

// Instance variables
_i.propertyName = null;                      // private String
_i.newValue = null;                          // private Object
_i.oldValue = null;                          // private Object

/**
 * This event id indicates a property change.
 */
_c.PROPERTY_CHANGED = 2101;                  // public static final int

/**
 * Gets the programatic name of the property that was changed.
 *
 * @return String
 *    The programatic name of the property that was changed.
 *    May be null if multiple properties have changed
 */
_i.getPropertyName = function() {
   return this.propertyName;
};

/**
 * Sets the new value for the property, expressed as an Object.
 *
 * @return Object
 *    The new value for the property, expressed as an Object.
 *    May be null if multiple properties have changed
 */
_i.getNewValue = function() {
   return this.newValue;
};

/**
 * Gets the old value for the property, expressed as an Object.
 *
 * @return Object
 *    The old value for the property, expressed as an Object.
 *    May be null if multiple properties have changed
 */
_i.getOldValue = function() {
   return this.oldValue;
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   var typeStr;
   switch (this.id) {
   case vpx.gui.event.PropertyChangeEvent.PROPERTY_CHANGED:
      typeStr = "PROPERTY_CHANGED";
      break;
   default:
      typeStr = "unknown type";
      break;
   }

   return typeStr;
};
