/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class PropertyChangeSupport
 * extends Object
 *
 * This is a utility class that can be used by components that support bound
 * properties.  You can use an instance of this class as a member field
 * of your component and delegate various work to it.
 */

vpx.gui.event.PropertyChangeSupport = function(source) {
   if (!source) {
      throw new Error("vpx.gui.event.PropertyChangeSupport: null source");
   }
   this.source = source;
};

// Shorthand for brevity's sake
var _c = vpx.gui.event.PropertyChangeSupport;  // Class
var _i = _c.prototype;                         // Instance
_i._concrete = true;                           // vpx system flag for concrete classes

// Instance variables
_i.source = null;                       // private Object
_i.listeners = null;                    // private vpx.gui.event.PropertyChangeListener[]
_i.children = null;                     // private Hashtable

/**
 * Add a PropertyChangeListener to the listener list.  The listener is
 * registered for all properties.  The same listener object may be added more
 * than once, and will be called as many times as it is added.
 * If <code>listener</code> is null, no exception is thrown, and no action
 * is taken.
 *
 * @param listener vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be added
 */
_i.addGlobalPropertyChangeListener = function(l) {
   if (!l) {
      return;
   }
   if (this.listeners == null) {
      this.listeners = [];
   }
   this.listeners.push(l);
};

/**
 * Remove a PropertyChangeListener from the listener list.  This removes a
 * PropertyChangeListener that was registered for all properties.
 * If <code>listener</code> was added more than once to the same event
 * source, it will be notified one less time after being removed.
 * If <code>listener</code> is null, or was never added, no exception is
 * thrown, and no action is taken.
 *
 * @param l vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be removed
 */
_i.removeGlobalPropertyChangeListener = function(l) {
   if (!l || this.listeners == null) {
      return;
   }
   var lList = this.listeners;
   for (var i = 0; i < lList.length; i++) {
      if (arePolyEqual(lList[i], l)) {
         lList.splice(i, 1);
         return;
      }
   }
};

/**
 * Returns an array of all the listeners that were added to the
 * PropertyChangeSupport object with addPropertyChangeListener().
 * <p/>
 * If some listeners have been added with a named property, then the returned
 * array will contain a mixture of Listeners (some that have been added with a
 * named property, and some that have not).
 *
 * @return vpx.gui.event.PropertyChangeListener[]
 *    All of the <code>PropertyChangeListeners</code> added, or an empty array
 *    if no listeners have been added
 */
_i.getGlobalPropertyChangeListeners = function() {
   var i;
   var result = [];

   if (this.listeners != null) {
      for (i = 0; i < this.listeners.length; i++) {
         result.push(this.listeners[i]);
      }
   }

   if (this.children != null) {
      for (var key in this.children) {
         var child = this.children[key];
         if (child instanceof vpx.gui.event.PropertyChangeSupport) {
            var childListeners = child.getGlobalPropertyChangeListeners();
            for (i = childListeners.length - 1; i >= 0; i--) {
               result.push(childListeners[i]);
            }
         }
      }
   }

   return result;
};

/**
 * Add a PropertyChangeListener for a specific property.  The listener
 * will be invoked only when a call on firePropertyChange names that
 * specific property.
 * The same listener object may be added more than once.  For each
 * property,  the listener will be invoked the number of times it was added
 * for that property.
 * If <code>propertyName</code> or <code>listener</code> is null, no
 * exception is thrown and no action is taken.
 *
 * @param property String
 *    The name of the property to listen on.
 * @param listener vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be added
 */
_i.addSpecificPropertyChangeListener = function(property, l) {
   if (!property || !l) {
      return;
   }
   if (this.children == null) {
      this.children = {};
   }
   var child = this.children[property];
   if (!child) {
      child = new vpx.gui.event.PropertyChangeSupport(this.source);
      this.children[property] = child;
   }
   child.addGlobalPropertyChangeListener(l);
};

/**
 * Remove a PropertyChangeListener for a specific property.  If
 * <code>listener</code> was added more than once to the same event source for
 * the specified property, it will be notified one less time after being
 * removed.  If <code>propertyName</code> is null,  no exception is thrown, and
 * no action is taken.
 * <p/>
 * If <code>listener</code> is null, or was never added for the specified
 * property, no exception is thrown and no action is taken.
 *
 * @param propertyName String
 *    The name of the property that was listened on
 * @param listener vpx.gui.event.PropertyChangeListener
 *    The PropertyChangeListener to be removed
 */
_i.removeSpecificPropertyChangeListener = function(property, l) {
   if (!property || !l) {
      return;
   }
   if (this.children == null) {
      return;
   }
   var child = this.children[property];
   if (!child) {
      return;
   }
   child.removeGlobalPropertyChangeListener(l);
};

/**
 * Returns an array of all the listeners that have been associated with the
 * named property.
 *
 * @param property String
 *    The name of the property being listened to
 * @return vpx.gui.event.PropertyChangeListener[]
 *    All of the <code>PropertyChangeListeners</code> associated with the named
 *    property.  If no such listeners have been added, or if
 *    <code>propertyName</code> is null, an empty array is returned
 */
_i.getSpecificPropertyChangeListeners = function(property) {
   if (this.children != null && property) {
      var child = this.children[property];
      if (child) {
         return child.getGlobalPropertyChangeListeners();
      }
   }

   return [];
};

/**
 * Report a bound property update to any registered listeners.  No event is
 * fired if old and new are equal and non-null.
 *
 * @param property String
 *    The programatic name of the property that was changed
 * @param oldValue Object
 *    The old value of the property.
 * @param newValue Object
 *    The new value of the property.
 */
_i.firePropertyChange = function(property, oldValue, newValue) {
   if (oldValue && newValue && arePolyEqual(oldValue, newValue)) {
      return;
   }
   var event = new vpx.gui.event.PropertyChangeEvent(this.source, property, oldValue, newValue);
   this._firePropertyChange(event);
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Fire an existing PropertyChangeEvent to any registered listeners.  No event
 * is fired if the given event's old and new values are equal and non-null.
 *
 * @param e vpx.gui.event.PropertyChangeEvent
 *    The PropertyChangeEvent object
 */
_i._firePropertyChange = function(e) {
   var oldValue = e.getOldValue();
   var newValue = e.getNewValue();
   var propertyName = e.getPropertyName();
   if (oldValue && newValue && arePolyEqual(oldValue, newValue)) {
      return;
   }

   if (this.listeners != null) {
      for (var i = 0; i < this.listeners.length; i++) {
         var target = this.listeners[i];
         target.propertyChange(e);
      }
   }

   if (this.children != null && propertyName) {
      var child = this.children[propertyName];
      if (child) {
         child._firePropertyChange(e);
      }
   }
};
