/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public abstract class ComponentUI
 * extends Object
 *
 * The interface for all UI delegate objects in the VPX pluggable look and feel
 * architecture.  The UI delegate object for a VPX component is responsible for
 * implementing the aspects of the component that depend on the look and feel.
 * The <code>vpx.gui.Component</code> class invokes methods from this class in
 * order to delegate operations (painting, layout calculations, etc.) that may
 * vary depending on the look and feel installed.  <b>Client programs should
 * not invoke methods on this class directly.</b>
 * <p/>
 * VPX components are all backed by native DOM </code>HTMLElement</code>s.  The
 * UI will transparently delegate all work to the DOM API.  It will, however,
 * provide access to the DOM object that backs this component, as well as the
 * DOM object that contains this component.
 *
 * @version 1.0 (Oct 6, 2005)
 */

/**
 *
 */
vpx.gui.plaf.ComponentUI = function() {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }
   if (!this._concrete) {
      throw new Error("vpx.gui.plaf.ComponentUI: cannot instantiate abstract class");
   }

   this._initSections();
};

// Shorthand for brevity's sake
var _c = vpx.gui.plaf.ComponentUI;     // Class
var _i = _c.prototype;                 // Instance
_i._concrete = false;                  // vpx system flag for concrete classes

// Class variables
_i.PEER_SECTION_CONTENT  = 0;
_i.PEER_SECTION_CHILDREN = 1;

// Instance variables
_i.comp = null;                        // protected vpx.gui.Component
_i.peer = null;                        // protected HTMLElement
_i.sections = null;                    // private List<HTMLElement>
_i.docPlacementHandler= null;          // private Function

/**
 * Configures the specified component appropriate for the look and feel.  This
 * method is invoked when the <code>ComponentUI</code> instance is being
 * installed as the UI delegate on the specified component.  This method should
 * completely configure the component for the look and feel, including the
 * following:
 * <ol>
 *   <li>Install any default property values for color, fonts, borders,
 *       icons, opacity, etc. on the component.  Whenever possible,
 *       property values initialized by the client program should <i>not</i>
 *       be overridden.
 *   <li>Install a <code>LayoutManager</code> on the component if necessary.
 *   <li>Create/add any required sub-components to the component.
 *   <li>Create/install event listeners on the component.
 *   <li>Create/install a <code>PropertyChangeListener</code> on the component
 *       in order to detect and respond to component property changes
 *       appropriately.
 *   <li>Initialize any appropriate instance data.
 * </ol>
 *
 * @param c vpx.gui.Component
 *    The component where this UI delegate is being installed
 * @see #uninstallUI
 */
_i.installUI = function(c) {
   this.comp = c;
   var view = c.getView();
   var handler = this.getDocPlacementHandler();
   vpx.xua.event.listen(view, "load", handler);
   vpx.xua.event.listen(view, "resize", handler);
};

/**
 * Reverses configuration which was done on the specified component during
 * <code>installUI</code>.  This method is invoked when this
 * <code>ComponentUI</code> instance is being removed as the UI delegate
 * for the specified component.  This method should undo the configuration
 * performed in <code>installUI</code>, being careful to leave the
 * <code>Component</code> instance in a clean state (no extraneous listeners,
 * look-and-feel-specific property objects, etc.). This should include the
 * following:
 * <ol>
 *   <li>Remove any UI-set borders from the component.
 *   <li>Remove any UI-set layout managers on the component.
 *   <li>Remove any UI-added sub-components from the component.
 *   <li>Remove any UI-added event/property listeners from the component.
 *   <li>Remove any UI-installed keyboard UI from the component.
 *   <li>Nullify any allocated instance data objects to allow for GC.
 * </ol>
 *
 * @param c vpx.gui.Component
 *    The component from which this UI delegate is being removed; this argument
 *    is often ignored, but might be used if the UI object is stateless
 *    and shared by multiple components
 * @see #installUI
 */
_i.uninstallUI = function(c) {
   var view = c.getView();
   var handler = this.getDocPlacementHandler();
   vpx.xua.event.ignore(view, "load", handler);
   vpx.xua.event.ignore(view, "resize", handler);
   delete this.comp;
};

/**
 * Creates a new native peer DOM element that will be responsible for the
 * actual rendering of this UI and associates it with this UI.  Subclasses may
 * override this method to either provide an alternate peer object or
 * initialize ancillary peer objects that may be required for the specific UI
 * implementation.
 * <p/>
 * For instance, the default peer object for all components is a "span"
 * <code>HTMLElement</code> with an inlnie block display, but you may prefer to
 * use an <code>HTMLHRElement</code>.  Or you may require an
 * <code>HTMLInputElement</code> to accompany your base peer.  Such logic would
 * be added to this method when overridden.
 *
 * @param c vpx.gui.Component
 *    The component to which this peer will be attached
 * @return HTMLElement
 *    A newly created native peer
 */
_i.initPeer = function(c) {
   var p = c.getView().document.createElement("span");
   p.style.display = vpx.xua.css.getValue("display", "inline-block");
   this.peer = p;
   return p;
};

/**
 * Gets the native DOM element that is responsible for the actual rendering of
 * this UI.
 *
 * @return HTMLElement
 *    The opaque DOM element that renders this UI
 */
_i.getPeer = function() {
   return this.peer;
};

/**
 * Gets the native peer object associated with this ui that is capable of
 * receiving <code>MouseEvent</code>s.  This default implementation simply
 * returns the UI's peer, but subclasses may override this method to provide
 * alternate peer elements that are responsible for receiving mouse events.
 *
 * @return HTMLElement
 *    The opaque DOM element that can receive mouse events for this ui
 */
_i.getClickablePeer = function() {
   return this.peer;
};

/**
 * Gets the peer responsible for serving up the tooltip text to the user, if
 * applicable.  This default implementation returns null, signaling that
 * tooltips are not used in this UI.  Subclasses should override this method to
 * provide the ability to add tooltips to the component.
 *
 * @return HTMLElement
 *    The peer that serves up tooltip text for this component
 */
_i.getTooltipPeer = function() {
   return null;
};

/**
 * Gets the specified native DOM element that is responsible for the actual
 * rendering of the specified section of the UI.  UIs will often have optional
 * elements within them (such as text, icon, children), and there's an inherant
 * ambiguity of where in the UI's peer structure a given element lives.  This
 * construct removes that ambiguity by enumerating which "sections" a given UI
 * supports, and ensuring that each section lives in its correct place with
 * respect to other sections.
 * <p/>
 * Subclasses should override this method if they support any sections other
 * than the two default sections: <code>PEER_SECTION_CONTENT</code> and
 * <code>PEER_SECTION_CHILDREN</code>.  Subclasses are then responsible for
 * extending this enumeration to add the sections that they also support
 * (@see _initSections()).
 *
 * @param section const int
 *    The section that you want to retrieve
 * @return HTMLElement
 *    The opaque DOM element that renders this UI's specified section
 * @throws Error
 *    Thrown if the UI's peer has not yet been initialized
 * @see #PEER_SECTION_CONTENT
 * @see #PEER_SECTION_CHILDREN
 */
_i.getPeerSection = function(sctn) {
   if (this.peer == null) {
      throw new Error("Can't create peer section without peer");
   }

   var s = this.sections[sctn];
   if (s != null) {
      // We've already created this section
      return s;
   }

   // We need to create the section before returning it
   var view = vpx.xua.getView(this.peer);
   s = view.document.createElement("span");
   this.sections[sctn] = s;

   // Place the section in order within the peer
   for (var i = sctn + 1; this.sections[i] == null && i < this.sections.length; i++) {
      ;
   }
   if (i == this.sections.length) {
      // We've gone out of bounds; append the section to the end of the peer
      this.peer.appendChild(s);
   } else {
      this.peer.insertBefore(s, this.sections[i]);
   }

   return s;
};

/**
 * Appends the given <code>Component</code> to this ui's peer. Behind the
 * scenes, this really gets the <code>Component</code>'s peer, since
 * lightweight components cannot be added directly to peer objects. This method
 * simply appends the given element as a child of this ui's peer.  UIs with
 * complex peer structures should override this method to correctly place the
 * given element into this ui's peer hierarchy.
 *
 * @param c vpx.gui.Component
 *    The component whose peer should be appended to this ui's peer element
 */
_i.appendToPeer = function(c) {
   var s = this.getPeerSection(this.PEER_SECTION_CHILDREN);
   s.appendChild(c.getPeer());
};

/**
 * Removes the given <code>Component</code> from this ui's peer. Behind the
 * scenes, this really gets the <code>Component</code>'s peer, since
 * lightweight components cannot be added or removed directly from peer
 * objects. This method simply removes the given element from this this ui's
 * peer.  UIs with complex peer structures should override this method to
 * correctly locate and remove the given element from this ui's peer hierarchy.
 *
 * @param c vpx.gui.Component
 *    The component whose peer should be removed from this ui's peer element
 */
_i.removeFromPeer = function(c) {
   var s = this.getPeerSection(this.PEER_SECTION_CHILDREN);
   s.removeChild(c.getPeer());
};

/**
 * Initializes the <code>sections</code> member variable to an array that has
 * slots for each of the supported sections.
 * <p/>
 * Subclasses should override this method if they support any sections other
 * than the two default sections: <code>PEER_SECTION_CONTENT</code> and
 * <code>PEER_SECTION_CHILDREN</code>.
 *
 * @see getPeerSection(int)
 */
_i._initSections = function() {
   this.sections = [];
   this.sections[this.PEER_SECTION_CONTENT] = null;
   this.sections[this.PEER_SECTION_CHILDREN] = null;
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Gets the document placement handler for this ui.
 *
 * @return Function
 *    The doc placement handler callback, created as needed and stored for
 *    future use
 */
_i.getDocPlacementHandler = function() {
   if (this.docPlacementHandler == null) {
      this.docPlacementHandler = this._firePlacementChanged.bind(this);
   }
   return this.docPlacementHandler;
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type for this ui's component.
 *
 * @param e [DOM Level 2 Events]Event
 *    The native DOM <code>Event</code>. This event is ignored and will be
 *    replaced by a <code>vpx.gui.event.ComponentEvent</code> before being
 *    sent to listeners
 */
_i._firePlacementChanged = function(e) {
   this.comp._firePlacementChanged();
};
