/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class SplitButtonUI
 * extends ButtonUI
 *
 * UI implementation for split buttons.
 *
 * @version 1.0 (Apr 17, 2006)
 */

/**
 *
 */
vpx.gui.plaf.SplitButtonUI = function() {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   // super()
   vpx.gui.plaf.ButtonUI.call(this);
};

// SplitButtonUI extends vpx.gui.plaf.ButtonUI
vpx.gui.plaf.SplitButtonUI.prototype = new vpx.gui.plaf.ButtonUI(vpx.ABSTRACT_PASS);
vpx.gui.plaf.SplitButtonUI.prototype.constructor = vpx.gui.plaf.SplitButtonUI;

// Shorthand for brevity's sake
var _c = vpx.gui.plaf.SplitButtonUI;   // Class
var _i = _c.prototype;                 // Instance
_i._concrete = true;                   // vpx system flag for concrete classes

// Class variables
_i.PEER_SECTION_ICON     = 0;
_i.PEER_SECTION_CONTENT  = 1;
_i.PEER_SECTION_PULLDOWN = 2;
_i.PEER_SECTION_CHILDREN = 3;

// Instance variables
_i.button = null;                      // protected vpx.gui.SplitButton
_i.mainPeer = null;                    // protected HTMLElement
_i.menuPeer = null;                    // protected HTMLElement
_i.handler  = null;                    // protected vpx.gui.plaf.SplitButtonUI.Handler

/**
 * Creates a new native peer DOM element that will be responsible for the
 * actual rendering of this UI.  This implementation creates an
 * <code>HTMLTableElement</code>, with a corresponding body and row.
 *
 * @param c vpx.gui.Component
 *    The component that this peer will be attached to
 * @return HTMLElement
 *    A newly created native peer
 * @see vpx.gui.plaf.ComponentUI#initPeer(vpx.gui.Component)
 */
_i.initPeer = function(c) {
   var doc = c.getView().document;
   var p = doc.createElement("span");
   var mainP = doc.createElement("span");
   var menuP = c.menuButton.getPeer();
   p.style.display = vpx.xua.css.getValue("display", "inline-block");
   mainP.style.display = vpx.xua.css.getValue("display", "inline-block");

   p.appendChild(mainP);
   p.appendChild(menuP);
   c.menuButton._firePlacementChanged();

   this.peer = p;
   this.mainPeer = mainP;
   this.menuPeer = menuP;

   delete doc;
   delete p;
   delete mainP;
   delete menuP;

   return this.peer;
};

/**
 * Gets the native peer object associated with this ui that is capable of
 * receiving <code>MouseEvent</code>s.  This implementation returns the
 * <code>HTMLDivElement</code> that lives in the main peer's table.
 *
 * @return HTMLElement
 *    The opaque DOM element that can receive mouse events for this ui
 * @see vpx.gui.plaf.ComponentUI#getClickablePeer()
 */
_i.getClickablePeer = function() {
   return this.mainPeer;
};

/**
 *
 * @param section const int
 *    The section that you want to retrieve
 * @return HTMLElement
 *    The opaque DOM element that renders this UI's specified section
 * @see vpx.gui.plaf.ComponentUI#getPeerSection(int)
 */
_i.getPeerSection = function(sctn) {
   if (this.peer == null) {
      throw new Error("Can't create peer section without peer");
   }

   var s = this.sections[sctn];
   if (s != null) {
      // We've already created this section
      return s;
   }

   // We need to create the section before returning it
   var doc = vpx.xua.getView(this.peer).document;
   switch (sctn) {

   case this.PEER_SECTION_ICON:
      var ref = this.sections[this.PEER_SECTION_CONTENT];
      s = doc.createElement("span");
      s.style.display = vpx.xua.css.getValue("display", "inline-block");
      if (ref == null) {
         this.mainPeer.appendChild(s);
      } else {
         this.mainPeer.insertBefore(s, ref);
      }
      break;

   case this.PEER_SECTION_CONTENT:
      s = doc.createElement("span");
      s.style.display = vpx.xua.css.getValue("display", "inline-block");
      this.mainPeer.appendChild(s);
      break;

   case this.PEER_SECTION_PULLDOWN:
      s = this.menuPeer;
      break;

   case this.PEER_SECTION_CHILDREN:
      throw new Error("SplitButtinUI: PEER_SECTION_CHILDREN not supported");

   default:
      throw new Error("SplitButtinUI: Unsupported section: " + sctn);

   }

   this.sections[sctn] = s;
   return s;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.plaf.ComponentUI#installUI(vpx.gui.Component)
 */
_i.installUI = function(c) {
   // super.installUI(c)
   var spr = vpx.gui.plaf.ComponentUI.prototype;
   spr.installUI.call(this, c);

   this.initPeer(c);
   this.button = c;

   this.installDefaults();
   this.installListeners(c);
};

/**
 * Installs the default look of this UI.
 */
_i.installDefaults = function() {
   var pStyle = this.peer.style;
   var mPStyle = this.mainPeer.style;

   pStyle.border = "1px solid #cccac2";
   mPStyle.padding = "2px";
   mPStyle.cursor = "pointer";
   this.menuPeer.style.borderLeft = "1px solid #cccac2";

   delete pStyle;
   delete mPStyle;
};

/**
 *
 *
 * @param b vpx.gui.AbstractButton
 *    The button on which this ui is being installed
 */
_i.installListeners = function(c) {
   // super.installListeners(c)
   var spr = vpx.gui.plaf.ButtonUI.prototype;
   spr.installListeners.call(this, c);

   c.addChangeListener(this.getHandler());
   c.addComponentListener(this.getHandler());
   this.button.menuButton.addChangeListener(this.getMBHandler());
};

_i.getHandler = function() {
   if (this.handler == null) {
      this.handler = new vpx.gui.plaf.SplitButtonUI.Handler(this);
   }
   return this.handler;
};

_i.getMBHandler = function() {
   if (this.mbHandler == null) {
      this.mbHandler = new vpx.gui.plaf.SplitButtonUI.MBHandler(this);
   }
   return this.mbHandler;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.plaf.ComponentUI#_initSections()
 */
_i._initSections = function() {
   this.sections = [];
   this.sections[this.PEER_SECTION_ICON] = null;
   this.sections[this.PEER_SECTION_CONTENT] = null;
   this.sections[this.PEER_SECTION_PULLDOWN] = null;
   this.sections[this.PEER_SECTION_CHILDREN] = null;
};


/**
 * class Handler
 * extends Object
 * implements vpx.core.event.ChangeListener,
 *            vpx.gui.event.ComponentListener
 *
 * Listener that automatically gets registered with this ui's button.
 *
 * @version 1.0 (Apr 17, 2006)
 */

/**
 * Constructs a new Handler.
 */
_c.Handler = function(ui) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   this.ui = ui;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.core.event.ChangeListener#stateChanged(vpx.gui.event.ChangeEvent)
 */
_c.Handler.prototype.stateChanged = function(e) {
   var btn = this.ui.button;
   var model = btn.getModel();

   if (model.isRollover()) {
      this.ui.peer.style.border = "1px solid #999";
      this.ui.peer.style.backgroundColor = "#e6e3da";
      this.ui.menuPeer.style.borderLeft = "1px solid #999";
   } else {
      this.ui.peer.style.border = "1px solid #cccac2";
      this.ui.peer.style.backgroundColor = "";
      this.ui.menuPeer.style.borderLeft = "1px solid #cccac2";
   }
};

/**
 * (non-Javadoc)
 *
 * @see vpx.gui.event.ComponentListener#placementChanged(vpx.gui.event.ComponentEvent)
 */
_c.Handler.prototype.placementChanged = function(e) {
   var ui = this.ui;
   var mb = ui.button.menuButton;

   mb.setMenuLocation((1 + ui.mainPeer.offsetWidth) * -1, ui.mainPeer.offsetHeight);
   mb.getPopupMenu().setPreferredSize(new vpx.gui.Dimension(ui.peer.offsetWidth, -1));
};

/**
 * class MBHandler
 * extends Object
 * implements vpx.core.event.ChangeListener
 *
 * Listener that automatically gets registered with this ui's menu button.
 *
 * @version 1.0 (Apr 17, 2006)
 */

/**
 * Constructs a new Handler.
 */
_c.MBHandler = function(ui) {
   if (arguments[0] == vpx.ABSTRACT_PASS) {
      // Skip object initialization
      return;
   }

   this.ui = ui;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.core.event.ChangeListener#stateChanged(vpx.gui.event.ChangeEvent)
 */
_c.MBHandler.prototype.stateChanged = function(e) {
   var mb = this.ui.button.menuButton;
   var model = mb.getModel();

   if (model.isSelected()) {
      this.ui.menuPeer.style.borderLeft = "1px solid #cccac2";
      this.ui.menuPeer.style.backgroundColor = "#fff";
   } else if (model.isRollover()) {
      this.ui.peer.style.border = "1px solid #999";
      this.ui.peer.style.backgroundColor = "#e6e3da";
      this.ui.menuPeer.style.borderLeft = "1px solid #e6e3da";
      this.ui.menuPeer.style.backgroundColor = "";
   } else {
      this.ui.peer.style.border = "1px solid #cccac2";
      this.ui.peer.style.backgroundColor = "";
      this.ui.menuPeer.style.borderLeft = "1px solid #cccac2";
      this.ui.menuPeer.style.backgroundColor = "";
   }
};
