/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * Constructs a new Logger.
 */
vpx.logging.Logger = function Logger() {
   var c = vpx.logging.Logger;
   this.level = c.DEFAULT_LEVEL;
   this.url = c.DEFAULT_URL;
};

// Shorthand for brevity's sake
var _c = vpx.logging.Logger;          // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

/**
 * Levels decrease in order of severity. Setting your log level will log all
 * messages of that type and higher severity.
 */
_c.LEVEL = {
   "TRACE"    : 0,
   "DEBUG"    : 1,
   "INFO"     : 2,
   "WARN"     : 3,
   "ERROR"    : 4,
   "FATAL"    : 5,
   "NONE"     : 6
};

_c.DEFAULT_LEVEL = _c.LEVEL.WARN;
_c.DEFAULT_URL  = vpx.getTle().getContextPath() + "/jsLog.do";

// Convenience methods to log different levels
_i.trace = function(msg) { this._log("TRACE", msg); };
_i.debug = function(msg) { this._log("DEBUG", msg); };
_i.info = function(msg) { this._log("INFO", msg); };
_i.warn = function(msg) { this._log("WARN", msg); };
_i.error = function(msg) { this._log("ERROR", msg); };
_i.fatal = function(msg) { this._log("FATAL", msg); };

// Convenience methods to test the log level
_i.isTraceEnabled = function() { return this._testLevel("TRACE"); };
_i.isDebugEnabled = function() { return this._testLevel("DEBUG"); };
_i.isInfoEnabled = function() { return this._testLevel("INFO"); };
_i.isWarnEnabled = function() { return this._testLevel("WARN"); };
_i.isErrorEnabled = function() { return this._testLevel("ERROR"); };
_i.isFatalEnabled = function() { return this._testLevel("FATAL"); };


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/*
 * Tests whether the given log level is enabled.
 *
 * @param type const String
 *    The type of log level (e.g. _c.LEVEL.DEBUG)
 * @return boolean
 *    true if the level is enabled; false otherwise
 */
_i._testLevel = function(type) {
   var c = vpx.logging.Logger;
   var level = c.LEVEL[type];
   return (isDefined(level) && this.level <= level);
};

/*
 * Logs the given message of the given type, if and only if the corresponding
 * log level has been enabled.
 *
 * @param type const String
 *    The type of log level (e.g. _c.LEVEL.DEBUG)
 * @param msg String
 *    The message to log
 */
_i._log = function(type, msg) {
   var c = vpx.logging.Logger;
   var level = c.LEVEL[type];
   if (isDefined(level) && this.level <= level) {
      var fullMsg = "<client " + (new Date()).toLogString() + ">: " + msg;
      var url = this.url + "?type=" + type + "&msg=" + encodeURIComponent(fullMsg);
      var pool = vpx.net.HttpRequestPool.getInstance();
      var req = pool.getRequest();
      if (req == null) {
         // Skip the log; it's a low priority task
         delete fullMsg;
         delete url;
         delete pool;
         delete req;
         return;
      }
      try {
         req.setUrl(url);
         req.send();
      } finally {
         pool.releaseRequest(req);
      }
      delete fullMsg;
      delete url;
      delete pool;
      delete req;
   }
   delete level;
};

// Static logger to be used by all
vpx.log = new vpx.logging.Logger();
