/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class HttpRequest
 * extends vpx.core.VpxObject
 *
 * A request that is sent to the server in the form of an HTTP GET or POST and
 * which expects a response back from the server.  Upon receiving a response,
 * this will serialize the response into a <code>HttpResponse</code> object
 * and notify any interested listeners that the response has been received.
 *
 * @version 1.0 (Dec 7, 2005)
 */

/**
 * Constructs a new, uninitialized <code>HttpRequest</code> object.
 *
 * @param reuseTransports boolean
 *    true to reuse transport objects; false to create new transport objects
 *    every time a request is reset
 */
vpx.net.HttpRequest = function HttpRequest(reuseTransports) {
   this.reuseTransports = reuseTransports;
   this.listenerList = new vpx.core.event.EventListenerList();
   this.transport = this._createTransport();
   this.boundFunctions = {
      processStateChange : this._processStateChange.bind(this)
   };
   this._ensureTransportHandlerCreated();
};

// HttpRequest extends vpx.core.VpxObject
vpx.net.HttpRequest.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.net.HttpRequest.prototype.constructor = vpx.net.HttpRequest;

// Shorthand for brevity's sake
var _c = vpx.net.HttpRequest;         // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Static class variables

/**
 * Valid states that an <code>HttpRequest</code> can be in.
 */
_c.UNINITIALIZED = 0;
_c.LOADING       = 1;
_c.LOADED        = 2;
_c.INTERACTIVE   = 3;
_c.COMPLETE      = 4;
_c.ABORTED       = 5;

/**
 * Valid methods that an <code>HttpRequest</code> can use.
 */
_c.GET           = "get";
_c.POST          = "post";

// Instance variables
_i.reuseTransports = true;    // protected boolean
_i.url = null;                // protected String
_i.async = true;              // protected boolean
_i.method = _c.GET;           // protected String
_i.state = _c.UNINITIALIZED;  // protected int
_i.transport = null;          // protected XMLHttpRequest
_i.listenerList  = null;      // protected vpx.core.event.EventListenerList
_i.changeEvent   = null;      // protected transient vpx.core.event.ChangeEvent
_i.response      = null;      // protected
_i.poolSlot = null;           // private int

/**
 * Tells whether or not this requst is set to asynchronous mode.
 *
 * @return boolean
 *    true if this request is set to asynchronous; false otherwise
 */
_i.isAsynchronous = function() {
   return this.async;
};

/**
 * Sets the asynchronous flag for this request.  When asynchronous, requests
 * will continue JavaScript processing immediately after sending the request.
 * When asynchronous is set to false, the JavaScript interpreter will halt
 * processing until the request has received a response from the server.
 *
 * @param b boolean
 *    true to make this request asynchronous; false otherwise.
 */
_i.setAsynchronous = function(b) {
   this.async = b;
};

/**
 * Gets the URL for this request's connection.
 *
 * @return String
 *    This request's URL
 */
_i.getUrl = function() {
   return this.url;
};

/**
 * Sets the value of this request's URL.
 *
 * @param str String
 *    The new value of this request's URL
 */
_i.setUrl = function(str) {
   this.url = str;
};

/**
 * Gets the HTTP method that this request is set to use.
 *
 * @return String
 *    The HTTP method currently set for this request
 */
_i.getMethod = function() {
   return this.method;
};

/**
 * Sets the HTTP method you intend for the request.  The only two accepted
 * values are <code>vpx.net.HttpRequest.GET</code> and
 * <code>vpx.net.HttpRequest.POST</code>.
 *
 * @param str String
 *    A valid HTTP method
 */
_i.setMethod = function(str) {
   if (str == this.method) {
      return;
   }
   var c = vpx.net.HttpRequest;
   if (str != c.GET && str != c.POST) {
      throw new Error("HttpRequest#setMethod(): Invalid method: " + str);
   }
   this.method = str;
};

/**
 * Gets the state of this request. (e.g. vpx.net.HttpRequest.COMPLETE). To
 * listen for changes in a request's state, register for notification on events
 * of that type by using <code>addChangeListener()</code>.
 *
 * @return int
 *    The current state of this request
 */
_i.getState = function() {
   return this.state;
};

/**
 * Gets the HttpResponse Object. Makes only sense when the Request has been sent in
 * synchronous mode
 * @return vpx.net.HttpResponse
 */
_i.getResponse = function() {
   if (this.async) {
      throw new Error("HttpRequest#getResponse(): Invalid method when in asynchronous mode." );
   }
   return this.response;
}

/**
 * Opens a connection to the server, and transmits the request to the server.
 */
_i.send = function() {
   if (this.dying) {
      return;
   }
   this._ensureTransportHandlerCreated();
   var reqUrl = this.url;
   if (!isNull(reqUrl) && reqUrl.indexOf("//") == -1) {
      // We need to make this a fully qualified url
      var tle = vpx.getTle();
      if (reqUrl.charAt(0) == "/") {
         // reqUrl was already absolute; we need only to add fqdn
         reqUrl = tle.urlUtil.getFqdn() + this.url;
      } else {
         // We need to add fqdn *and* the context path
         reqUrl = tle.urlUtil.getFqdn() + tle.urlUtil.getPath() + "/" + this.url;
      }
   }
   this.transport.open(this.method, reqUrl, this.async);
   this.transport.send(null);

   if (!this.async) {
      this._setState(vpx.net.HttpRequest.COMPLETE);
   }
};

/**
 * Aborts an open connection to the server, halting the transmission or
 * receiving of any partial data that has crossed the wire.
 */
_i.abort = function() {
   if (this.dying) {
      return;
   }
   var state = this.state;
   this._setState(vpx.net.HttpRequest.ABORTED);

   try {
      this.transport.abort();
   } catch (e) {
      // Ignore
   }
};

/**
 * Resets a request to its uninitialized state, aborting it if the request is
 * currently communicating with the server.  This also removes any listeners
 * that have been registered on this request.
 */
_i.reset = function() {
   if (this.dying) {
      return;
   }
   this._removeAllListeners();
   this.setAsynchronous(true);
   this.setUrl(null);
   this.setMethod(vpx.net.HttpRequest.GET);
   this._setState(vpx.net.HttpRequest.UNINITIALIZED);
   if (!this.reuseTransports) {
      delete this.transport;
      this.transport = this._createTransport();
   }
};

/**
 * Prepares this object for garbage collection by aborting any current request
 * and destroying all external references that this object maintains.
 */
_i.destroy = function() {
   // super.destroy()
   vpx.core.VpxObject.prototype.destroy.call(this);

   this._removeAllListeners();

   try {
      this.transport.onreadystatechange = function() {};
      this.transport.abort();
   } catch (ex) {
      // Ignore
   }

   delete this.url;
   delete this.method;
   delete this.state;
   delete this.transport;
   delete this.listenerList;
   delete this.changeEvent;
   delete this.boundFunctions;
};

/**
 * Adds a ResponseListener to the request.
 *
 * @param l vpx.net.event.ResponseListener
 *    The listener to add
 */
_i.addResponseListener = function(l) {
   if (this.dying) {
      return;
   }
   this._ensureTransportHandlerCreated();
   this.listenerList.add("ResponseListener", l);
};

/**
 * Removes a ResponseListener from the request.
 *
 * @param l vpx.net.event.ResponseListener
 *    The listener to remove
 */
_i.removeResponseListener = function(l) {
   this.listenerList.remove("ResponseListener", l);
};

/**
 * Returns an array of all the response listeners registered on this
 * <code>HttpRequest</code>.
 *
 * @return vpx.net.event.ResponseListener[]
 *    All of this request's <code>ResponseListener</code>s, or an empty array
 *    if no response listeners are currently registered
 */
_i.getResponseListeners = function() {
   return this.listenerList.getListeners("ResponseListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The <code>ResponseEvent</code> to deliver to listeners
 */
_i._fireResponseReceived = function(e) {
   if (this.dying) {
      return;
   }
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ResponseListener") {
         // listeners[i + 1] is of type vpx.net.event.ResponseListener
         try {
            listeners[i + 1].responseReceived(e);
         } catch (ex) {
            /**
             * We catch errors here to prevent application users from
             * introducing bugs that can interfere with VPX-JS-LIB core
             * processing.  Log and continue.
             */
            vpx.log.error(ex.message);

            if (ex.number == -2146823277 ||
                ex.message == "Can't execute code from a freed script") {
               // Temp fix for bug #88581; still don't know why 'ex' is getting thrown
               vpx.log.warn("HttpRequest#_fireResponseReceived(): " +
                            "listener's context has been freed");
               this.removeResponseListener(listeners[i + 1]);
            }
         }
      }
   }
};

/**
 * Adds a ChangeListener to the request.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to add
 */
_i.addChangeListener = function(l) {
   if (this.dying) {
      return;
   }
   this._ensureTransportHandlerCreated();
   this.listenerList.add("ChangeListener", l);
};

/**
 * Removes a ChangeListener from the request.
 *
 * @param l vpx.core.event.ChangeListener
 *    The listener to remove
 */
_i.removeChangeListener = function(l) {
   this.listenerList.remove("ChangeListener", l);
};

/**
 * Returns an array of all the change listeners registered on this
 * <code>HttpRequest</code>.
 *
 * @return vpx.core.event.ChangeListener[]
 *    All of this request's <code>ChangeListener</code>s, or an empty array if
 *    no change listeners are currently registered
 */
_i.getChangeListeners = function() {
   return this.listenerList.getListeners("ChangeListener");
};

/**
 * Notifies all listeners that have registered interest for notification on
 * this event type. The event instance is created lazily.
 */
_i._fireStateChanged = function() {
   if (this.dying) {
      return;
   }
   // Guaranteed to return a non-null array
   var listeners = this.listenerList.getListenerList();

   // Process listeners last->first, notifying those who're interested in event
   for (var i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == "ChangeListener") {
         // Lazily create the event:
         if (this.changeEvent == null) {
            this.changeEvent = new vpx.core.event.ChangeEvent(this);
         }

         // listeners[i + 1] is of type vpx.core.event.ChangeListener
         listeners[i + 1].stateChanged(this.changeEvent);
      }
   }
};

/**
 * (non-Javadoc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.net.HttpRequest]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/*
 * Sets a request's "pool slot".  See the description in
 * <code>_getPoolSlot</code>.
 *
 * @param i int
 *    The request's pool slot
 */
_i._setPoolSlot = function(i) {
   this.poolSlot = i;
};

/*
 * Gets a request's "pool slot", which is the reserved slot in a request pool's
 * array of requests.  The pool will ask a request for this value to determine
 * where in the pool's array the request should reside.  This notion exists to
 * dampen multithreading issues that exist with requests being checked in and
 * out of a request pool.
 *
 * @return int
 *    This request's pool slot
 */
_i._getPoolSlot = function() {
   return this.poolSlot;
};

/*
 * Sets the state of this request to the given state.
 *
 * @param state int
 *    The new state of this request
 */
_i._setState = function(state) {
   if (this.state == state) {
      return;
   }
   this.state = state;

   if (state == vpx.net.HttpRequest.COMPLETE) {
      // Fire a ResponseEvent
      var t = this.transport;
      try {
         var contentType = t.getResponseHeader("Content-Type").split(";")[0];
      } catch (ex) {
         // An error retrieving headers signals request did not go through
         this.abort();
         return;
      }
      var resp = new vpx.net.HttpResponse(contentType,
                                          t.responseText,
                                          t.responseXML,
                                          t.status,
                                          t.statusText);
      this.response = resp;
      var event = new vpx.net.event.ResponseEvent(resp, this);
      this._fireResponseReceived(event);
   }

   this._fireStateChanged();
};

/*
 * Creates a new native transport object that performs the actual work of this
 * request.  This transport object is platform-dependant.
 *
 * @return XMLHttpRequest
 *    A native xml http request object
 */
_i._createTransport = function() {
   return vpx.xua.cull(
      function() {return new ActiveXObject('Msxml2.XMLHTTP');},
      function() {return new ActiveXObject('Microsoft.XMLHTTP');},
      function() {return new XMLHttpRequest();}
   );
};

/*
 * Makes sure that the <code>onreadystatechange</code> handler of the native
 * transport object is set.  Transports can sometimes unset this value
 * automatically in certain conditions, so it's up to us to make sure it's set
 * when it needs to be.
 */
_i._ensureTransportHandlerCreated = function() {
   try {
      this.transport.onreadystatechange = this.boundFunctions.processStateChange;
   } catch (ex) {
      // Mozilla occasionally throws exception trying to read this property
      return;
   }
};

/*
 * Processes a state change in the transport object of this request.  In
 * general, this delegates its work to <code>_setState(int)</code>.  However,
 * native transport objects will change their state to "Complete" when they
 * are aborted, which is deceiving, so this will catch such an occurance and
 * handle it appropriately.
 */
_i._processStateChange = function() {
   if (this.dying) {
      return;
   }

   var newState = this.transport.readyState;
   var oldState = this.state;

   var c = vpx.net.HttpRequest;
   if (oldState == c.ABORTED && newState == c.COMPLETE) {
      // State is not really "Complete"; ignore
      return;
   }
   this._setState(newState);
};

/**
 *
 */
_i._removeAllListeners = function() {
   var lstnrs, i;

   lstnrs = this.getResponseListeners();
   for (i = 0; i < lstnrs.length; i++) {
      this.removeResponseListener(lstnrs[i]);
   }
   lstnrs = this.getChangeListeners();
   for (i = 0; i < lstnrs.length; i++) {
      this.removeChangeListener(lstnrs[i]);
   }
};
