/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class HttpRequestPool
 * extends vpx.core.VpxObject
 *
 * TODO
 *
 * @version 1.0 (Dec 8, 2005)
 * @see vpx.net.HttpRequest
 */

/**
 * Constructs a new <code>HttpRequestPool</code> with the given number of
 * requests.
 *
 * @param minRequests int
 *    The minimum number of requests that the pool should maintain
 */
vpx.net.HttpRequestPool = function HttpRequestPool(minRequests) {
   this.minRequests = minRequests;
   this.avail = [];
   this.reserved = new Array(minRequests);
   this.busy = new Array(minRequests);
   this.all = new Array(minRequests);

   // Instantiate the minimum number of requests
   for (var i = 0; i < minRequests; i++) {
      var req = new vpx.net.HttpRequest(vpx.net.HttpRequestPool.DEFAULT_REUSE_TRANSPORT);
      req._setPoolSlot(i);
      this.avail.push(req);
      this.reserved[i] = null;
      this.busy[i] = null;
      this.all[i] = req;
      delete req;
   }
};

// HttpRequestPool extends vpx.core.VpxObject
vpx.net.HttpRequestPool.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.net.HttpRequestPool.prototype.constructor = vpx.net.HttpRequestPool;

// Shorthand for brevity's sake
var _c = vpx.net.HttpRequestPool;  // Class
var _i = _c.prototype;             // Instance
_i._concrete = true;               // vpx system flag for concrete classes

// Static class variables
_c.instances = {};                 // private static Map<String,HttpRequestPool>

// Constants
_c.DEFAULT_POOL_KEY = "shared";    // public static final String
_c.DEFAULT_MIN_REQUESTS = 10;      // public static final int
_c.DEFAULT_REUSE_TRANSPORT = true; // public static final boolean

// Instance variables
_i.minRequests = null;             // protected int;
_i.handler = null;                 // protected vpx.net.HttpRequestPool.Handler

/**
 * protected List<vpx.net.HttpRequest>
 *
 * The list of available requests.
 */
_i.avail = null;

/**
 * protected List<vpx.net.HttpRequest>
 *
 * The list of requests that are currently reserved by external callers.
 */
_i.reserved = null;

/**
 * protected List<vpx.net.HttpRequest>
 *
 * The list of requests that have been released back into the pool but are
 * still processing.  They will become "available" once they complete the
 * request cycle.
 */
_i.busy = null;

/**
 * protected List<vpx.net.HttpRequest>
 *
 * The list of all requests in the pool, regardless of state. As requests are
 * moved from one list to another, this master list provides a way to iterate
 * through all the requests in the pool.  Another way would be to provide
 * synchronization constructs around the various lists, but since JS has no
 * such consructs built-in, this method suffices as well.
 */
_i.all = null;

/**
 * Checks out an available request from the pool.  It is the caller's
 * responsibility to set up the request using the request's public API.  This
 * includes setting the request's url, its method, registering any
 * listeners, and sending the request.
 * <p/>
 * NOTE: You <b>must</b> release this request after you are done with it.
 * Otherwise, requests will be leaked, and the pool will run out of requests.
 */
_i.getRequest = function() {
   if (this.avail.length == 0) {
      return null;
   }
   var req = this.avail.pop();
   this.reserved[req._getPoolSlot()] = req;
   return req;
};

/**
 * Releases a request back into the pool.  This MUST be called by the same
 * caller who aquired the request from the pool.  Otherwise, requests will be
 * leaked, and the pool will run out of requests.
 *
 * @param req vpx.net.HttpRequest
 *    The request that had previously been aquired from the pool
 */
_i.releaseRequest = function(req) {
   var i = req._getPoolSlot();
   if (i == null || i == -1 || this.reserved[i] == null) {
      throw new Error("HttpRequestPool#releaseRequest(): request not reserved");
   }
   this.reserved[i] = null;

   // Concurrency issue here, but JS provides no exclusivity constructs, so...
   if (req.getState() == vpx.net.HttpRequest.COMPLETE) {
      // Resetting will remove all change listeners, including our handler
      req.reset();
      this.avail.push(req);
   } else {
      req.addChangeListener(this._getHandler());
      this.busy[i] = req;
   }
};

/**
 * Destroys this pool and all requests that this pool manages.
 */
_i.destroy = function() {
   // super.destroy()
   vpx.core.VpxObject.prototype.destroy.call(this);

   var req = null;
   while (req = this.all.pop()) {
      if (req != null) {
         req.destroy();
      }
   }
   delete req;
};

/**
 * Gets the global shared request pool.
 *
 * @return vpx.net.HttpRequestPool
 *    The shared request pool
 */

/**
 * Gets a shared request pool that has been indexed by the given key.  If no
 * pool has been stored under the given lookup, then a new one will be created
 * and stored for future use and will be configured to have the default number
 * of minimum requests.
 *
 * @param key String
 *    The key with which to look up the request pool
 * @return vpx.net.HttpRequestPool
 *    The static request pool found under the given lookup
 */

/**
 * Gets a shared request pool that has been indexed by the given key.  If no
 * pool has been stored under the given lookup, then a new one will be created
 * and stored for future use and will be configured to have the given number
 * of minimum requests.
 *
 * @param key String
 *    The key with which to look up the request pool
 * @param minRequests int
 *    The minimum number of requests that the pool should maintain, if this is
 *    the first time the given pool is being retrieved
 * @return vpx.net.HttpRequestPool
 *    The static request pool found under the given lookup
 */
_c.getInstance = function(key, minRequests) {
   // Shard instance always lives in the top level window
   var c = vpx.getTle().vpx.net.HttpRequestPool;

   if (arguments.length <= 1) {
      // Mimic 1-arg getInstance(String)
      minRequests = c.DEFAULT_MIN_REQUESTS;

      if (arguments.length == 0) {
         // Mimic no-arg getInstance()
         key = c.DEFAULT_POOL_KEY;
      }
   }

   if (!c.instances[key]) {
      c.instances[key] = new c(minRequests);
   }

   return c.instances[key];
};

/**
 * Gets this request pool's event handler.
 *
 * @return vpx.net.HttpRequestPool.Handler
 *    The handler for this request pool
 */
_i._getHandler = function() {
   if (this.handler == null) {
      this.handler = new vpx.net.HttpRequestPool.Handler(this);
   }
   return this.handler;
};

/**
 * class Handler
 * extends Object
 * implements vpx.core.event.ChangeListener
 *
 * Listener that automatically gets registered with this request pool's
 * requests as they get released back into the pool.  It listens for state
 * changes in the requests and moves the requests from the "busy" pool
 * to the "avail" pool as the requests complete their processing.
 *
 * @version 1.0 (Dec 9, 2005)
 */

/**
 * Constructs a new Handler.
 *
 * @param pool vpx.net.HttpRequestPool
 *    The pool for which this handler works
 */
_c.Handler = function HttpRequestPoolHandler(pool) {
   this.pool = pool;
};

/**
 * (non-Javadoc)
 *
 * @see vpx.core.event.ChangeListener#stateChanged(vpx.core.event.ChangeEvent)
 */
_c.Handler.prototype.stateChanged = function(e) {
   var req = e.getSource();
   var state = req.getState();
   if (state == vpx.net.HttpRequest.COMPLETE || state == vpx.net.HttpRequest.ABORTED) {
      req.reset();
      this.pool.busy[req._getPoolSlot()] = null;
      this.pool.avail.push(req);
   }
};
