/* Copyright 2005 VMware, Inc.	All rights reserved. -- VMware Confidential */

/**
 * public class Query
 * extends Object
 *
 * TODO
 *
 * @version 1.0 (Aug 5, 2005)
 */

/**
 * Constructs a new Query object.
 *
 * @param str String
 *    A query string (e.g. 'param1=one&param2=two')
 */
vpx.net.Query = function(str) {
	this.params = {};
	this.keys = [];

	var pairs = str.split("&");
	for (var i = 0; i < pairs.length; i++) {
		var index = pairs[i].indexOf("=");
		if (index <= 0) {
			continue;
		}
		var key = pairs[i].substring(0, index);
		var value = pairs[i].substring(index + 1);
		if (value != "") {
			value = value.replace(/\+/g, " ");
			this.params[key] = vpx.xua.urlDecode(value);
			this.keys.push(key);
		}
	}
};

// Shorthand for brevity's sake
var _c = vpx.net.Query;               // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.keys = null;                       // protected String[]
_i.params = null;                     // protected Map<String,String>

/**
 * Gets the list of parameter keys contained in this query.
 *
 * @return String[]
 *    List of parameter keys, or empty array is no parameters exist
 */
_i.getParams = function() {
	return this.keys;
};

/**
 * Tells whether or not this query contains the given parameter.
 *
 * @param key String
 *		The parameter key
 * @return boolean
 *		true if the parameter exists; false otherwise
 */
_i.hasParam = function(key) {
	return isDefined(this.params[key]);
};

/**
 * Gets a parameter value from the query.
 *
 * @param key String
 *		The parameter key
 * @return String
 *		The parameter value, or null if no such parameter exists
 */
_i.getParam = function(key) {
	if (isNull(this.params[key])) {
		return null;
	}
	return this.params[key];
};

/**
 * Sets or overrides a parameter value in the query.
 *
 * @param key String
 *		The parameter key
 * @param value String
 *		The parameter value
 */
_i.setParam = function(key, value) {
	if (this.keys.indexOf(key) == -1) {
		this.keys.push(key);
	}
	this.params[key] = value;
};

/**
 * Removes a parameter from the query.  If the parameter did not
 * exist to begin with, nothing happens.
 *
 * @param key String
 *		The parameter key
 */
_i.removeParam = function(key) {
	var index = this.keys.indexOf(key);
	if (index == -1) {
		return;
	}
	this.keys.splice(index, 1);
	delete this.params[key];
};

/**
 * Performs a union on this query and the given query, meaning that all
 * parameters contained in the given query will be merged into this
 * query. When parameters are found to exist in both queries, the existing
 * query will keep its parameter value (the given query will NOT override
 * existing parameter values).
 *
 * @param query vpx.net.Query
 *    The query with which to perform the union
 */
_i.union = function(query) {
	var keys = query.getParams();
	for (var i = 0; i < keys.length; i++) {
		var key = keys[i];
		if (!this.hasParam(key)) {
			this.setParam(key, query.getParam(key));
		}
	}
};

/**
 * Converts this query to a string, with each parameter value being encoded
 * for insertion into a parent query parameter.
 *
 * @return String
 *    This query, converted to string with values encoded
 */
_i.toEncodedString = function() {
	var pairs = [];

	for (var i = 0; i < this.keys.length; i++) {
		var key = this.keys[i];
		if (isDefined(this.params[key])) {
			pairs.push(key + "=" + vpx.xua.urlEncode(this.params[key]));
		}
	}

	return pairs.join("&");
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
	var pairs = [];

	for (var i = 0; i < this.keys.length; i++) {
		var key = this.keys[i];
		if (isDefined(this.params[key])) {
			pairs.push(key + "=" + this.params[key]);
		}
	}

	return pairs.join("&");
};

/**
 * Creates a new Query object that represents the current conn query.
 *
 * @return vpx.net.Query
 *    A new Query object
 */
_c.fromConn = function() {
	var str = location.search.substring(1);
	return new vpx.net.Query(str);
};
