/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class XmlSpec
 * extends vpx.core.VpxObject
 *
 * TODO
 *
 * @version 1.0 (Jul 21, 2005)
 */

/**
 * Constructs a new XmlSpec with no initial attributes.
 *
 * @param path String
 *    The path part of the url (before the "?")
 */
vpx.net.XmlSpec = function(path) {
   this.path = path;
   this.attrKeys = [];
   this.attributes = {};
};

// XmlSpec extends vpx.core.VpxObject
vpx.net.XmlSpec.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.net.XmlSpec.prototype.constructor = vpx.net.XmlSpec;

// Shorthand for brevity's sake
var _c = vpx.net.XmlSpec;             // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.path = null;                       // protected String
_i.attrKeys = null;                   // protected String[]
_i.attributes = null;                 // protected Map<String,String>

/**
 * Gets this xml spec's path - the url from which to fetch server data.
 *
 * @return String
 *    The path part of the url (before the "?")
 */
_i.getPath = function() {
   return this.path;
};

/**
 * Sets the value of this xml spec's path - the url from which to fetch server
 * data.
 *
 * @param path String
 *    The path part of the url (before the "?")
 */
_i.setPath = function(path) {
   this.path = path;
};

/**
 * Tells whether or not this xml spec has the given attribute set in its list
 * of attributes.
 *
 * @param key String
 *    The attribute key
 * @return boolean
 *    true if the attribute exists; false otherwise
 */
_i.hasAttribute = function(key) {
   return isDefined(this.attributes[key]);
};

/**
 * Gets an attribute of the xml spec. An attribute is a key/value pair that
 * will be passed on to the server in the form of a query parameter.
 *
 * @param key String
 *    The attribute key
 * @return String
 *    The attribute value, or null if no such attribute exists
 */
_i.getAttribute = function(key) {
   if (isNull(this.attributes[key])) {
      return null;
   }
   return this.attributes[key];
};

/**
 * Sets or overrides an attribute of the xml spec.  This attribute will be
 * passed on to the server in the form of query parameters.
 *
 * @param key String
 *    The attribute key
 * @param value String
 *    The attribute value
 */
_i.setAttribute = function(key, value) {
   if (this.attrKeys.indexOf(key) == -1) {
      this.attrKeys.push(key);
   }
   this.attributes[key] = value;
};

/**
 * Removes an attribute from the xml spec.  This attribute will no longer
 * be passed to the server as a query parameter.  If the attribute did not
 * exist to begin with, nothing happens.
 *
 * @param key String
 *    The attribute key
 */
_i.removeAttribute = function(key) {
   var index = this.attrKeys.indexOf(key);
   if (index == -1) {
      return;
   }
   this.attrKeys.splice(index, 1);
   delete this.attributes[key];
};

/**
 * Removes all attributes associated with this xml spec. Then, no attributes
 * will be passed to the server as query parameters.
 */
_i.clearAttributes = function() {
   delete this.attrKeys;
   delete this.attributes;
   this.attrKeys = [];
   this.attributes = {};
};

/**
 * Returns a url representation of this object (with query parameters).
 *
 * @return String
 *    A url suitable for using as an HTTP request
 */
_i.toUrl = function() {
   var url = this.path;

   var query = [];
   for (var i = 0; i < this.attrKeys.length; i++) {
      var attr = this.attrKeys[i];
      if (!isNull(this.attributes[attr])) {
         query.push(attr + "=" + this.attributes[attr]);
      }
   }
   var queryStr = query.join("&");
   if (query.length > 0) {
      url += "?" + queryStr;
   }

   return url;
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.net.XmlSpec (" + this.toUrl() + ")]";
};

/**
 * Constructs an xml spec from a url.
 *
 * @param url String
 *    A full url (e.g. /myAction.do?page=0&count=10)
 * @return vpx.net.XmlSpec
 *    A new xml spec
 */
_c.fromUrl = function(url) {
   var elements = url.split("?");
   var path = elements[0];

   var spec = new vpx.net.XmlSpec(path);

   if (elements.length > 1) {
      var query = elements[1].split("&");
      for (var i = 0; i < query.length; i++) {
         var pair = query[i].split("=");
         spec.setAttribute(pair[0], pair[1]);
      }
   }

   return spec;
};
