/* Copyright 2005 VMware, Inc.   All rights reserved. -- VMware Confidential */

/**
 * public class Agent
 * extends vpx.core.VpxObject
 * implements vpx.net.event.ResponseListener
 *
 * TODO
 *
 * @version 1.0 (Jul 21, 2005)
 */

/**
 * Constructs a new updates Agent. The new agent will look for updates at the
 * default location (class constant).  It will NOT automatically start
 * itself - that is left up to the caller after having registered all relevant
 * listeners.
 *
 * @param interval int
 *    The number of milliseconds to wait in between requests for updates
 */
vpx.updates.Agent = function Agent(desiredInterval) {
   this.setDesiredInterval(desiredInterval);
   this.hash = {};
   this.viewIds = [];
   this.xmlSpec = new vpx.net.XmlSpec(vpx.updates.Agent.UPDATE_URL);
   this.watchDog = new vpx.updates.WatchDog(this);

   this.boundFunctions = {
      poll : vpx.updates.Agent.prototype._doPoll.bind(this),
      destroy : vpx.updates.Agent.prototype.destroy.bind(this)
   };

   vpx.xua.event.listen(window, "unload", this.boundFunctions.destroy);
};

// Agent extends vpx.core.VpxObject
vpx.updates.Agent.prototype = new vpx.core.VpxObject(vpx.ABSTRACT_PASS);
vpx.updates.Agent.prototype.constructor = vpx.updates.Agent;

// Shorthand for brevity's sake
var _c = vpx.updates.Agent;           // Class
var _i = _c.prototype;                // Instance
_i._concrete = true;                  // vpx system flag for concrete classes

// Instance variables
_i.desiredInterval = null;            // private int
_i.interval = null;                   // private transient int
_i.timeoutId = null;                  // private transient int
_i.req = null;                        // private transient vpx.net.HttpRequest
_i.running = false;                   // private transient boolean
_i.hash = null;                       // private Map<String,String>
_i.viewIds = null;                    // private List<String>
_i.xmlSpec = null;                    // private vpx.net.XmlSpec
_i.watchDog = null;                   // private vpx.updates.WatchDog

// Static (class) constants
_c.UPDATE_URL    = "viewCheckForUpdatesXml.do";
_c.PARAM_VIEWIDS = "viewIds";

_c._lstnrId = 0;

/**
 * Sets the value of this agent's desired update interval.  The real interval
 * may be higher if the agent is in a "cooling off" period (a period of higher
 * intervals to allow the server to lower its load).  The real interval may
 * never be lower (more frequent) than the desired interval.
 *
 * @param i int
 *    The new interval, in milliseconds
 */
_i.setDesiredInterval = function(i) {
   if (this.dying) {
      return;
   }
   if (this.interval == null || this.interval < i) {
      this.interval = i;
   }
   this.desiredInterval = i;
};

/**
 * Gets the value of this agent's desired update interval.  The real interval
 * may be higher if the agent is in a "cooling off" period (a period of higher
 * intervals to allow the server to lower its load).  The real interval may
 * never be lower (more frequent) than the desired interval.
 *
 * @return int
 *    The interval, in milliseconds
 */
_i.getDesiredInterval = function() {
   if (this.dying) {
      return null;
   }
   return this.desiredInterval;
};

/**
 * Sets the value of this agent's real interval.  The real interval may never
 * be lower (more frequent) than the agent's desired interval.  An attempt to
 * do so will result in the real interval being set to the desired interval.
 *
 * @param i int
 *    The new interval, in milliseconds
 */
_i.setInterval = function(i) {
   if (this.dying) {
      return;
   }
   if (i == null || i < this.desiredInterval) {
      this.interval = this.desiredInterval;
   } else {
      this.interval = i;
   }
};

/**
 * Gets the value of this agent's real update interval.
 *
 * @return int
 *    The interval, in milliseconds
 */
_i.getInterval = function() {
   if (this.dying) {
      return null;
   }
   return this.interval;
};

/**
 * Causes this agent to begin execution. Once executing, it will request
 * updates from the server at the agent's interval.
 */
_i.start = function() {
   if (this.dying) {
      return;
   }
   if (this.running) {
      throw new Error("Agent#start(): Agent already running");
   }
   this.running = true;
   vpx.log.info("Agent#start(): Starting up...");
   this._queuePoll();
   this.watchDog.recordNoOp();
   this.watchDog.start();
};

/**
 * Forces the agent to stop requesting updates from the server.  Any data that
 * is currently being processed by the agent will finish it processing.
 */
_i.stop = function() {
   if (!this.running) {
      throw new Error("Agent#stop(): Agent not running");
   }
   this.running = false;
   vpx.log.info("Agent#stop(): Stopping...");
   this.watchDog.stop();
   this._cancelPoll();
};

/**
 * Destroys this agent, freeing up any memory and resources that the agent was
 * using.
 */
_i.destroy = function() {
   // super.destroy()
   vpx.core.VpxObject.prototype.destroy.call(this);

   vpx.xua.event.ignore(window, "unload", this.boundFunctions.destroy);

   if (this.running) {
      this.stop();
   }

   delete this.boundFunctions.poll;
   delete this.boundFunctions.destroy;
   delete this.boundFunctions;
   this.watchDog.destroy();
   this.xmlSpec.destroy();
   delete this.watchDog;
   delete this.xmlSpec;
   delete this.hash;
   delete this.viewIds;
   delete this.desiredInterval;
   delete this.interval;
};

/**
 * Tells whether this agent is currently executing, meaning that it is
 * periodically requsting updates from the server.
 *
 * @return boolean
 *    true if this agent is running; false otherwise
 */
_i.isRunning = function() {
   if (this.dying) {
      return false;
   }
   return this.running;
};

/**
 * Registers the given callback to be executed when updates are ready for the
 * given view.  This ensures that the agent will receive server updates
 * pertaining to the given view (by updating the Agent's internal
 * <code>vpx.net.XmlSpec</code> object).
 *
 * @param viewId String
 *    The id of the back-end <code>View</code> whose updates you're listening
 *    for
 * @param name String
 *    A logical name for the view, used for debugging purposes
 * @param callback Function
 *    The function to invoke when an update is ready for the given view
 * @return int
 *    A listenerId that can be used to later remove this listener
 */
_i.registerView = function(viewId, name, callback) {
   if (this.dying) {
      return;
   }
   if (isNull(this.hash[viewId])) {
      // Need to add this viewId to the list of views we're checking
      vpx.log.info("Agent#registerView(): Adding view: " + viewId + " (" + name + ")");
      this.hash[viewId] = {
         name   : name,
         lstnrs : []
      };
      this.viewIds.push(viewId);
   }

   var listener = {
      id       : ++vpx.updates.Agent._lstnrId,
      callback : callback
   };

   this.hash[viewId].lstnrs.push(listener);
   return listener.id;
};

/**
 * Unregisters all callback associated with the given view.  This ensures that
 * the agent will no longer request updates for the given view.  The caller may
 * inadvertently unregister callbacks that were registered by foreign code, so
 * this should only be called when the view in question is known to be no
 * longer valid.
 *
 * @param viewId String
 *    The id of the back-end <code>View</code> whose updates you want to cancel
 */
_i.unregisterView = function(viewId) {
   if (this.dying) {
      return;
   }
   if (isNull(this.hash[viewId])) {
      // Trivial success
      return;
   }

   vpx.log.info("Agent#()unregisterView: Removing view: " + viewId + " (" +
                this.hash[viewId].name + ")");
   delete this.hash[viewId];
   var index = this.viewIds.indexOf(viewId);
   if (index >= 0) {
      // Should always be the case, but multithreading could cause otherwise
      this.viewIds.splice(index, 1);
   }
};

/**
 * Unregisters the given listener. If the listener was the only one associated
 * with its view, then this also ensures that the agent will no longer request
 * updates for that view.  If the listener is not found in the agent's list of
 * listeners, no action is taken.
 *
 * @param listenerId int
 *    The id returned upon adding the listener
 */
_i.unregisterListener = function(listenerId) {
   if (this.dying) {
      return;
   }
   var array;

   for (var i = 0; i < this.viewIds.length; i++) {
      var viewId = this.viewIds[i];
      var viewInfo = this.hash[viewId];

      for (var j = 0; j < viewInfo.lstnrs.length; j++) {
         var listener = viewInfo.lstnrs[j];

         if (listener.id == listenerId) {
            // We've found the listener to remove
            if (viewInfo.lstnrs.length == 1) {
               // It's the only listener for that view
               this.unregisterView(viewId);
            } else {
               // We still listen on the viewId; just remove this listener
               if (vpx.log.isInfoEnabled()) {
                  vpx.log.info("Agent#()unregisterListener: Removing listener: " +
                               viewId + " (" + viewInfo.name + "); " +
                               (viewInfo.lstnrs.length - 1) + " listeners remain");
               }
               viewInfo.lstnrs.splice(j, 1);
            }

            // Listener ids are unique; no need to keep looking
            return;
         }
      }
   }
};

/*
 * (non-doc)
 *
 * @see Object#toString()
 */
_i.toString = function() {
   return "[Object vpx.updates.Agent]";
};


/*************************************************************************
 * All data and procedures below this point are part of the internal     *
 * implementation, should not be accessed outside of this module, and    *
 * are subject to change.                                                *
 *************************************************************************/


/**
 * Queues a poll cycle to run in the future.  The delay is determined by the
 * agent's update interval.
 */
_i._queuePoll = function() {
   if (this.dying) {
      return;
   }
   if (this._isPollQueued()) {
      throw new Error("Agent#_queuePoll(): Poll already queued");
   }
   this.timeoutId = window.setTimeout(this.boundFunctions.poll, this.interval);
};

/**
 * Cancels a pending poll cycle.  If a poll is currently active, this will also
 * abort the request.  If there is no active poll and no poll cycle queued,
 * nothing happens.
 */
_i._cancelPoll = function() {
   if (this._isPollActive()) {
      try {
         this.req.abort();
      } catch (e) {
         ; // Ignore
      }
      this.req = null;
   }
   if (this._isPollQueued()) {
      try {
         window.clearTimeout(this.timeoutId);
      } catch (e) {
         ; // Ignore
      }
      this.timeoutId = null;
   }
};

/**
 * Tells whether or not a poll cycle has been queued.
 *
 * @return boolean
 *    true if a poll cycle is queued; false otherwise
 */
_i._isPollQueued = function() {
   if (this.timeoutId != null) {
      return true;
   }
   return false;
};

/**
 * Tells whether or not a poll cycle is active, meaning that the agent has
 * requested data from the server but not yet processed a result.
 *
 * @return boolean
 *    true if the agent has made a request and not yet processed the response;
 *    false otherwise
 */
_i._isPollActive = function() {
   if (this.req != null) {
      return true;
   }
   return false;
};

/*
 * Requests an update from the server via an XML HTTP request per this agent's
 * xml spec.  It listens for a response from the server and registers a
 * callback to automatically process the response.
 *
 * @see vpx.updates.Agent#responseReceived(vpx.net.event.ResponseEvent)
 */
_i._doPoll = function() {
   if (this.dying) {
      return;
   }
   this.timeoutId = null;

   if (!this.running) {
      return;
   }

   if (this.viewIds.length > 0) {
      // We only request if we're listening for at least 1 update
      var viewIdsStr = this.viewIds.join(",");
      this.xmlSpec.setAttribute(vpx.updates.Agent.PARAM_VIEWIDS, viewIdsStr);
      vpx.log.debug("Agent#_doPoll(): Requesting update: " + viewIdsStr);

      var pool = vpx.net.HttpRequestPool.getInstance();
      var req = pool.getRequest();
      if (req == null) {
         // Assume another one will be available next time around
         this.watchDog.recordNoOp();
         this._queuePoll();
         return;
      }
      try {
         this.req = req;
         req.setUrl(this.xmlSpec.toUrl());
         req.addResponseListener(this);
         this.watchDog.recordRequest();
         req.send();
      } finally {
         pool.releaseRequest(req);
      }
   } else {
      // We didn't need to send a request; just re-schedule
      this.watchDog.recordNoOp();
      this._queuePoll();
   }
};

/*
 * Processes an update response from the server.
 *
 * @param e vpx.net.event.ResponseEvent
 *    The response event generated from the server response
 * @see vpx.net.event.ResponseListener#responseReceived(vpx.net.event.ResponseEvent)
 */
_i.responseReceived = function(e) {
   if (this.dying) {
      return;
   }
   var viewId, i;
   var resp = e.getSource();

   this.watchDog.recordResponse();
   this.req = null;

   if (!this.running) {
      return;
   }

   if (!this._isPollQueued()) {
      this._queuePoll();
   }

   var status = resp.getStatus();
   if (status != 200) {
      // Only proceed if HTTP status is "200 OK" (otherwise throw error or return)
      var url = this.xmlSpec.toUrl();
      var msg = resp.getStatusMsg();
      switch (status) {
      case 404:
         throw new Error("Agent#responseReceived(): 404 Not Found: " + url);
      case 500:
         throw new Error("Agent#responseReceived(): 500 Application Error: " + url);
      default:
         /**
          * All other non-200 status codes represent some (esoteric) failure of the
          * request/response transaction.  In these cases, we know the server didn't
          * successfully process the request in the manner we intended.  If the server
          * didn't process the request, it's harmless to return here without action,
          * as the Agent will re-request on the next poll cycle.
          */
         vpx.log.warn("Agent#responseReceived(): " + status + " " + msg);
         return;
      }
   }

   var contentType = resp.getContentType();
   if (contentType != "text/xml") {
      throw new Error("Agent#responseReceived(): invalid content-type: " +
                      contentType + " (status " + status + ")");
   }

   vpx.log.trace("Agent#responseReceived(): Processing data from server");
   var xml = resp.getXml();
   var root = xml.documentElement;
   if (root == null) {
      // XML was not well-formed
      throw new Error("Agent#responseReceived(): xml was not well-formed " +
                      "while processing " + req.getUrl());
   }
   root.normalize();

   if (root.nodeName == "errors") {
      tle.processErrors(root, false);
      return;
   }

   var updateNodes = root.getElementsByTagName("update");
   for (i = 0; i < updateNodes.length; i++) {
      // Each <update> node pertains to a particular view
      var update = updateNodes[i];
      viewId = update.getAttribute("viewId");

      // "refresh" is at the <update> level
      var refresh = false;
      var refreshStr = update.getAttribute("refresh");
      if (isDefined(refreshStr)) {
         refresh = refreshStr.toBool();
      }

      if (refresh) {
         // No need to process individual <changeSet> nodes
         vpx.log.debug("Agent#responseReceived(): Refresh update for view: " + viewId);
         this._fireEvent(viewId, refresh);
         continue;
      }

      var changeSetNodes = update.getElementsByTagName("changeSet");
      if (changeSetNodes.length == 0) {
         // No updates
         continue;
      }

      vpx.log.debug("Agent#responseReceived(): Line item update for view: " + viewId);
      var changeSets = [];
      for (var j = 0; j < changeSetNodes.length; j++) {
         changeSets.push(this._processChangeSet(changeSetNodes[j]));
      }

      this._fireEvent(viewId, refresh, changeSets);
   }

   var expiredNodes = root.getElementsByTagName("expired");
   for (i = 0; i < expiredNodes.length; i++) {
      viewId = expiredNodes[i].getAttribute("viewId");
      vpx.log.info("Agent#responseReceived(): Unregistering expired view: " + viewId);
      this.unregisterView(viewId);
   }
};

/*
 * Processes a single xml <update> node and creates a JS object representing
 * the data in that node.
 *
 * @param node Element
 *    The <changeSet> xml node
 * @return Object
 *    A representation of the node having the following properties:
 *       id      : String
 *       type    : String
 *       changes : Object[] {property:String, value:String}
 */
_i._processChangeSet = function(node) {
   var changeSet = {
      id      : node.getAttribute("id"),
      type    : node.getAttribute("type"),
      changes : []
   };

   var changeNodes = node.getElementsByTagName("change");
   for (var i = 0; i < changeNodes.length; i++) {
      var change = {
         property : changeNodes[i].getAttribute("property"),
         value    : vpx.xua.getInnerContent(changeNodes[i])
      };
      changeSet.changes.push(change);
   }

   return changeSet;
};

/*
 * Fires an update event for the given view, causing all registered listeners
 * to get notified (i.e. their callback routines to get executed).
 *
 * All listeners will be invoked with the first argument being the agent
 * that triggered the event.
 *
 * @param viewId String
 *    The id of the back-end <code>View</code> whose updates you want to cancel
 * @param arguments [variable]
 *    Any other arguments will be passed to the listeners after the first
 *    argument (this object)
 */
_i._fireEvent = function(viewId) {
   if (this.dying) {
      return;
   }
   var i;
   vpx.log.trace("Agent#_fireEvent(): Firing event for view " + viewId);

   var viewInfo = this.hash[viewId];
   if (isNull(viewInfo)) {
      // No listeners
      return;
   }

   var args = [this];
   for (i = 1; i < arguments.length; i++) {
      args.push(arguments[i]);
   }

   var errors = null;

   /**
    * Copy listener array to avoid concurrent modification issues (listenerA
    * invokes callback1, which registers listenerB with callback1: infinite
    * loop).
    */
   var lstnrs = [];
   vpx.core.System.arraycopy(viewInfo.lstnrs, 0, lstnrs, 0, viewInfo.lstnrs.length);

   for (i = 0; i < lstnrs.length; i++) {
      var l = lstnrs[i];
      if (!isNull(l.callback)) {
         try {
            l.callback.apply(self, args);
         } catch (ex) {
            if (ex.number == -2146823277 ||
                ex.message == "Can't execute code from a freed script") {
               // Temp fix for bug #88581; still don't know why 'ex' is getting thrown
               vpx.log.error("Agent#_fireEvent(): callback has been freed");
               this.unregisterListener(l.id);
            } else {
               /**
                * We intentionally delay throwing any errors so that one
                * listener's error doesn't prevent other listeners from getting
                * notified of updates
                */
               if (errors == null) {
                  // Delay object instantiation
                  errors = [];
               }
               errors.push(ex.message);
            }
         }
      } else {
         // The callback has disappeared
         vpx.log.error("Agent#_fireEvent(): callback has disappeared");
         this.unregisterListener(l.id);
      }
   }

   if (errors != null) {
      throw new Error(errors.join("... AND "));
   }
};
